# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys
import Tac
import subprocess
import LazyMount
import BasicCli
import CliCommand
import CliMatcher
from ContainerMgrShowCli import isContainerMgrRunning
from ContainerMgrCliLib import isEnoughSpacePresent, imageSize
from ContainerMgrCliLib import imagePresent
from ContainerMgrCliLib import containerNameRe, imageNameRe, pathRe

containerConfig = None
containerMgrConfig = None

matcherBackup = CliMatcher.KeywordMatcher( 'backup',
      helpdesc='Backup' )
matcherContainerManager = CliMatcher.KeywordMatcher( 'container-manager',
      helpdesc='Container Manager' )
matcherForce = CliMatcher.KeywordMatcher( 'force',
      helpdesc='Bypass space availability check before backing up image/container' )
matcherImage = CliMatcher.KeywordMatcher( 'image',
      helpdesc='Image' )
matcherRemove = CliMatcher.KeywordMatcher( 'remove',
      helpdesc='Remove' )
containerNameMatcher = CliMatcher.PatternMatcher( pattern=containerNameRe,
                                           helpdesc='name of the container',
                                           helpname='WORD' )
imageNameMatcher = CliMatcher.PatternMatcher( pattern=imageNameRe,
                                       helpdesc='name of the image',
                                       helpname='WORD' )
serverNameMatcher = CliMatcher.PatternMatcher( pattern=pathRe,
                                        helpdesc='Server path',
                                        helpname='WORD' )
searchTermMatcher = CliMatcher.PatternMatcher( pattern=imageNameRe,
                                        helpdesc='Term to search for images '
                                        'in the docker hub',
                                        helpname='WORD' )
persistPathMatcher = CliMatcher.PatternMatcher( pattern=pathRe,
                                         helpdesc='Remove from persistpath',
                                         helpname='WORD' )

def runCmd( mode, args ):
   if not isContainerMgrRunning():
      mode.addWarning( 'containerMgr daemon is not running' )
      return
   cmd = [ 'docker' ]
   err = Tac.run( cmd + args, stdout=sys.stdout, stderr=Tac.CAPTURE, asRoot=True,
                  ignoreReturnCode=True )
   if err:
      for line in err.strip().split( "\n" ):
         mode.addError( line )

def getOrCreateContainerMgrBackupPath():
   path = containerMgrConfig.persistentPath + '.containermgr'
   if not os.path.exists( path ):
      os.makedirs( path, 0770 )
   return path

#--------------------------------------------------------------------------------
# container-manager start CONTAINER_NAME
#--------------------------------------------------------------------------------
class ContainerManagerStartContainernameCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager start CONTAINER_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'start' : 'Start the container',
      'CONTAINER_NAME' : containerNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      #TODO: Need to check if container exists. If it does not exist, need to check
      #      if config exists. If config exists we call run with the cmdArgs provided
      #      in config. If config does not exist we bail out printing an error
      #      message.
      runCmd( mode, [ 'start', args[ 'CONTAINER_NAME' ] ] )

BasicCli.EnableMode.addCommandClass( ContainerManagerStartContainernameCmd )

#--------------------------------------------------------------------------------
# container-manager stop CONTAINER_NAME
#--------------------------------------------------------------------------------
class ContainerManagerStopContainernameCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager stop CONTAINER_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'stop' : 'Stop the container',
      'CONTAINER_NAME' : containerNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      runCmd( mode, [ 'stop', args[ 'CONTAINER_NAME' ] ] )

BasicCli.EnableMode.addCommandClass( ContainerManagerStopContainernameCmd )

#--------------------------------------------------------------------------------
# container-manager push IMAGE_NAME
#--------------------------------------------------------------------------------
class ContainerManagerPushImagenameCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager push IMAGE_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'push' : 'Push the image to repository',
      'IMAGE_NAME' : imageNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      runCmd( mode, [ 'push', args[ 'IMAGE_NAME' ] ] )

BasicCli.EnableMode.addCommandClass( ContainerManagerPushImagenameCmd )

#--------------------------------------------------------------------------------
# container-manager pull IMAGE_NAME
#--------------------------------------------------------------------------------
class ContainerManagerPullImagenameCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager pull IMAGE_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'pull' : 'Pull the image from repository',
      'IMAGE_NAME' : imageNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      runCmd( mode, [ 'pull', args[ 'IMAGE_NAME' ] ] )

BasicCli.EnableMode.addCommandClass( ContainerManagerPullImagenameCmd )

#--------------------------------------------------------------------------------
# container-manager commit CONTAINER_NAME IMAGE_NAME
#--------------------------------------------------------------------------------
class CommitContainerCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager commit CONTAINER_NAME IMAGE_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'commit' : 'Commit the container',
      'CONTAINER_NAME' : containerNameMatcher,
      'IMAGE_NAME' : imageNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      runCmd( mode, [ 'commit', args[ 'CONTAINER_NAME' ], args[ 'IMAGE_NAME' ] ] )

BasicCli.EnableMode.addCommandClass( CommitContainerCmd )

#--------------------------------------------------------------------------------
# container-manager backup container CONTAINER_NAME [ IMAGE_NAME ] [ force ]
#--------------------------------------------------------------------------------
class BackupContainerCmd( CliCommand.CliCommandClass ):
   syntax = ( 'container-manager backup container CONTAINER_NAME [ IMAGE_NAME ] '
                                                                 '[ force ]' )
   data = {
      'container' : 'Backup the container',
      'container-manager' : matcherContainerManager,
      'backup' : matcherBackup,
      'CONTAINER_NAME' : containerNameMatcher,
      'IMAGE_NAME' : imageNameMatcher,
      'force' : matcherForce,
   }

   @staticmethod
   def handler( mode, args ):
      containerName = args[ 'CONTAINER_NAME' ]
      imageName = args.get( 'IMAGE_NAME' )
      force = 'force' in args
      if not containerConfig.container.get( containerName ):
         print 'container %s not found' % containerName
         return
      if not containerConfig.container[ containerName ].imageName:
         print 'No image configured for container %s' % containerName
         return
      if imageName == 'force':
         force = imageName
         imageName = None
      # Check whether we have atleast 90% space available
      # after we save the backed up image.
      # Find the space left at /mnt/flash/
      path = getOrCreateContainerMgrBackupPath()
      actualImage = containerConfig.container[ containerName ].imageName
      if not imageName:
         imageName = actualImage
      if not force:
         isEnoughSpaceLeft, maxImageSize = isEnoughSpacePresent( path )
         if isEnoughSpaceLeft:
            # Find the imageSize
            size = imageSize( actualImage )
            if size > maxImageSize:
               mode.addError( 'Less space available. '
                              'Unable to save image for container %s' %
                              containerName )
               return
         else:
            mode.addError( 'Less space available. '
                           'Unable to save image for container %s' % containerName )
            return

      if imageName != actualImage:
         mode.addWarning( 'Image %s is different from configured image %s for '\
                          'container %s. Consider changing the configured image '\
                          'for container %s to %s' %
                          ( imageName, actualImage, containerName, containerName,
                            imageName ) )
      path += '/%s.tar' % containerName
      commitArgs = [ 'commit', containerName, imageName ]
      try:
         imageId = Tac.run( [ 'docker' ] + commitArgs, stdout=Tac.CAPTURE,
                            stderr=sys.stdout, asRoot=True )
         imageId = imageId.split( '\n' )[ 0 ]
         print ( 'Container %s has been committed. Backing up created %s image at '
                  '%s' % ( containerName, imageId, path ) )
      except Tac.SystemCommandError as e:
         mode.addError( 'container cannot be committed due to %s' % e.output )
         return
      args = [ 'save' ]
      args += [ '-o', path, imageName ]
      runCmd( mode, args )
      subprocess.call( 'sudo chmod 770 %s' % path, shell=True )

BasicCli.EnableMode.addCommandClass( BackupContainerCmd )

#--------------------------------------------------------------------------------
# container-manager backup image IMAGE_NAME [ force ]
#--------------------------------------------------------------------------------
class ContainerManagerBackupImageImagenameCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager backup image IMAGE_NAME [ force ]'
   data = {
      'container-manager' : matcherContainerManager,
      'backup' : matcherBackup,
      'image' : matcherImage,
      'IMAGE_NAME' : imageNameMatcher,
      'force' : matcherForce,
   }

   @staticmethod
   def handler( mode, args ):
      imageName = args[ 'IMAGE_NAME' ]
      force = 'force' in args
      # Check if image is present
      if not imagePresent( imageName ):
         print 'image %s not found' % imageName
         return
      # Check whether we have atleast 90% space available
      # after we save the backed up image.
      # Find the space left at /mnt/flash/
      path = getOrCreateContainerMgrBackupPath()
      if not force:
         isEnoughSpaceLeft, maxImageSize = isEnoughSpacePresent( path )
         if isEnoughSpaceLeft:
            # Find the imageSize
            size = imageSize( imageName )
            if size > maxImageSize:
               mode.addError( 'Less space available. '
                              'Unable to save image %s' % imageName )
               return
         else:
            mode.addError( 'Less space available. '
                           'Unable to save image %s' % imageName )
            return
      args = [ 'save' ]
      # rename fails with invlaid argument when dest name has : in it. So
      # replacing it with _
      path += '/%s.tar' % imageName.replace( ':', '_' )
      args += [ '-o', path, imageName ]
      runCmd( mode, args )
      subprocess.call( 'sudo chmod 770 %s' % path, shell=True )

BasicCli.EnableMode.addCommandClass( ContainerManagerBackupImageImagenameCmd )

#--------------------------------------------------------------------------------
# container-manager backup remove CONTAINER_NAME [ PERSISTENT_PATH ]
#--------------------------------------------------------------------------------
class RemoveBackupCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager backup remove CONTAINER_NAME [ PERSISTENT_PATH ]'
   data = {
      'container-manager' : matcherContainerManager,
      'backup' : matcherBackup,
      'remove' : matcherRemove,
      'CONTAINER_NAME' : containerNameMatcher,
      'PERSISTENT_PATH' : persistPathMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      containerName = args[ 'CONTAINER_NAME' ]
      persistentPath = args.get( 'PERSISTENT_PATH' )
      if not persistentPath:
         persistentPath = containerMgrConfig.persistentPath
      persistPath = persistentPath + '.containermgr'
      if not os.path.exists( persistPath ):
         print 'No backup found'
         return
      if '.tar' not in containerName:
         containerName += '.tar'
      path = persistPath + '/' + containerName
      cmd = [ 'rm', '-rf', path ]
      err = Tac.run( cmd, stdout=sys.stdout, stderr=Tac.CAPTURE, asRoot=True,
                     ignoreReturnCode=True )
      if err:
         for line in err.strip().split( "\n" ):
            mode.addError( line )

BasicCli.EnableMode.addCommandClass( RemoveBackupCmd )

#--------------------------------------------------------------------------------
# container-manager remove image IMAGE_NAME
#--------------------------------------------------------------------------------
class ContainerManagerRemoveImageImagenameCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager remove image IMAGE_NAME'
   data = {
      'container-manager' : matcherContainerManager,
      'remove' : matcherRemove,
      'image' : matcherImage,
      'IMAGE_NAME' : imageNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      runCmd( mode, [ 'rmi', args[ 'IMAGE_NAME' ] ] )

BasicCli.EnableMode.addCommandClass( ContainerManagerRemoveImageImagenameCmd )

#--------------------------------------------------------------------------------
# container-manager search SEARCH_TERM
#--------------------------------------------------------------------------------
class SearchImageCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager search SEARCH_TERM'
   data = {
      'container-manager' : matcherContainerManager,
      'search' : 'Search for the images in the docker hub',
      'SEARCH_TERM' : searchTermMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      runCmd( mode, [ 'search', args[ 'SEARCH_TERM' ] ] )

BasicCli.EnableMode.addCommandClass( SearchImageCmd )

#------------------------------------------------------------------------------------
# Have the Cli Agent mount all neeeded state from sysdb
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global containerMgrConfig, containerConfig
   containerMgrConfig = LazyMount.mount( entityManager, 'containerMgr/config',
                                         'ContainerMgr::ContainerMgrConfig', 'r' )
   containerConfig = LazyMount.mount( entityManager,
                                      'containerMgr/container/config',
                                      'ContainerMgr::ContainerConfig', 'r' )
