#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliParser
import CliCommand
import CliMatcher
import ConfigMount
import math
from CliMode.ContainerMgrMode import ContainerMgrMode
from CliMode.RegistryMode import RegistryMode
from CliMode.ContainerMode import ContainerMode
import DesCrypt
from ContainerMgrCliLib import containerNameRe, imageNameRe, pathRe

containerMgrConfig = None
containerConfig = None

#------------------------------------------------------------------------------------
# ContainerMgr tokens
#------------------------------------------------------------------------------------
registryNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: containerMgrConfig.registryConfig.keys(),
   pattern=pathRe,
   helpname='WORD',
   helpdesc='name of the registry' )

containerNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: containerConfig.container.keys(),
   pattern=containerNameRe,
   helpname='WORD',
   helpdesc='name of the container' )

imageNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: [ container.imageName for container in \
                  containerConfig.container.values() ],
   pattern=imageNameRe,
   helpname='WORD',
   helpdesc='Image name of the container' )

class ContainerMgrConfigMode( ContainerMgrMode, BasicCli.ConfigModeBase ):
   name = "Container Manager Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.session_ = session
      ContainerMgrMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RegistryConfigMode( RegistryMode, BasicCli.ConfigModeBase ):
   name = "Container Manager Registry Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, registryName ):
      self.registryName_ = registryName
      self.session_ = session
      RegistryMode.__init__( self, registryName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ContainerConfigMode( ContainerMode, BasicCli.ConfigModeBase ):
   name = "Container Manager Container Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, containerName ):
      self.containerName_ = containerName
      self.session_ = session
      ContainerMode.__init__( self, containerName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#--------------------------------------------------------------------------------
# [ no | default ] container-manager
#--------------------------------------------------------------------------------
class ContainerManagerCmd( CliCommand.CliCommandClass ):
   syntax = 'container-manager'
   noOrDefaultSyntax = syntax
   data = {
      'container-manager' : 'Configure Container Manager',
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ContainerMgrConfigMode )
      containerMgrConfig.daemonEnable = True
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Delete all the container configs
      for containerName in containerConfig.container:
         ContainerContainernameCmd.noOrDefaultHandler( mode,
               { 'CONTAINER_NAME': containerName } )

      # Delete all the registry configs
      for registryName in containerMgrConfig.registryConfig:
         RegistryRegistrynameCmd.noOrDefaultHandler( mode,
               { 'REGISTRY_NAME': registryName } )

      # Delete all the daemon Arguments
      DaemonArgsStringCmd.noOrDefaultHandler( mode, {} )

      # Finally, at the end stop the  daemon
      containerMgrConfig.daemonEnable = False

BasicCli.GlobalConfigMode.addCommandClass( ContainerManagerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] registry REGISTRY_NAME
#--------------------------------------------------------------------------------
class RegistryRegistrynameCmd( CliCommand.CliCommandClass ):
   syntax = 'registry REGISTRY_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'registry' : 'Configure Container Manager registry',
      'REGISTRY_NAME' : registryNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      registryName = args[ 'REGISTRY_NAME' ]
      childMode = mode.childMode( RegistryConfigMode,
                                  registryName=registryName )
      mode.session_.gotoChildMode( childMode )
      containerMgrConfig.newRegistryConfig( registryName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      registryName = args[ 'REGISTRY_NAME' ]
      if containerMgrConfig.registryConfig.has_key( registryName ):
         containerMgrConfig.registryConfig[ registryName ].serverName = ""
         containerMgrConfig.registryConfig[ registryName ].userName = ""
         containerMgrConfig.registryConfig[ registryName ].password = ""
         containerMgrConfig.registryConfig[ registryName ].insecure = False
         del containerMgrConfig.registryConfig[ registryName ]

ContainerMgrConfigMode.addCommandClass( RegistryRegistrynameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] insecure
#--------------------------------------------------------------------------------
class InsecureCmd( CliCommand.CliCommandClass ):
   syntax = 'insecure'
   noOrDefaultSyntax = syntax
   data = {
      'insecure' : 'Configure the registry to be insecure',
   }

   @staticmethod
   def handler( mode, args ):
      registryName = mode.registryName_
      containerMgrConfig.registryConfig[ registryName ].insecure = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      registryName = mode.registryName_
      containerMgrConfig.registryConfig[ registryName ].insecure = False

RegistryConfigMode.addCommandClass( InsecureCmd )

#--------------------------------------------------------------------------------
# [ no | default ] username USERNAME
#--------------------------------------------------------------------------------
class UsernameUsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'username USERNAME'
   noOrDefaultSyntax = 'username ...'
   data = {
      'username' : 'Configure username for the registry',
      'USERNAME' : CliMatcher.PatternMatcher( pattern=pathRe,
         helpdesc='username for the registry', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      registryName = mode.registryName_
      userName = args[ 'USERNAME' ]
      containerMgrConfig.registryConfig[ registryName ].userName = userName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      registryName = mode.registryName_
      containerMgrConfig.registryConfig[ registryName ].userName = ""

RegistryConfigMode.addCommandClass( UsernameUsernameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] password ( [ 0 ] UNENCRYPTED_PASSWD | ( 7 ENCRYPTED_PASSWD ) )
#--------------------------------------------------------------------------------
class PasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'password ( [ 0 ] UNENCRYPTED_PASSWD | ( 7 ENCRYPTED_PASSWD ) )'
   noOrDefaultSyntax = 'password ...'
   data = {
      'password' : 'Configure password for the registry',
      '0' : 'Encryption type - unencrypted',
      'UNENCRYPTED_PASSWD' : CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern=r'[^\s]{1,80}',
            helpdesc='password (up to 80 characterss)', helpname='LINE' ),
         sensitive=True ),
      '7' : 'Encryption type - encrypted',
      'ENCRYPTED_PASSWD' : CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern=r'[^\s]+',
            helpdesc='encrypted password', helpname='LINE' ),
         sensitive=True ),
   }

   @staticmethod
   def handler( mode, args ):
      registryName = mode.registryName_
      if 'UNENCRYPTED_PASSWD' in args:
         regPasswd = args[ 'UNENCRYPTED_PASSWD' ]
      else:
         try:
            regPasswd = DesCrypt.decrypt( registryName, args[ 'ENCRYPTED_PASSWD' ] )
         except: #pylint: disable-msg=bare-except
            mode.addError( "Invalid encrypted password" )
            return
      containerMgrConfig.registryConfig[ registryName ].password = regPasswd

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      registryName = mode.registryName_
      containerMgrConfig.registryConfig[ registryName ].password = ""

RegistryConfigMode.addCommandClass( PasswordCmd )

#--------------------------------------------------------------------------------
# server SERVER_NAME
#--------------------------------------------------------------------------------
class ServerServernameCmd( CliCommand.CliCommandClass ):
   syntax = 'server SERVER_NAME'
   noOrDefaultSyntax = 'server ...'
   data = {
      'server' : 'Server URL of the registry',
      'SERVER_NAME' : CliMatcher.PatternMatcher( pattern=pathRe, helpdesc='server',
         helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      registryName = mode.registryName_
      servername = args[ 'SERVER_NAME' ]
      containerMgrConfig.registryConfig[ registryName ].serverName = servername

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      registryName = mode.registryName_
      containerMgrConfig.registryConfig[ registryName ].serverName = ""

RegistryConfigMode.addCommandClass( ServerServernameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] container CONTAINER_NAME
#--------------------------------------------------------------------------------
class ContainerContainernameCmd( CliCommand.CliCommandClass ):
   syntax = 'container CONTAINER_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'container' : 'Configure Container Manager container',
      'CONTAINER_NAME' : containerNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      containerName = args[ 'CONTAINER_NAME' ]
      childMode = mode.childMode( ContainerConfigMode,
                                  containerName=containerName )
      mode.session_.gotoChildMode( childMode )
      containerConfig.newContainer( containerName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = args[ 'CONTAINER_NAME' ]
      if containerConfig.container.has_key( containerName ):
         # Remove the image from container. This will make container to stop and
         # deleted.
         containerConfig.container[ containerName ].imageName = ""
         del containerConfig.container[ containerName ]
      else:
         mode.addError( "Container %s does not exist" % containerName )

ContainerMgrConfigMode.addCommandClass( ContainerContainernameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] daemon args ARGS
#--------------------------------------------------------------------------------
class DaemonArgsStringCmd( CliCommand.CliCommandClass ):
   syntax = 'daemon args ARGS'
   noOrDefaultSyntax = 'daemon args ...'
   data = {
      'daemon' : 'Provide Container Manager Daemon Arguments',
      'args' : 'Provide Container Manager Daemon Arguments',
      'ARGS' : CliMatcher.StringMatcher(
         helpdesc='Container Manager Daemon Arguments',
         helpname='ARUGMENTS' ),
   }

   @staticmethod
   def handler( mode, args ):
      containerMgrConfig.containerMgrArgs = args[ 'ARGS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerMgrConfig.containerMgrArgs = ""

ContainerMgrConfigMode.addCommandClass( DaemonArgsStringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] persistent-path PERSIST_PATH
#--------------------------------------------------------------------------------
class PersistentPathPersistpathCmd( CliCommand.CliCommandClass ):
   syntax = 'persistent-path PERSIST_PATH'
   noOrDefaultSyntax = 'persistent-path ...'
   data = {
      'persistent-path' : 'Persistent path for storing images',
      'PERSIST_PATH' : CliMatcher.PatternMatcher( pattern=pathRe,
         helpdesc='path', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      containerMgrConfig.persistentPath = args[ 'PERSIST_PATH' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerMgrConfig.persistentPath = containerMgrConfig.defaultPersistentPath

ContainerMgrConfigMode.addCommandClass( PersistentPathPersistpathCmd )

#--------------------------------------------------------------------------------
# [ no | default ] image IMAGE_NAME
#--------------------------------------------------------------------------------
class ImageImagenameCmd( CliCommand.CliCommandClass ):
   syntax = 'image IMAGE_NAME'
   noOrDefaultSyntax = 'image ...'
   data = {
      'image' : 'Image name of the container',
      'IMAGE_NAME' : imageNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      imageName = args[ 'IMAGE_NAME' ]
      containerName = mode.containerName_
      prevImageName = containerConfig.container[ containerName ].imageName
      if prevImageName and prevImageName != imageName:
         mode.addWarning( "%s container already had %s image configured. Now, "
                        "configured with %s image, this will stop the old container "
                        "%s and restart it using %s image."
                          % ( containerName, prevImageName,
                              imageName, containerName, imageName ) )
      containerConfig.container[ containerName ].imageName = imageName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].imageName = ""
      mode.addWarning( "There is no image configured now for this container. "
                       "Container will be stopped." )

ContainerConfigMode.addCommandClass( ImageImagenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] on-boot
#--------------------------------------------------------------------------------
class OnBootCmd( CliCommand.CliCommandClass ):
   syntax = 'on-boot'
   noOrDefaultSyntax = syntax
   data = {
      'on-boot' : 'Enable onBoot configuration for the container',
   }

   @staticmethod
   def handler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].onBoot = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].onBoot = False

ContainerConfigMode.addCommandClass( OnBootCmd )

#--------------------------------------------------------------------------------
# [ no | default ] cpu cores CPU_CORES
#--------------------------------------------------------------------------------
matcherCpu = CliMatcher.KeywordMatcher( 'cpu',
      helpdesc='CPU configuration for the container' )

class CpuCoresStringCmd( CliCommand.CliCommandClass ):
   syntax = 'cpu cores CPU_CORES'
   noOrDefaultSyntax = 'cpu cores ...'
   data = {
      'cpu' : matcherCpu,
      'cores' : 'Configure the CPU cores for the container',
      'CPU_CORES' : CliMatcher.StringMatcher( helpdesc='Cpu Cores of the container',
         helpname='CPU_CORES' ),
   }

   @staticmethod
   def handler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].cpuCores = args[ 'CPU_CORES' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].cpuCores = "0"

ContainerConfigMode.addCommandClass( CpuCoresStringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] cpu shares CPU_SHARES
#--------------------------------------------------------------------------------
class CpuSharesCpusharesCmd( CliCommand.CliCommandClass ):
   syntax = 'cpu shares CPU_SHARES'
   noOrDefaultSyntax = 'cpu shares ...'
   data = {
      'cpu' : matcherCpu,
      'shares' : 'Configure the relative CPU share for the container',
      'CPU_SHARES' : CliMatcher.IntegerMatcher( 1, 2**32-1,
         helpdesc='CPU relative shares for the container' ),
   }

   @staticmethod
   def handler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].cpuShares = args[ 'CPU_SHARES' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].cpuShares = 1024

ContainerConfigMode.addCommandClass( CpuSharesCpusharesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] memory MEMORY
#--------------------------------------------------------------------------------
class MemoryMemoryCmd( CliCommand.CliCommandClass ):
   syntax = 'memory MEMORY'
   noOrDefaultSyntax = 'memory ...'
   data = {
      'memory' : ( 'Configure the maximum memory to be used by the container. '
         'Default unit is bytes. Use b for bytes, k for kilobytes, m for megabytes '
         'and g for gigabytes' ),
      'MEMORY' : CliMatcher.PatternMatcher( pattern='^[0-9]+[bkmg]?$',
         helpdesc='Maximum memory allocated to the container', helpname='Memory' ),
   }

   @staticmethod
   def handler( mode, args ):
      memory = args[ 'MEMORY' ]
      containerName = mode.containerName_
      memoryUnits = [ 'b', 'k', 'm', 'g' ]
      # check if minimum memory requirement for 5 MB is satisfied or not
      if memory.endswith( tuple( memoryUnits ) ):
         memoryUnit = memory[ len( memory ) - 1 ]
         index = memoryUnits.index( memoryUnit )
         memoryInBytes = int( int( memory[ 0:len( memory ) - 1 ] ) * math.pow( 1024,
                                                                            index ) )
      else:
         memoryInBytes = int( memory )
      if memoryInBytes < 4194304:
         mode.addError( "Minimum memory limit allowed is 4MB" )
         return
      containerConfig.container[ containerName ].memory = memory

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      # XXX_navneet: Change the default value
      defaultMemory = containerConfig.container[ containerName ].defaultMemory
      containerConfig.container[ containerName ].memory = defaultMemory

ContainerConfigMode.addCommandClass( MemoryMemoryCmd )

#--------------------------------------------------------------------------------
# [ no | default ] command COMMAND
#--------------------------------------------------------------------------------
class CommandStringCmd( CliCommand.CliCommandClass ):
   syntax = 'command COMMAND'
   noOrDefaultSyntax = 'command ...'
   data = {
      'command' : 'Configure command for the container',
      'COMMAND' : CliMatcher.StringMatcher( helpdesc='Container Command Arguments',
         helpname='Container Command' ),
   }

   @staticmethod
   def handler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].command = args[ 'COMMAND' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].command = ""

ContainerConfigMode.addCommandClass( CommandStringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] options OPTIONS
#--------------------------------------------------------------------------------
class OptionsStringCmd( CliCommand.CliCommandClass ):
   syntax = 'options OPTIONS'
   noOrDefaultSyntax = 'options ...'
   data = {
      # FIX : helpdesc needs to be more formal
      'options' :  'Configure options for container should run with',
      'OPTIONS' : CliMatcher.StringMatcher( helpdesc='Container run options',
         helpname='Container_run_options' ),
   }

   @staticmethod
   def handler( mode, args ):
      options = args[ 'OPTIONS' ]
      if '--rm' in options:
         mode.addError( "--rm option is not supported" )
         return
      containerName = mode.containerName_
      containerConfig.container[ containerName ].options = options

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      containerName = mode.containerName_
      containerConfig.container[ containerName ].options = ""

ContainerConfigMode.addCommandClass( OptionsStringCmd )


def Plugin( entityManager ):
   global containerMgrConfig, containerConfig

   containerMgrConfig = ConfigMount.mount( entityManager, 'containerMgr/config',
                                           'ContainerMgr::ContainerMgrConfig', 'w' )
   containerConfig = ConfigMount.mount( entityManager,
                                        'containerMgr/container/config',
                                        'ContainerMgr::ContainerConfig', 'w' )
