#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import math
import Tac
import sys
import re
import os
from time import time
import simplejson as json
import dateutil.parser as dp
from datetime import timedelta, datetime

containerNameRe = r'[a-zA-Z0-9][a-zA-Z0-9_.\-]+$'
imageNameRe = r'[A-Za-z0-9_\-\/\.]+(:[A-Za-z0-9_\-\.\/:]+)?$'
pathRe = r'[A-Za-z0-9_\-@:\.\/]+'
dockerRemoteAPICmd = 'localhost:4243/'
authConfigFile = '/root/.docker/config.json'
containerState = ( "exited", "running", "created", "restarting", "paused",
                   "removing", "dead" )

def isEnoughSpacePresent( path ):
   expectedFreeSpacePercent = 10
   sizeStat = os.statvfs( path )
   freeSpace = sizeStat.f_bavail * sizeStat.f_frsize
   totalSpace = sizeStat.f_blocks * sizeStat.f_frsize
   freeSpacePercent = ( freeSpace / float( totalSpace ) ) * 100
   if freeSpacePercent > expectedFreeSpacePercent:
      thresholdSpace = ( 1 - ( expectedFreeSpacePercent / 100.0 ) ) * totalSpace
      MaxImageSize = thresholdSpace - freeSpace
      return True, MaxImageSize
   return False, 0

def removeNonAscii( text ):
   '''Removing unwanted unicode and garbage ASCII characters'''
   text = re.sub( r'[\x00-\x08\x0d-\x1f]*', r'', text )
   text = re.sub( r'\n.([^a-z])', r'\n\1', text )
   return text

def imagePresent( imageName ):
   cmd = [ 'curl', '-sS', dockerRemoteAPICmd + 'images/' + imageName + '/json' ]
   output = ""
   try:
      output = Tac.run( cmd, stdout=Tac.CAPTURE,
                        stderr=sys.stderr, asRoot=True )
   except Tac.SystemCommandError:
      return False

   if output == 'No such image: %s\n' % imageName:
      return False
   return True

def imageSize( image ):
   cmd = [ 'curl', '-sS', dockerRemoteAPICmd + 'images/' + image + '/json' ]
   output = ""
   try:
      output = Tac.run( cmd, stdout=Tac.CAPTURE,
                        stderr=sys.stderr, asRoot=True )
   except Tac.SystemCommandError, e:
      print 'curl failed due to %s' % e.output
   output = json.loads( output )
   return int ( output[ 'Size' ] / 8 )

def timestampToStr( timestamp ):
   # time format is ISO 8601
   if type( timestamp ) is str:
      timestamp = timestamp.split( '.' )[ 0 ]
      parsedTime = dp.parse( timestamp )
      # pylint: disable=E1103
      timeDiff = int( datetime.utcnow().strftime( "%s" ) ) - int( time() )
      timestamp = int ( parsedTime.strftime( "%s" ) )
      timestamp -= timeDiff
   return timestamp

def unixTimestampToStr( timestamp ):
   currentTime = int( time() )
   diffTime = currentTime - timestamp
   sec = timedelta( seconds=diffTime )
   duration = datetime( 1, 1, 1 ) + sec
   if ( duration.day - 1 ) > 0:
      return '%s days ago' % ( duration.day - 1 )
   if duration.hour > 0:
      return '%s hours ago' % duration.hour
   if duration.minute > 0:
      return '%s minutes ago' % duration.minute
   if duration.second > 0:
      return '%s seconds ago' % duration.second

# Docker Remote API returns image/container size in bytes.
# This function will take bytes as input and convert it into a more
# readable string
def bytesInStr( size ):
   if ( size == 0 ):
      return '0 B'
   sizeUnits = ( "B", "KB", "MB", "GB" )
   unitIndex = int ( math.floor( math.log( size, 1024 ) ) )
   unitSizeInKb = math.pow( 1024, unitIndex )
   size = round( size / unitSizeInKb, 2 )
   return '%s %s' % ( size, sizeUnits[unitIndex] )

def getContainerArgs( args ):
   argStr = ''
   for arg in args:
      argStr += '%s ' % arg
   return argStr

def portsToString( ports ):
   listStr = []
   for port in ports:
      listStr += [ '%s:%d->%d/%s' % ( port[ 'ip' ], port[ 'publicPort' ],
                                      port[ 'privatePort' ],
                                      port[ 'portType' ] ) ]
   return ', '.join( listStr )
