# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliPlugin.ConnectivityMonitorCli as ConnectivityMonitorCli
import CliToken.Tracers
import ConnectivityMonitorModel
import LazyMount
import ShowCommand
import Tac
import TacSigint

# Globals
_status = None
_config = None

#--------------------------------------------------------------------------------
# show monitor connectivity [ host HOSTNAME ]
#--------------------------------------------------------------------------------
class MonitorConnectivityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor connectivity [ host HOSTNAME ]'
   data = {
      'monitor' : CliToken.Tracers.monitorMatcherForShow,
      'connectivity' : 'Connectivity monitor information',
      'host' : 'Connectivity monitor information of a single host',
      'HOSTNAME' : ConnectivityMonitorCli.matcherHostName,
   }
   cliModel = ConnectivityMonitorModel.ConnMonitorVrfs

   @staticmethod
   def handler( mode, args ):
      def checkNonZero( stat ):
         return bool( stat.latency or stat.packetLoss or
                      stat.jitter or stat.httpResponseTime )

      host = args.get( 'HOSTNAME' )
      connMonVrfs = ConnectivityMonitorModel.ConnMonitorVrfs()
      hostStatus = []
      vrfModels = {}
      for key in _config.hostConfig:
         if key in _status.hostStatus:
            if host:
               if key.hostName == host:
                  hostStatus.append( _status.hostStatus[ key ] )
            else:
               hostStatus.append( _status.hostStatus[ key ] )
      for status in hostStatus:
         hostConfig = _config.hostConfig[ status.key ]
         hostName = status.key.hostName
         vrfName = status.key.vrfName
         vrfConfig = _config.vrf[ vrfName ]
         vrfModel = ConnectivityMonitorModel.ConnMonitorVrf(
                                             description=vrfConfig.description )
         if vrfName in vrfModels:
            vrfModel = vrfModels[ vrfName ]
         if hostName in vrfModel.hosts:
            hostModel = vrfModel.hosts[ hostName ]
         else:
            if ( not hostConfig.ipv4Addr.isAddrZero ) or hostConfig.url :
               hostModel = ConnectivityMonitorModel.ConnMonitorHost()
               hostModel.fromTacc( hostConfig )
            else:
               continue

         for intf, stat in status.intfStats.iteritems():
            if checkNonZero( stat ):
               statModel = ConnectivityMonitorModel.Stat()
               statModel.fromTacc( stat )
               hostModel.interfaces[ intf ] = statModel
               vrfModel.hosts[ hostName ] = hostModel
         defaultStats = status.defaultStats
         if checkNonZero( defaultStats ):
            statModel = ConnectivityMonitorModel.Stat()
            statModel.fromTacc( defaultStats )
            hostModel.interfaces[ 'default' ] = statModel
            vrfModel.hosts[ hostName ] = hostModel
         vrfModels[ vrfName ] = vrfModel
         TacSigint.check()
      for name, vm in vrfModels.iteritems():
         connMonVrfs.vrfs[ name ] = vm
      return connMonVrfs

#--------------------------------------------------------------------------------
# show monitor connectivity [ host HOSTNAME ]
#--------------------------------------------------------------------------------
class MonitorConnectivityCmdV2( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor connectivity [ host HOSTNAME ]'
   data = {
      'monitor' : CliToken.Tracers.monitorMatcherForShow,
      'connectivity' : 'Connectivity monitor information',
      'host' : 'Connectivity monitor information of a single host',
      'HOSTNAME' : ConnectivityMonitorCli.matcherHostName,
   }

   cliModel = ConnectivityMonitorModel.ConnMonitorHosts

   @staticmethod
   def handler( mode, args ):
      host = args.get( 'HOSTNAME' )
      connMonitorHosts = ConnectivityMonitorModel.ConnMonitorHosts()
      if host:
         if host in _status.hostStatus.keys() and \
               host in _config.hostConfig.keys():
            hostStatus = _status.hostStatus[ host ]
            hostConfig = _config.hostConfig[ host ]
            if not hostConfig.ipv4Addr.isAddrZero:
               connMonitorHost = ConnectivityMonitorModel.ConnMonitorHostV2()
               connMonitorHost.fromTacc( hostStatus, hostConfig )
               connMonitorHosts.hosts[ host ] = connMonitorHost
      else:
         for host, hostStatus in _status.hostStatus.iteritems():
            if host in _config.hostConfig:
               hostConfig = _config.hostConfig[ host ]
               if not hostConfig.ipv4Addr.isAddrZero:
                  connMonitorHost = ConnectivityMonitorModel.ConnMonitorHostV2()
                  connMonitorHost.fromTacc( hostStatus, hostConfig )
                  connMonitorHosts.hosts[ host ] = connMonitorHost
            TacSigint.check()

      return connMonitorHosts

BasicCli.addShowCommandClass( MonitorConnectivityCmd )

def Plugin( entityManager ):
   global _status, _config

   _status = LazyMount.mount( entityManager,
         'connectivityMonitor/status', 'ConnectivityMonitor::Status', 'r' )
   _config = LazyMount.mount( entityManager,
         'connectivityMonitor/config', 'ConnectivityMonitor::Config', 'r' )
