# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import Model, Dict, Float, Str, Int
from ArnetModel import Ip4Address
import TableOutput

class Stat( Model ):
   latency = Float( help="Network latency in milliseconds to reach the host" )
   jitter = Float( help="Packet delay variance observed while probing the host" )
   packetLoss = Int( help="Percentage packet loss when reaching the host" )
   httpResponseTime = Float( help="HTTP response time in milliseconds to reach "
                             "the host" )

   def fromTacc( self, stat ):
      self.latency = round( stat.latency, 3 )
      self.jitter = round( stat.jitter, 3 )
      self.packetLoss = stat.packetLoss
      self.httpResponseTime = round( stat.httpResponseTime, 3 )

class ConnMonitorHost( Model ):
   description = Str( help="A brief description of the host" )
   ipAddr = Ip4Address( help="IPv4 address of the host" )
   url = Str( help="URL of the host" )
   interfaces = Dict( help="A mapping between source interface and the statistics "
                 "generated using it", keyType=str, valueType=Stat )

   def fromTacc( self, hostConfig ):
      self.description = hostConfig.description
      if not hostConfig.ipv4Addr.isAddrZero:
         self.ipAddr = hostConfig.ipv4Addr.stringValue
      self.url = hostConfig.url

class ConnMonitorVrf( Model ):
   description = Str( help="A brief description of the VRF" )
   hosts = Dict( help="A mapping between a host name and its probe statistics",
                 keyType=str, valueType=ConnMonitorHost )

   def render( self ):
      def createTable( headings ):
         table = TableOutput.createTable( headings )
         f = TableOutput.Format( justify='left' )
         table.formatColumns( *( f for _ in headings ) )
         return table

      nsHeadings = ( 'IP Address', 'Local Interface', 'Latency',
                   'Jitter', 'Packet Loss' )
      hsHeadings = ( 'Local Interface', 'Response Time' )
      if self.description:
         print 'Description:', self.description
      for name, host in sorted( self.hosts.items() ):
         print 'Host:', name
         if host.description:
            print 'Description:', host.description
         interfaces = sorted( host.interfaces.items() )
         if host.ipAddr:
            print 'Network Statistics:'
            table = createTable( nsHeadings )
            for intf, stat in interfaces:
               if intf == 'default':
                  intf = 'none'
               table.newRow( host.ipAddr, intf,
                             str( stat.latency ) + 'ms', str( stat.jitter ) + 'ms',
                             str( stat.packetLoss ) + '%' )
            print table.output()
         if host.url:
            print 'HTTP statistics:'
            print host.url
            table = createTable( hsHeadings )
            for intf, stat in interfaces:
               rt = str( stat.httpResponseTime ) + 'ms' \
                    if stat.httpResponseTime !=0 else 'no response'
               if intf == 'default':
                  intf = 'none'
               table.newRow( intf, rt )
            print table.output()

class ConnMonitorVrfs( Model ):
   __revision__ = 2
   vrfs = Dict( help="A mapping between VRF and the hosts which are probed "
                "in it", keyType=str, valueType=ConnMonitorVrf )

   def render( self ):
      if self.vrfs:
         print
         for vrf, connection in sorted( self.vrfs.items() ):
            if connection.hosts:
               print 'VRF:', vrf
               connection.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         ret = { 'hosts': {} }
         if 'default' not in dictRepr[ 'vrfs' ]:
            return ret
         ret[ 'hosts' ] = dictRepr[ 'vrfs' ][ 'default' ][ 'hosts' ]
         for host in list( ret[ 'hosts' ] ):
            info = ret[ 'hosts' ][ host ]
            interfaces = info[ 'interfaces' ]
            if 'default' not in interfaces:
               del ret[ 'hosts' ][ host ]
               continue
            defaultStats = interfaces[ 'default' ]
            info[ 'latency' ] = defaultStats[ 'latency' ]
            info[ 'jitter' ] = defaultStats[ 'jitter' ]
            info[ 'packetLoss' ] = defaultStats[ 'packetLoss' ]
            info[ 'httpResponseTime' ] = defaultStats[ 'httpResponseTime' ]
            info[ 'hostName' ] = host
            del info[ 'interfaces' ]
         return ret
      return dictRepr

class ConnMonitorHostV2( Model ):

   hostName = Str( help="Name of the host" )
   description = Str( help="A brief description of the host" )
   ipAddr = Ip4Address( help="IP v4 address of the host" )
   latency = Float( help="Network latency in milliseconds to reach the host" )
   jitter = Float( help="Network jitter in milliseconds to reach the host" )
   packetLoss = Int( help="Percentage packet loss when reaching the host" )
   httpResponseTime = Float( help="HTTP response time in milliseconds to reach "
                             "the host" )
   url = Str( help="URL of the host" )

   def fromTacc( self, hostStatus, hostConfig ):
      self.hostName = hostStatus.hostName
      self.description = hostConfig.description
      self.ipAddr = hostConfig.ipv4Addr.stringValue
      self.latency = float( "{0:.3f}".format( hostStatus.defaultStats.latency ) )
      self.jitter = float( "{0:.3f}".format( hostStatus.defaultStats.jitter ) )
      self.packetLoss = hostStatus.defaultStats.packetLoss
      self.httpResponseTime = \
            float( "{0:.3f}".format( hostStatus.defaultStats.httpResponseTime ) )
      self.url = hostConfig.url

   def render( self ):
      print 'Host: %s' % self.hostName
      if self.description:
         print 'Description: %s' % self.description
      print 'Network statistics:'
      print "%-25s %-15s %-15s %-10s" % \
            ( 'Ip address', 'Latency', 'Jitter', 'Packet loss' )
      print "%-25s %-15s %-15s %-30s" % \
            ( self.ipAddr,
              str( self.latency ) + 'ms',
              str( self.jitter ) + 'ms',
              str( self.packetLoss ) + '%' )
      if self.url:
         print 'HTTP statistics:'
         indentation = ' ' * 4
         if self.httpResponseTime == 0:
            display = 'NR'
         else:
            display = str( self.httpResponseTime ) + 'ms'
         print self.url
         print indentation + 'Response Time: %s' % display
      print ''

class ConnMonitorHosts( Model ):
   hosts = Dict( help="A mapping between hosts and monitor connectivity "
                      "information of the host",
                      keyType=str, valueType=ConnMonitorHostV2 )

   def render( self ):
      if not len( self.hosts ):
         return

      print ''
      for key in sorted( self.hosts ):
         self.hosts[ key ].render()

