# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import urlparse
import socket
import Arnet
import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
from CliMode.ConnectivityMonitor import ConnectivityMonitorBaseMode
from CliMode.ConnectivityMonitor import ConnectivityMonitorHostMode
from CliMode.ConnectivityMonitor import ConnectivityMonitorVrfMode
import CliParser
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.VrfCli import VrfExprFactory, getVrfNames
import CliToken.Monitor
import ConfigMount
import LazyMount
import Intf.IntfRange as IntfRange
from TypeFuture import TacLazyType
import Tac
import IpLibConsts
from IpLibTypes import VrfNameType

# To be able to mount connectivityMonitor/config when loading plugin:
# pkgdeps: rpmwith %{_libdir}/preinit/ConnectivityMonitor

IntfId = TacLazyType( 'Arnet::IntfId' )
config = None
status = None

def getVrf( mode ):
   vrf = getattr( mode, 'vrfName', None )
   if vrf is None:
      # Try to fetch from the parent mode. This is for the case when this
      # function is called from host config mode inside vrf mode
      return getattr( mode.parent_, 'vrfName', IpLibConsts.DEFAULT_VRF )
   return vrf

def getHostNames( mode=None ):
   vrfName = getVrf( mode )
   members = []
   for host in config.hostConfig:
      if host.vrfName == vrfName:
         members.append( host.hostName )
   return members

matcherHostName = CliMatcher.DynamicNameMatcher( getHostNames, 'Name of the host',
      pattern=r'[.A-Za-z0-9_:{}\[\]-]+' )

#------------------------------------------------------------------------------------
# (config)# monitor connectivity
#------------------------------------------------------------------------------------
class ConnMonitorConfigMode( ConnectivityMonitorBaseMode, BasicCli.ConfigModeBase ):
   name = 'Connectivity monitor configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConnectivityMonitorBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ConnMonitorConfigVrfMode( ConnectivityMonitorVrfMode,
                                BasicCli.ConfigModeBase ):
   name = "Connectivity monitor vrf configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      ConnectivityMonitorVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getVrf( self ):
      if self.vrfName in config.vrf:
         return config.vrf[ self.vrfName ]
      else:
         # Could be deleted from a concurrent session
         self.addWarning( 'Configuration for %s has been deleted.' % self.vrfName )
         return None

   def addDescription( self, args ):
      text = BasicCliUtil.getSingleLineInput(
             self, 'Enter VRF description (max 140 characters): ' )
      if len( text ) > 140:
         self.addError( 'Description text is too long' )
         return
      vrf = self.getVrf()
      if not vrf:
         return
      vrf.description = text

   def noDescription( self, args ):
      vrf = self.getVrf()
      if not vrf:
         return
      vrf.description = ''

class VrfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
      'VRF': VrfExprFactory( helpdesc='Configure VRF for connectivity probes' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrf = args[ 'VRF' ]
      if vrf not in getVrfNames():
         mode.addWarning( 'Invalid vrf %s. Hosts will not be operational.' % vrf )
      if vrf not in config.vrf:
         config.vrf.newMember( vrf )
      childMode = mode.childMode( ConnMonitorConfigVrfMode, vrfName=vrf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrf = args[ 'VRF' ]
      if vrf in config.vrf:
         for key in config.intfSet:
            if key.vrfName == vrf:
               del config.intfSet[ key ]
         for key in config.hostConfig:
            if key.vrfName == vrf:
               del config.hostConfig[ key ]
         del config.vrf[ vrf ]

ConnMonitorConfigMode.addCommandClass( VrfConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description
#--------------------------------------------------------------------------------
class VrfDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description'
   noOrDefaultSyntax = syntax
   data = {
      'description' : 'Configure a brief description of the VRF',
   }

   handler = ConnMonitorConfigVrfMode.addDescription
   noOrDefaultHandler = ConnMonitorConfigVrfMode.noDescription

ConnMonitorConfigVrfMode.addCommandClass( VrfDescriptionCmd )

def getIntfSets( mode ):
   vrf = getVrf( mode )
   return sorted( key.setName for key in config.intfSet if key.vrfName == vrf )

def createIntfSetKey( vrf, setName ):
   return Tac.newInstance( 'ConnectivityMonitor::IntfSetKey', vrf, setName )

def getDefaultIntfSetKey():
   # Is this the right way to set the default value ?
   return createIntfSetKey( '', '' )

setNameMatcher = CliMatcher.DynamicNameMatcher( getIntfSets, 'Interface sets',
     pattern=r'[.A-Za-z0-9_:{}\[\]-]+' )

def setIntfKeyInHosts( vrfToCompare, setKeyToCompare, valueToSet ):
   for key, host in config.hostConfig.iteritems():
      if key.vrfName == vrfToCompare and host.intfSetKey == setKeyToCompare:
         host.intfSetKey = valueToSet

class LocalInterfacesExpression( CliCommand.CliExpression ):
   expression = 'local-interfaces SET'
   data = {
      'local-interfaces': 'Configure the default interface set for probing',
      'SET': setNameMatcher
   }

#------------------------------------------------------------------------------------
# (config-mon-connectivity)# no|default local-interfaces SET default
#------------------------------------------------------------------------------------
class LocalInterfacesConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'EXPR address-only default'
   noOrDefaultSyntax = 'EXPR ...'
   data = {
      'EXPR': LocalInterfacesExpression,
      'address-only': 'Use interfaces for source addresses only',
      'default': 'Configure the given interface set as default'
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = getVrf( mode )
      vrf = VrfNameType( vrfName )
      setName = args[ 'SET' ]
      intfKey = createIntfSetKey( vrf, setName )
      if intfKey in config.intfSet:
         config.vrf[ vrf ].defaultIntfSet = intfKey
         setIntfKeyInHosts( vrfName, getDefaultIntfSetKey(), intfKey )
      else:
         mode.addError( 'Invalid set %s' % setName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setName = args[ 'SET' ]
      vrfName = getVrf( mode )
      vrf = VrfNameType( vrfName )
      defaultIntfSet = config.vrf[ vrf ].defaultIntfSet
      defaultKey = getDefaultIntfSetKey()
      if defaultIntfSet != defaultKey and defaultIntfSet.setName == setName:
         intfKey = createIntfSetKey( vrf, setName )
         setIntfKeyInHosts( vrfName, intfKey, defaultKey )
         config.vrf[ vrf ].defaultIntfSet = defaultKey

ConnMonitorConfigMode.addCommandClass( LocalInterfacesConfigCmd )
ConnMonitorConfigVrfMode.addCommandClass( LocalInterfacesConfigCmd )

def getHostKey( mode ):
   return getattr( mode, 'hostKey', None )

#------------------------------------------------------------------------------------
# (config-mc-<host>)# no|default local-interfaces SET
#------------------------------------------------------------------------------------
class LocalInterfacesHostConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'EXPR address-only'
   noOrDefaultSyntax = syntax
   data = {
      'EXPR': LocalInterfacesExpression,
      'address-only': 'Use interfaces for source addresses only'
   }

   @staticmethod
   def handler( mode, args ):
      hostKey = getHostKey( mode )
      if hostKey:
         setName = args[ 'SET' ]
         key = createIntfSetKey( hostKey.vrfName, setName )
         if key in config.intfSet:
            config.hostConfig[ hostKey ].intfSetKey = key
         else:
            mode.addError( 'Invalid set %s' % setName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostKey = getHostKey( mode )
      if hostKey:
         setName = args[ 'SET' ]
         host = config.hostConfig[ hostKey ]
         if host.intfSetKey.setName == setName:
            host.intfSetKey = config.vrf[ hostKey.vrfName ].defaultIntfSet

#------------------------------------------------------------------------------------
# (config-mon-connectivity)# no|default interface set SET INTFS
#------------------------------------------------------------------------------------
class InterfaceSetConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'interface set SET INTFS'
   noOrDefaultSyntax = 'interface set SET ...'
   data = {
      'interface': 'Configure a set of source interfaces',
      'set': 'Name of the interface set',
      'SET': setNameMatcher,
      'INTFS': IntfRange.IntfRangeMatcher()
   }

   @staticmethod
   def handler( mode, args ):
      vrf = getVrf( mode )
      setName = args[ 'SET' ]
      key = createIntfSetKey( VrfNameType( vrf ), setName )
      intfSet = config.intfSet.newMember( key )
      ids = [ IntfId( intf ) for intf in args[ 'INTFS' ].intfNames() ]
      for intf in intfSet.intf:
         if IntfId( intf ) not in ids:
            intfSet.intf.remove( IntfId( intf ) )
      for intf in ids:
         intfSet.intf.add( intf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = getVrf( mode )
      vrf = VrfNameType( vrfName )
      setName = args[ 'SET' ]
      key = createIntfSetKey( vrf, setName )
      if key in config.intfSet:
         defaultIntfSet = config.vrf[ vrf ].defaultIntfSet
         if defaultIntfSet == key:
            defaultIntfSet = getDefaultIntfSetKey()
         setIntfKeyInHosts( vrfName, key, defaultIntfSet )
         del config.intfSet[ key ]

ConnMonitorConfigMode.addCommandClass( InterfaceSetConfigCmd )
ConnMonitorConfigVrfMode.addCommandClass( InterfaceSetConfigCmd )

class ConnMonitorConfigHostMode( ConnectivityMonitorHostMode,
      BasicCli.ConfigModeBase ):
   name = 'Connectivity monitor host configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, hostKey ):
      self.hostKey = hostKey
      ConnectivityMonitorHostMode.__init__( self, ( hostKey.hostName,
                                                    hostKey.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getHost( self ):
      if self.hostKey in config.hostConfig:
         return config.hostConfig[ self.hostKey ]
      else:
         # Could be deleted from a concurrent session
         self.addWarning( 'Configuration for %s has been deleted.' % self.hostName )
         return None

   def addIp( self, addr ):
      host = self.getHost()
      if not host:
         return
      ipAddr = Arnet.IpGenAddr( addr )
      if ipAddr.af == 'ipv4':
         host.ipv4Addr = Arnet.IpGenAddr( addr )
      else:
         host.ipv6Addr = Arnet.IpGenAddr( addr )

   def noIp( self, args ):
      addr = args[ 'ADDR' ]
      host = self.getHost()
      if not host:
         return
      if host.ipv4Addr.stringValue == addr:
         host.ipv4Addr = Arnet.IpGenAddr('0.0.0.0')
      elif host.ipv6Addr.stringValue == addr:
         host.ipv6Addr = Arnet.IpGenAddr('0.0.0.0')

   def addUrl( self, args ):
      # Sanity check the URL.
      url = args[ 'URL' ]
      urlObj = urlparse.urlparse( url )
      if not urlObj.hostname:
         self.addError( '%s is an invalid url' % url )
         return None

      # Check if we can get an Ip address from the URL.
      urlIp = None
      try:
         socket.inet_aton( urlObj.hostname )
         urlIp = urlObj.hostname
      except socket.error:
         pass

      # If an Ip address is not specified, add it now.
      host = self.getHost()
      if not host:
         return None
      if urlIp and host.ipv4Addr.isAddrZero:
         self.addIp( urlIp )

      # Add the URL.
      host.url = url
      return None

   def noUrl( self, args ):
      host = self.getHost()
      if not host:
         return
      host.url = ''

   def addDescription( self, args ):
      text = BasicCliUtil.getSingleLineInput(
            self, 'Enter host description (max 140 characters): ' )
      if len( text ) > 140:
         self.addError( 'Description text is too long' )
         return
      host = self.getHost()
      if not host:
         return
      host.description = text

   def noDescription( self, args ):
      host = self.getHost()
      if not host:
         return
      host.description = ''

   def onExit( self ):
      host = self.getHost()
      if not host:
         return
      if ( not host.ipv4Addr or host.ipv4Addr.isAddrZero ) and ( not host.url ):
         self.addWarning(
               'No IP address or URL specified for host %s. Host will not be '
               'operational.' % host.key.hostName )

      BasicCli.ConfigModeBase.onExit( self )

ConnMonitorConfigHostMode.addCommandClass( LocalInterfacesHostConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Shutdown connectivity monitor',
   }

   @staticmethod
   def handler( mode, args ):
      disableConnectivityMonitor( mode )

   @staticmethod
   def noHandler( mode, args ):
      config.enabled = True

   defaultHandler = handler

ConnMonitorConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitor connectivity
#--------------------------------------------------------------------------------
def disableConnectivityMonitor( mode ):
   config.enabled = False

def noInterval( mode ):
   config.probeInterval = config.probeIntervalDefault

def delHost( mode, hostName ):
   del config.hostConfig[ hostName ]

class MonitorConnectivityCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor connectivity'
   noOrDefaultSyntax = syntax
   data = {
      'monitor' : CliToken.Monitor.monitorMatcher,
      'connectivity' : 'Connectivity monitor configuration',
   }

   @staticmethod
   def handler( mode, args ):
      config.vrf.newMember( IpLibConsts.DEFAULT_VRF )
      childMode = mode.childMode( ConnMonitorConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Restore defaults for all parameter.
      config.intfSet.clear()
      config.hostConfig.clear()
      config.vrf.clear()
      noInterval( mode )
      disableConnectivityMonitor( mode )

BasicCli.GlobalConfigMode.addCommandClass( MonitorConnectivityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] interval INTERVAL
#--------------------------------------------------------------------------------
class IntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'interval INTERVAL'
   noOrDefaultSyntax = 'interval ...'
   data = {
      'interval' : 'Configure probe interval in seconds (default is 10 seconds, ' +\
            'minimum is 5 seconds)',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 65535, helpdesc='Probe interval' ),
   }

   @staticmethod
   def handler( mode, args ):
      interval =  args[ 'INTERVAL' ]
      if interval < 5 :
         mode.addWarning( 'Using the minimum allowed interval of 5 seconds.' )
         interval = 5
      config.probeInterval = interval

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      noInterval( mode )

ConnMonitorConfigMode.addCommandClass( IntervalIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] logging
#--------------------------------------------------------------------------------
class LoggingCmd( CliCommand.CliCommandClass ):
   syntax = 'logging'
   noOrDefaultSyntax = syntax
   data = {
      'logging' : 'Enable connectivity monitor logging',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      config.loggingEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.loggingEnabled = False

ConnMonitorConfigMode.addCommandClass( LoggingCmd )

def createHostNameVrfKey( hostName, vrf ):
   return Tac.newInstance( 'ConnectivityMonitor::HostNameVrfKey', hostName, vrf )

#--------------------------------------------------------------------------------
# [ no | default ] host HOSTNAME
#--------------------------------------------------------------------------------
class HostHostnameCmd( CliCommand.CliCommandClass ):
   syntax = 'host HOSTNAME'
   noOrDefaultSyntax = syntax
   data = {
      'host' : 'Configure host parameters',
      'HOSTNAME' : matcherHostName,
   }

   @staticmethod
   def handler( mode, args ):
      hostName = args[ 'HOSTNAME' ]
      vrf = VrfNameType( getVrf( mode ) )
      key = createHostNameVrfKey( hostName, vrf )
      host = config.hostConfig.newMember( key )
      defaultKey = getDefaultIntfSetKey()
      if config.vrf[ vrf ].defaultIntfSet != defaultKey and \
         host.intfSetKey == defaultKey:
         host.intfSetKey = config.vrf[ vrf ].defaultIntfSet

      # Transition into the new mode.
      childMode = mode.childMode( ConnMonitorConfigHostMode, hostKey=key )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostName = args[ 'HOSTNAME' ]
      vrf = VrfNameType( getVrf( mode ) )
      key = createHostNameVrfKey( hostName, vrf )
      del config.hostConfig[ key ]

ConnMonitorConfigMode.addCommandClass( HostHostnameCmd )
ConnMonitorConfigVrfMode.addCommandClass( HostHostnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip ADDR
#--------------------------------------------------------------------------------
class IpCmd( CliCommand.CliCommandClass ):
   syntax = 'ip ADDR'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : 'Configure IP address of the host',
      'ADDR' : IpAddrMatcher.ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.addIp( args[ 'ADDR' ] )

   noOrDefaultHandler = ConnMonitorConfigHostMode.noIp

ConnMonitorConfigHostMode.addCommandClass( IpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] url URL
#--------------------------------------------------------------------------------
class UrlUrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url' : 'Configure a URL for the host',
      'URL' : CliMatcher.PatternMatcher( pattern='.+',
         helpdesc='URL of the host', helpname='url' ),
   }

   handler = ConnMonitorConfigHostMode.addUrl
   noOrDefaultHandler = ConnMonitorConfigHostMode.noUrl

ConnMonitorConfigHostMode.addCommandClass( UrlUrlCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description
#--------------------------------------------------------------------------------
class HostDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description'
   noOrDefaultSyntax = syntax
   data = {
      'description' : 'Configure a brief description of the host',
   }

   handler = ConnMonitorConfigHostMode.addDescription
   noOrDefaultHandler = ConnMonitorConfigHostMode.noDescription

ConnMonitorConfigHostMode.addCommandClass( HostDescriptionCmd )

def Plugin( entityManager ):
   global config, status

   config = ConfigMount.mount( entityManager, 'connectivityMonitor/config',
                               'ConnectivityMonitor::Config', 'w' )
   status =  LazyMount.mount( entityManager, 'connectivityMonitor/status',
                              'ConnectivityMonitor::Status', 'r' )
