# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.FileCli
import CliSession
import CommonGuards
import ConfigSessionCli
import ConfigSessionCommon
import ConfigSessionModel
import FileCliUtil
import FileUrl
import SessionUrlUtil
import ShowCommand
import ShowRunOutputModel
import Url

configKwMatcher = CliMatcher.KeywordMatcher(
   'configuration',
   helpdesc='Show configuration information' )

sessionsKwMatcher = CliMatcher.KeywordMatcher(
   'sessions',
   helpdesc='List configuration sessions' )
sessionConfigKwMatcher = CliMatcher.KeywordMatcher(
   'session-config',
   helpdesc='Operating configuration of session' )
namedKwMatcher = CliMatcher.KeywordMatcher( 'named', helpdesc='Session named' )
sessionNameMatcher = CliMatcher.PatternMatcher( r'\S+',helpname='WORD',
                                            helpdesc='Name for session' )
pythonReMatcher = CliMatcher.PatternMatcher( r'\S+', helpname='WORD',
                                         helpdesc='Python regular expression' )

#-----------------------------------------------------------------------------------
# show session-config [ named SESSION_NAME ] [ sanitized ] [ all ] [ detail ]
#-----------------------------------------------------------------------------------
class ShowSessionConfig( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show session-config [ named SESSION_NAME ] '
              '[ sanitized ] [ all ] [ detail ]' )
   data = { 
            'session-config': sessionConfigKwMatcher,
            'named': namedKwMatcher,
            'SESSION_NAME': sessionNameMatcher,
            'sanitized': 'Sanitized Output',
            'all': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'all',
                  helpdesc='Configuration with defaults' ),
               hidden=True ),
            'detail': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'detail',
                  helpdesc='Detail configuration with defaults' ),
               hidden=True ),
          }
   cliModel = ShowRunOutputModel.Mode
   privileged = True
   handler = ConfigSessionCli.showSessionConfig

BasicCli.addShowCommandClass( ShowSessionConfig )

#-----------------------------------------------------------------------------------
# show session-config [ named SESSION_NAME ] diffs [ ignore-sequence-number ]
#-----------------------------------------------------------------------------------
class ShowSessionConfigDiffs( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show session-config [ named SESSION_NAME ] diffs '
              '[ ignore-sequence-number ]' )
   data = { 
            'session-config': sessionConfigKwMatcher,
            'named': namedKwMatcher,
            'SESSION_NAME': sessionNameMatcher,
            'diffs': 'Differences from running-config',
            'ignore-sequence-number': 'Ignore sequence numbers in certain commands',
          }
   handler = ConfigSessionCli.showSessionConfig
   privileged = True

BasicCli.addShowCommandClass( ShowSessionConfigDiffs )

#-----------------------------------------------------------------------------------
# show configuration sessions [ detail ]
#-----------------------------------------------------------------------------------
class ShowConfigSession( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions [ detail ]'
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'detail': 'List configuration session details',
          }
   cliModel = ConfigSessionModel.Sessions
   handler = ConfigSessionCli.showConfigSessions

BasicCli.addShowCommandClass( ShowConfigSession )

#-----------------------------------------------------------------------------------
# show configuration sessions memory
#-----------------------------------------------------------------------------------
class ShowConfigSessionMem( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions memory'
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'memory': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'memory',
                  helpdesc='Show memory utilization (may take time to compute)' ),
               guard=CommonGuards.ssoStandbyGuard )
          }
   cliModel = ConfigSessionModel.SessionMemoryTable
   handler = ConfigSessionCli.showConfigSessionsMemory

BasicCli.addShowCommandClass( ShowConfigSessionMem )

#-----------------------------------------------------------------------------------
# show configuration sessions roots
# [ { ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) } ] [ detail ]
#-----------------------------------------------------------------------------------
class ShowConfigSessionsRoots( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show configuration sessions roots '
              '[ { ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) } ] '
              '[ detail ]' )
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'roots': 'roots',
            'attribute': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'attribute',
                  helpdesc='match attribute' ),
               maxMatches=1 ),
            'type': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher( 'type', helpdesc='match type' ),
               maxMatches=1 ),
            'ATTR_PATTERN': pythonReMatcher,
            'TYPE_PATTERN': pythonReMatcher,
            'detail': 'show root mount paths',
          }
   hidden = True
   handler = ConfigSessionCli.rootWalk

BasicCli.addShowCommandClass( ShowConfigSessionsRoots )

#-----------------------------------------------------------------------------------
# show configuration sessions debug
#-----------------------------------------------------------------------------------
class ShowConfigSessionDebug( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration sessions debug'
   data = { 
            'configuration': configKwMatcher,
            'sessions': sessionsKwMatcher,
            'debug': 'Debug command',
          }
   hidden = True
   handler = ConfigSessionCli.showEntityCopyDebug

BasicCli.addShowCommandClass( ShowConfigSessionDebug )

######################################################
#    show running-config diffs reparse
######################################################

def showRunningConfigDiffsBySession( mode ):
   surl = FileUrl.localStartupConfig( *Url.urlArgsFromMode( mode ) )
   result = ConfigSessionCommon.configReplaceSession( mode,
                                                      surl,
                                                      ignoreErrors=True,
                                                      ignoreENOENT=True,
                                                      replace=True )
   sessionName, response = result
   if response:
      mode.addError( response )
      return

   try:
      # now get the session config
      sessionConfigUrl = SessionUrlUtil.sessionConfig(
         *Url.urlArgsFromMode( mode ),
         sessionName=sessionName )

      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
      FileCliUtil.diffFile( mode, sessionConfigUrl, runningConfigUrl )
   finally:
      # abort exssions
      CliSession.abortSession( mode.entityManager )
      CliSession.exitSession( mode.entityManager )
      CliSession.deleteSession( mode.entityManager, sessionName )

class ShowRunningConfigDiffsReparse( ShowCommand.ShowCliCommandClass ):
   syntax = "show running-config diffs reparse"
   data = {
      "running-config" : CliPlugin.FileCli.runningConfigAfterShowKw,
      "diffs" : CliPlugin.FileCli.runningConfigDiffsKw,
      "reparse" : "Reparsing startup-config to detect diffs (slower)"
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      showRunningConfigDiffsBySession( mode )

BasicCli.addShowCommandClass( ShowRunningConfigDiffsReparse )
