#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool, Dict, Enum, Int, Model, Str
from CliModel import Float
import ArPyUtils
import TableOutput
import Tac
import datetime

class Instance( Model ):
   user = Str( help="User that opened the session" )
   terminal = Str( help="Terminal where the session is open" )
   currentTerminal = Bool( help="The session corresponds to the current terminal" )

# show configuration sessions [detail]
class Session( Model ):
   state = Enum( help="State of the session",
                 values=[ state for state in
                          Tac.Type( 'Cli::Session::SessionState' ).attributes ] )
   commitBy = Float( help="Timestamp by when the session needs to be committed",
                      optional=True )
   description = Str( help="Description of the session" )
   instances = Dict( keyType=int, valueType=Instance,
                     help="Instances where the session is open, indexed by process "
                          "ID" )

class Sessions( Model ):
   sessions = Dict( keyType=str, valueType=Session,
                    help="A mapping between the Session name and information" ) 

   maxSavedSessions = Int( help="Maximum number of saved sessions" )

   maxOpenSessions = Int( help="Maximum number of open sessions" )

   _renderDetail = Bool(
      help="Determines whether additional information should be rendered",
      default=False )

   _renderDescriptions = Bool( 
      help="Determines whether descriptions should be rendered i.e. at least one " 
      "session has a description" )

   def render( self ):
      # Format the table
      headings = [ "", "Name", "State", "User", "Terminal" ]
      fmt_header = TableOutput.Format( isHeading=True, border=True )
      fmt_header.borderStartPosIs( 1 )
      fmt_star = TableOutput.Format( justify="left", minWidth=1, maxWidth=1,
                                     isHeading=True )
      fmt_star.padLimitIs( True )
      fmt_name = TableOutput.Format( justify="left", isHeading=True )
      fmt_name.padLimitIs( True )
      fmt = TableOutput.Format( justify="left" )
      formats = [ fmt_star, fmt_name, fmt, fmt, fmt ]

      # Add additional headings to the table as necessary
      if self._renderDetail:
         headings.append( "PID" )
         fmt_pid = TableOutput.Format( justify="right" )
         formats.append( fmt_pid )
         headings.append( "Commit Time Left" )
         fmt_time = TableOutput.Format( justify="right" )
         formats.append( fmt_time )
      if self._renderDescriptions:
         headings.append( "Description" )
         fmt_description = TableOutput.Format( justify="left", wrap=True )
         formats.append( fmt_description )

      # Create table and create headings row
      tbl = TableOutput.TableFormatter( 0, None )
      tbl.startRow( fmt_header )
      for h in headings:
         tbl.newCell( h )

      tbl.formatColumns( *tuple( formats ) )

      # Fill the table 
      for name, session in sorted( self.sessions.items() ):
         if len( session.instances ) == 0:
            instances = ( ( "", Instance( user="", terminal="",
                                          currentTerminal=False ) ), )
         else:
            instances = session.instances.iteritems()
         for pid, inst in instances:
            star = "*" if inst.currentTerminal else ""
            tbl.newRow( star, name, session.state, inst.user, inst.terminal )
            if self._renderDetail:
               tbl.newCell( str( pid ) )

            if self._renderDetail:
               if session.commitBy and session.commitBy < Tac.endOfTime:
                  tbl.newCell( ArPyUtils.formatSeconds(
                     session.commitBy - Tac.now() ) )
               else:
                  tbl.newCell( '' )
            if self._renderDescriptions:
               if session.description:
                  tbl.newCell( session.description ) 

      print 'Maximum number of completed sessions:', self.maxSavedSessions
      print 'Maximum number of pending sessions:', self.maxOpenSessions
      print
      print tbl.output()

class SessionMemory( Model ):
   memory = Int( help='Memory in bytes used by the session' )

class SessionMemoryTable( Model ):
   sessions = Dict( keyType=str, valueType=SessionMemory,
                    help="A mapping between the Session name and information" ) 
   totalMemory = Int( help='Memory in bytes used by all sessions' )

   def render( self ):
      headings = [ "Name", "Memory (Bytes)" ]
      fmt_header = TableOutput.Format( isHeading=True, border=True )
      fmt_name = TableOutput.Format( justify="left", isHeading=True )
      fmt_name.padLimitIs( True )
      fmt_memory = TableOutput.Format( justify="right" )
      formats = [ fmt_name, fmt_memory ]

      tbl = TableOutput.TableFormatter( 0, None )
      tbl.startRow( fmt_header )
      for h in headings:
         tbl.newCell( h )
      tbl.formatColumns( *tuple( formats ) )
      for name, session in sorted( self.sessions.items() ):
         tbl.newRow( name, session.memory )

      print tbl.output()
      print 'Total memory used by configuration sessions:', self.totalMemory, 'bytes'

class Checkpoint( Model ):
   fileUser = Str( help="The user who created the checkpoint file")
   fileDate = Float( help="The utc timestamp of when the checkpoitn file created" )
   
class Checkpoints( Model ):
   """
   A model that represents information about checkpoint direcotory. This is 
   returned on execution of command 'show configure checkpoint'
   """
   checkpoints = Dict( keyType=str, valueType=Checkpoint,
                       help="A mapping between the checkpoint name and info" )
   maxCheckpoints = Int( help="Maximum number of checkpoints" )

   def render( self ):
      headings = [ 'Filename', 'Date', 'User' ]
      fmt_header = TableOutput.Format( isHeading=True, border=True )
      fmt_filename = TableOutput.Format( justify='center', isHeading=True )
      fmt_filename.padLimitIs( True )
      fmt_date = TableOutput.Format( justify='center' )

      table = TableOutput.TableFormatter( 0, None )
      table.startRow( fmt_header )
      for h in headings:
         table.newCell( h )
      table.formatColumns( fmt_filename, fmt_date )
      
      for name, info in sorted( self.checkpoints.items(), 
                                key=lambda x: x[ 1 ].fileDate ):
         # convert the utc timestamp into local time
         fileDateStr = datetime.datetime.fromtimestamp( 
                        info.fileDate ).strftime( '%Y-%m-%d %H:%M:%S' )
         table.newRow( name, fileDateStr, info.fileUser )

      print 'Maximum number of checkpoints:', self.maxCheckpoints
      print table.output()
