#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import cPickle
import re
import time

import Ark
import BasicCli
import BasicCliSession
import BasicCliModes
import CheckpointCliLib
import CliCommand
import CliMatcher
import CliPlugin.SessionCli as SessionCli
import CliPlugin.TechSupportCli
import CliSession
import CliToken.Configure
import ConfigMount
import ConfigSessionCommon
import ConfigSessionModel
import FileCliUtil
import FileUrl
import LazyMount
import ShowRunOutputModel
import TableOutput
import Tac
import Tracing
import Url, SessionUrlUtil
import UrlPlugin.SessionUrl as USU # pylint: disable-msg=F0401

t0 = Tracing.t0

serviceConfig = None
cliConfig = None

# configure replace <source>
# configure replace <source> force      [Hidden command, for backward compatibility]
# configure replace <source> ignore-errors
# <source> : url (session:, flash:, clean-config) | 'named <session name>'

@Ark.synchronized( SessionCli.sessionLock )
def configureReplace( mode, surl=None, ignoreErrors=False, debugAttrChanges=False,
                      sessionName="", md5=None, replace=True, noCkp=False ):
   # durl isn't literally running-config, but after commit that's what
   # we're going to end up modifying.  So, if surl is running-config, then...
   durl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ) )
   if surl == durl and not debugAttrChanges:
      mode.addError( "Source and destination are the same file" )
      return None

   if not BasicCliSession.CONFIG_LOCK.canRunConfigureCmds():
      raise ConfigMount.ConfigMountProhibitedError(
         ConfigMount.ConfigMountProhibitedError.CONFIG_LOCKED )

   em = mode.session_.entityManager_
   CliSession.exitSession( em )

   targetStr = "unknown"
   configSource = ""
   configSourceUrl = ""
   if ignoreErrors is None:
      ignoreErrors = False
   if surl and not sessionName:
      targetStr = surl
      configSource = "url"
      configSourceUrl = str( surl )
   if not surl and sessionName:
      configSource = "session"
      configSourceUrl = sessionName
      targetStr = 'configuration session ' + sessionName

   if CliSession.isCommitTimerInProgress():
      if configSourceUrl == CliSession.sessionStatus.preCommitConfig:
         # We allow replace back to pre-commit-config
         pass
      else:
         mode.addError( 'Cannot configure replace while session %s is pending '
                        'commit timer' % CliSession.commitTimerSessionName() )
         return None

   # Generate the session used for config replace
   result = ConfigSessionCommon.configReplaceSession( mode,
                                                      surl,
                                                      ignoreErrors=ignoreErrors,
                                                      sessionName=sessionName,
                                                      md5=md5,
                                                      replace=replace )
   targetSessionName, response = result
   if response:
      mode.addError( response )
      ConfigSessionCommon.doLog(
            ConfigSessionCommon.SYS_CONFIG_REPLACE_FAILURE, targetStr )
      return None

   if noCkp:
      checkpointUrl = None
   else:
      checkpointUrl = CheckpointCliLib.saveCheckpoint(
         mode,
         cliConfig.sysdbObj().maxCheckpoints )
   # Commit
   t0( 'Committing session', targetSessionName )
   response = CliSession.commitSession( em, debugAttrChanges, 
                                        sessionName=targetSessionName )
   if response:
      mode.addError( response )
      if replace:
         ConfigSessionCommon.doLog(
            ConfigSessionCommon.SYS_CONFIG_REPLACE_FAILURE, targetStr )
      return None
   else:
      # Invoke commit handlers and rollback to checkpointUrl if error
      onCommitStatus = ConfigSessionCommon.invokeOnCommitHandlersOrRevert( mode,
                         targetSessionName, checkpointUrl )
      if onCommitStatus is None:
         if replace:
            ConfigSessionCommon.doLog(
               ConfigSessionCommon.SYS_CONFIG_REPLACE_SUCCESS, targetStr )
         mode.session_.addToHistoryEventTable( "commandLine", configSource,
                                               "running", configSourceUrl, "",
                                               runningConfigChanged=True )
         return True
      else:
         mode.addError( onCommitStatus )
         return False

class ConfigureReplace( CliCommand.CliCommandClass ):
   syntax = ( "configure replace SURL [ ignore-errors | force ] "
              "[ debug-attribute ] [ md5 MD5 ] [ skip-checkpoint ]" )
   data = {
      "configure" : CliToken.Configure.configureParseNode,
      "replace" : 'Replace configuration state',
      "SURL" : FileCliUtil.copySourceUrlExpr(),
      "ignore-errors" : 'Replace config, ignoring any errors in loading the config',
      "force" : CliCommand.Node( CliMatcher.KeywordMatcher( "force",
                                                            helpdesc="force" ),
                                 hidden=True ),
      "debug-attribute" : CliCommand.Node(
         CliMatcher.KeywordMatcher( "debug-attribute",
                                    helpdesc="Record attribute changes" ),
         hidden=True ),
      "md5" : 'Validate input config file content against provided MD5 digest',
      'MD5' : CliMatcher.PatternMatcher( r'[0-9a-f]{32}', helpname='MD5SUM',
                                        helpdesc='32-digit hexadecimal MD5 digest' ),
      "skip-checkpoint" : 'Skip checkpointing of the current config'
      }
   @staticmethod
   def handler( mode, args ):
      surl = args[ "SURL" ]
      ignoreErrors = 'ignore-errors' in args or 'force' in args
      debugAttrChanges = 'debug-attribute' in args
      md5 = args.get( 'MD5' )
      noCkp = 'skip-checkpoint' in args
      configureReplace( mode, surl=surl, ignoreErrors=ignoreErrors,
                        debugAttrChanges=debugAttrChanges,
                        md5=md5, noCkp=noCkp )

BasicCli.EnableMode.addCommandClass( ConfigureReplace )

def showSessionConfig( mode, args ):
   sessionName = args.get( 'SESSION_NAME', '' )
   showSanitized = 'sanitized' in args
   showAll = 'all' in args
   showDetail = 'detail' in args
   showDiffs = 'diffs' in args
   showNoSeqNum = 'ignore-sequence-number' in args
   em = mode.session_.entityManager_
   if not sessionName:
      sessionName = CliSession.currentSession( em )
   if not sessionName: 
      mode.addError( "Not currently in a session" )
      return None
   pcs = CliSession.sessionStatus.pendingChangeStatusDir.status.get( sessionName )
   if not pcs:
      mode.addError( "Session %s does not exist" % sessionName )
      return None
   if ( pcs.completed and not pcs.success ):
      mode.addWarning( "Session %s has been aborted.  It has no config." % 
                       sessionName )
      return None

   # shouldPrint determines the type of output ( ASCII or json ) displayed 
   # through the HTTP interface, where shouldPrint is true for ASCII and false
   # for json
   # displayJson determines the type of output displayed from the Cli, where 
   # displayJson is true when "| json" is passed to a command, and false otherwise
   # We don't support JSON output of diffs
   showJson = ( ( mode.session_.displayJson() or not mode.session_.shouldPrint() )
                and not showDiffs )

   sessionConfigUrl = SessionUrlUtil.sessionConfig(
      *Url.urlArgsFromMode( mode ),
      showAll=showAll,
      showDetail=showDetail,
      showSanitized=showSanitized,
      sessionName=sessionName,
      showJson=showJson,
      showNoSeqNum=showNoSeqNum )

   if showDiffs:
      runningConfigUrl = FileUrl.localRunningConfig( *Url.urlArgsFromMode( mode ),
                                                  showSanitized=showSanitized,
                                                  showJson=showJson,
                                                  showNoSeqNum=showNoSeqNum )
      FileCliUtil.diffFile( mode, runningConfigUrl, sessionConfigUrl )
   else:
      if showJson:
         with sessionConfigUrl.open() as f: # pylint: disable-msg=E1103
            return cPickle.load( f )
      else: 
         mode.addWarning( 'Command: show session-configuration named %s' % 
                          sessionName )
         if serviceConfig.timeInRunningConfig:
            mode.addWarning( 'Time: %s' % time.asctime( time.localtime() ) )

         FileCliUtil.showFile( mode, sessionConfigUrl )
         return ShowRunOutputModel.Mode()
   return None

def showConfigSessions( mode, args ):
   detail = 'detail' in args
   sessions = {}
   openSession = {}
   renderDescription = False
   tty = CliSession.getPrettyTtyName()
   for oss in CliSession.sessionDir.openSession.values():
      pcc = oss.session
      if pcc:
         pidList = openSession.setdefault( pcc.name, [] )
         pidList.append( oss.processId )

   pccDir = CliSession.sessionConfig.pendingChangeConfigDir.config
   statusSet = CliSession.sessionStatus.pendingChangeStatusDir.status

   for sessionName, sessionPcs in statusSet.iteritems():
      pidList = openSession.get( sessionName ) or []
      if sessionPcs.state == 'completed' and not pidList and not detail:
         continue

      instances = {}
      for pid in pidList:
         osc = CliSession.sessionInputDir.get( str( pid ) )
         user = osc.user if osc else ""
         terminal = osc.terminal if osc else ""
         instances[ pid ] = ConfigSessionModel.Instance( user=user,
                                 terminal=terminal,
                                 currentTerminal=( tty and tty == terminal ) )

      commitBy = None
 
      if detail:
         if sessionPcs.completeBy < Tac.endOfTime:
            commitBy = sessionPcs.completeBy

      description = ''
      if sessionName in pccDir:
         pcc = pccDir[ sessionName ] 
         if pcc.userString:
            description = pcc.userString
            renderDescription = True

      sessions[ sessionName ] = ConfigSessionModel.Session( state=sessionPcs.state,
                                                            instances=instances,
                                                            commitBy=commitBy,
                                                            description=description )

   return ConfigSessionModel.Sessions( sessions=sessions,
                                       maxSavedSessions=cliConfig.maxSavedSessions,
                                       maxOpenSessions=cliConfig.maxOpenSessions,
                                       _renderDetail=bool( detail ),
                                       _renderDescriptions=renderDescription )

def showConfigSessionsMemory( mode, args ):
   sessions = {}
   statusSet = CliSession.sessionStatus.pendingChangeStatusDir.status
   totalMem = 0
   for name in statusSet:
      mem = int( CliSession.sessionSize( mode.session.entityManager, name,
                 warnAfter=None ) )
      totalMem += mem
      sessions[ name ] = ConfigSessionModel.SessionMemory( memory=mem )
   return ConfigSessionModel.SessionMemoryTable( sessions=sessions,
                                                 totalMemory=totalMem )

def rootWalk( mode, args ):
   """Walk all config roots and match the specified filters."""
   detail = 'detail' in args
   attrPattern = args.get( 'ATTR_PATTERN' )
   if attrPattern:
      attrPattern = attrPattern[ 0 ]
   typePattern = args.get( 'TYPE_PATTERN' )
   if typePattern:
      typePattern = typePattern[ 0 ]
   configTypes = {}

   rootTable = TableOutput.createTable( [ 'Attribute', 'Type', 'Parent Type' ] )
   f1 = TableOutput.Format( justify='left', maxWidth=30, wrap=True )
   f1.noPadLeftIs( True )
   rootTable.formatColumns( f1, f1, f1 )

   configRoot = CliSession.configRoot( mode.entityManager )
   if detail:
      # All roots
      print '\n'.join( configRoot.root )
   for root in configRoot.root:
      # maybe skip if the root is ConfigMounted 'r'?
      t = Tac.typeNode( configRoot.rootTrie.prefixTypename( root ) )
      walk( t, attrPattern, typePattern, configTypes, rootTable )
   print rootTable.output()

def walk( t, attrPattern, typePattern, configTypes, rootTable ):
   """Walk a given type, recursively."""

   typeName = t.fullTypeName

   if configTypes.get( typeName, False ):
      # already visited
      return

   configTypes[ typeName ] = True

   if typeName.startswith( "Tac::" ):
      # Not interested in children of Tac
      return

   for attr in t.attributeQ:
      attrName = attr.name
      attrType = attr.memberType.fullTypeName
      # .memberType gives you just the base type even if it's a collection or a Ptr
      # isPtr seems to be true even when it's not a ::Ptr
      if attr.isPtr:
         attrType += '::Ptr'
      if attr.isCollection:
         attrType += '[]'
      if ( ( attrPattern is None or
             re.search( attrPattern, attrName, re.IGNORECASE ) ) and
           ( typePattern is None or
             re.search( typePattern, attrType, re.IGNORECASE ) ) ):
         # maybe skip if the attribute has a filter or is not .isLogging
         rootTable.newRow( attrName, attrType, typeName )
      # Don't walk Ptrs (that are not part of any config type?)?
      walk( attr.memberType, attrPattern, typePattern, configTypes, rootTable )

def debugEntityCopy( mode, args ):
   editType = args[ 'EDIT_TYPE' ]
   if editType == 'delete':
      editType = 'del'

   if args[ 'ACTION' ] == 'assert':
      action = 'actionAssert'
   elif args[ 'ACTION' ] == 'trace':
      action = 'actionTrace'
   else:
      assert args[ 'ACTION' ] == 'none'
      action = None

   pathname = args[ 'PATH_PATTERN' ] if 'PATH_PATTERN' in args else ''
   attr = args[ 'ATTR_PATTERN' ] if 'ATTR_PATTERN' in args else ''
   entityType = args[ 'TYPE_PATTERN' ] if 'TYPE_PATTERN' in args else ''

   userConfig = CliSession.sessionConfig.userConfig
   assert userConfig == ConfigMount.force( cliConfig )
      
   CliSession.registerEntityCopyDebug( mode.entityManager,
                                       attr, entityType=entityType,
                                       pathname=pathname, editType=editType,
                                       action=action )   
   
#--------------------------------------------------------------------------------
# debug configuration sessions
#               [ { ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) |
#                   ( pathname PATH_PATTERN ) } ]
#               EDIT_TYPE ACTION
#--------------------------------------------------------------------------------
regexMatcher = CliMatcher.PatternMatcher( pattern='\\S+',
      helpdesc='Python regular expression', helpname='WORD' )

class DebugConfigurationSessionsCmd( CliCommand.CliCommandClass ):
   syntax = ( 'debug configuration sessions '
              '[ {  ( attribute ATTR_PATTERN ) | ( type TYPE_PATTERN ) | '
                   '( pathname PATH_PATTERN ) } ]'
              'EDIT_TYPE ACTION' )
   data = {
      'debug': 'Debug command',
      'configuration': 'Debug configuration',
      'sessions': 'Debug entity copy of obj/type/attr changes',
      'attribute': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'attribute',
            helpdesc='Match attribute' ),
         maxMatches=1 ),
      'ATTR_PATTERN': regexMatcher,
      'type': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'type', helpdesc='Match type' ),
         maxMatches=1 ),
      'TYPE_PATTERN': regexMatcher,
      'pathname': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'pathname', helpdesc='Match pathname' ),
         maxMatches=1 ),
      'PATH_PATTERN': regexMatcher,
      'EDIT_TYPE': CliMatcher.EnumMatcher(
         { 'delete': 'Match if deleted',
           'set': 'Match if attr is set'} ),
      'ACTION': CliMatcher.EnumMatcher(
         { 'assert': 'Assert if match description',
           'trace': 'Trace if match description',
           'none': 'Delete description' } )
   }
   handler = debugEntityCopy
   hidden = True

BasicCliModes.EnableMode.addCommandClass( DebugConfigurationSessionsCmd )

def showEntityCopyDebug( mode, args ):
   if cliConfig.debugEntityCopy:
      headings = [( "Pathname.attr", "lh" ), ( "Operation", "l" ), 
                  ( "Action", "l" ) ]
      tbl = TableOutput.createTable( headings )
      for pathname, debugConfigCollection in cliConfig.debugEntityCopy.iteritems():
         for attrEdit, debugConfig in debugConfigCollection.debugConfig.iteritems():
            tbl.newRow( "%s.%s" % ( pathname, attrEdit.attributeName ),
                        attrEdit.editType, debugConfig.action )
      print tbl.output()
      print

   if cliConfig.debugPerTypeEntityCopy:
      headings = [( "Type::attr", "lh" ), ( "Operation", "l" ), 
                  ( "[subtree]" ), ( "Action", "l" ) ]
      tbl = TableOutput.createTable( headings )
      for attrEdit, debugTypeConfig in cliConfig.debugPerTypeEntityCopy.iteritems():
         for pathname, action in debugTypeConfig.action.iteritems():
            tbl.newRow( "%s.%s" % ( attrEdit.attributeType, 
                                    attrEdit.attributeName ),
                        attrEdit.editType, pathname, action )
      print tbl.output()
      print


def getSessionConfigFromUrl( mode, match ):
   sessionName = CliSession.currentSession( mode.session_.entityManager_ )
   return SessionUrlUtil.sessionConfig( *Url.urlArgsFromMode( mode ),
                                         sessionName=sessionName )
def getCleanConfig( mode, match ):
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   pathname = "/clean-session-config" 
   url = "session:" + pathname
   fs = Url.getFilesystem( "session:" )
   return USU.SessionUrl( fs, url, pathname, pathname, context, clean=True )

FileCliUtil.registerCopySource(
   'session-cofig',
   CliCommand.Node(
      CliMatcher.KeywordMatcher( 'session-config',
                                helpdesc='Copy from session configuration',
                                value=getSessionConfigFromUrl ),
      guard=ConfigSessionCommon.sessionGuard ),
   'session-config' )
FileCliUtil.registerCopySource(
   'clean-config',
   CliMatcher.KeywordMatcher( 'clean-config',
                              helpdesc='Copy from clean, default, configuration',
                              value=getCleanConfig ),
   'clean-config' )

FileCliUtil.registerCopyDestination(
   'session-cofig',
   CliCommand.Node(
      CliMatcher.KeywordMatcher( 'session-config',
                 helpdesc='Update (merge with) current session configuration',
                                value=getSessionConfigFromUrl ),
      guard=ConfigSessionCommon.sessionSsoGuard ),
   'session-config' )

def showTechCmds():
   return [ 'show configuration sessions detail' ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2015-02-06 00:00:47', showTechCmds )

def copyRunningBySession( mode, surl ):
   return configureReplace( mode, surl=surl, ignoreErrors=True,
                            replace=False )

def Plugin( entityManager ):
   global serviceConfig, cliConfig

   serviceConfig = LazyMount.mount( entityManager, "sys/service/config",
                                    "System::ServiceConfig", "r" )
   cliConfig = ConfigMount.mount( entityManager, "cli/session/input/config", 
                                  "Cli::Session::CliConfig", "w" )

   # register configure replace handler
   import FileCli
   FileCli.copyRunningHandler = copyRunningBySession
