# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import datetime
import operator

import Ark
import CliModel
import TableOutput

class ConfigureLockModelUser( CliModel.Model ):
   username = CliModel.Str( help='The username of who holds the lock' )
   userTty = CliModel.Str( help='TTY for the user who holds the lock' )
   userLocation = CliModel.Str( help='Source IP address or hostname for VTY '
                                       'sessions and absent for serial connections',
                                optional=True )
   lockAcquireTime = CliModel.Float( help='UTC timestamp of when the lock was '
                                            'acquired' )
   lockAcquireReason = CliModel.Str( help='Reason why the lock was acquired',
                                     optional=True )

class ConfigureLockModel( CliModel.Model ):
   userInfo = CliModel.Submodel( valueType=ConfigureLockModelUser, optional=True,
         help='Information for the user who holds the Configuration Lock' )

   def render( self ):
      if not self.userInfo:
         print( 'Currently no user has the Configuration Lock held' )
      else:
         userInfo = self.userInfo
         headings = ( 'TTY', 'User', 'Time Acquired', 'Location', 'Reason' )
         table = TableOutput.createTable( headings )
         location = userInfo.userLocation if userInfo.userLocation else '-'
         lockTime = Ark.utcTimeRelativeToNowStr( userInfo.lockAcquireTime )
         reason = userInfo.lockAcquireReason if userInfo.lockAcquireReason else '-'
         table.newRow( userInfo.userTty, userInfo.username, lockTime, location,
                       reason )
         print( table.output() )

class ConfigureLockModelHistoryInfo( CliModel.Model ):
   username = CliModel.Str( help='The username of who held the lock' )
   userTty = CliModel.Str( help='TTY for the user who held the lock' )
   userLocation = CliModel.Str( help='Source IP address or hostname for VTY '
                                       'sessions and absent for serial connections',
                                optional=True )
   lockAcquireTime = CliModel.Float( help='UTC timestamp of when the lock was '
                                            'acquired' )
   lockReleaseTime = CliModel.Float( help='UTC timestamp of when the lock was '
                                            'released', optional=True )
   lockReleaseReason = CliModel.Str( help='Reason why the lock was released',
                                     optional=True )

class ConfigureLockModelHistory( CliModel.Model ):
   history = CliModel.List( valueType=ConfigureLockModelHistoryInfo,
                            help='Information for the previous users who held '
                                 'the Configuration Lock' )

   def render( self ):
      if not self.history:
         return

      headings = ( 'TTY', 'User', 'Acquired', 'Released', 'Location', 'Reason' )
      table = TableOutput.createTable( headings )
      for history in sorted( self.history,
                             key=operator.attrgetter( 'lockAcquireTime' ),
                             reverse=True ):
         location = history.userLocation if history.userLocation else '-'
         td = datetime.datetime.fromtimestamp( history.lockAcquireTime )
         lockAcquireTime = td.strftime( '%m-%d %H:%M:%S' )
         if history.lockReleaseTime:
            td = datetime.datetime.fromtimestamp( history.lockReleaseTime )
            lockReleaseTime = td.strftime( '%m-%d %H:%M:%S' )
         else:
            lockReleaseTime = '-'
         reason = history.lockReleaseReason if history.lockReleaseReason else '-'
         table.newRow( history.userTty, history.username, lockAcquireTime,
                       lockReleaseTime, location, reason )
      print( table.output() )
