# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Ark
import BasicCliSession
import CliPlugin.ConfigLockModels
import ConfigureLock
import Logging
import UtmpDump

Logging.logD( id='SYS_CONFIG_LOCK_CLEAR',
   severity=Logging.logNotice,
   format='The configuration lock was broken manually by \'%s\'.',
   explanation=( 'The configuration lock was broken by someone other than the user '
                 'who acquired it.' ),
   recommendedAction=Logging.NO_ACTION_REQUIRED )

def showConfigLockHandler( mode, args ):
   userInfo = BasicCliSession.CONFIG_LOCK.getCurrUserInfo()
   result = CliPlugin.ConfigLockModels.ConfigureLockModel()
   if userInfo is None:
      return result

   result.userInfo = CliPlugin.ConfigLockModels.ConfigureLockModelUser()
   result.userInfo.username = userInfo.user
   result.userInfo.userTty = userInfo.tty
   result.userInfo.userLocation = userInfo.location
   result.userInfo.lockAcquireTime = Ark.switchTimeToUtc( userInfo.grabTime )
   result.userInfo.lockAcquireReason = userInfo.acquireReason
   return result

def showConfigLockHistoryHandler( mode, args ):
   result = CliPlugin.ConfigLockModels.ConfigureLockModelHistory()
   for history in BasicCliSession.CONFIG_LOCK.getHistory():
      if history.releaseTime:
         lockReleaseTime = Ark.switchTimeToUtc( history.releaseTime )
      else:
         lockReleaseTime = None
      lockAcquireTime = Ark.switchTimeToUtc( history.grabTime )
      historyInfo = CliPlugin.ConfigLockModels.ConfigureLockModelHistoryInfo()
      historyInfo.username = history.user
      historyInfo.userTty = history.tty
      historyInfo.userLocation = history.location
      historyInfo.lockAcquireTime = lockAcquireTime
      historyInfo.lockReleaseTime = lockReleaseTime
      historyInfo.lockReleaseReason = history.releaseReason
      result.history.append( historyInfo )
   return result

def configureLockCmdHandler( mode, args ):
   if BasicCliSession.CONFIG_LOCK.isLockOwnedByThread():
      mode.addWarning( 'Current thread is already holding the Configure Lock.' )
      return

   try:
      BasicCliSession.CONFIG_LOCK.acquireLock( args.get( 'REASON' ) )
   except ConfigureLock.UnableToAcquireLockException:
      mode.addError( 'Unable to acquire Configure Lock. Held by another thread' )

def configureUnlockCmdHandler( mode, args ):
   if not BasicCliSession.CONFIG_LOCK.isLockedOwned():
      mode.addWarning( 'No thread is currently holding the Configure Lock.' )
      return

   if not BasicCliSession.CONFIG_LOCK.isLockOwnedByThread():
      mode.addError( 'Unable to unlock. Thread is not holding the Config Lock.' )
      return

   BasicCliSession.CONFIG_LOCK.releaseLock()

def configureUnlockForceCmdHandler( mode, args ):
   if not BasicCliSession.CONFIG_LOCK.isLockedOwned():
      mode.addWarning( 'Configure Lock is not held by any thread.' )
      return

   if not BasicCliSession.CONFIG_LOCK.isLockOwnedByThread():
      userInfo = UtmpDump.getUserInfo()
      # pylint: disable-msg=undefined-variable
      SYS_CONFIG_LOCK_CLEAR( userInfo[ 'user' ] )
      # pylint: enable-msg=undefined-variable
   BasicCliSession.CONFIG_LOCK.releaseLock( reason=args.get( 'REASON' ) )
