#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliPlugin.CloudHaCliLib as CloudHaCliLib
import CliPlugin.CloudHaModel as CloudHaModel
import CliPlugin.CloudHaModelAws as CloudHaModelAws
import CliPlugin.CloudHaModelAzure as CloudHaModelAzure
import CliPlugin.CloudHaModelGcp as CloudHaModelGcp
import LazyMount
import ShowCommand
import Tac
from CloudUtil import defaultRecoveryWaitTime, getHypervisor
from Tracing import t0

cloudType = getHypervisor()

cloudStatus = None
proxyConfig = None

#--------------------------------------------------------------------------------
# show cloud high-availability
#--------------------------------------------------------------------------------
class CloudHighAvailabilityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cloud high-availability'
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'high-availability' : 'Cloud High Availability',
   }
   cliModel = CloudHaModel.CloudHa

   @staticmethod
   def handler( mode, args ):
      ha = CloudHaModel.CloudHa()
      config = cloudStatus.config
      if not config:
         # This happens when cloud agent is not running as in standalone cli mode
         ha.enabled = False
         return ha

      ha.failoverRecoveryTime = defaultRecoveryWaitTime
      # Make sure there is only one peer that we support for now
      assert len( config.peer ) <= 1
      for i in config.peer.values():
         # Fill from status
         defaultIp = Tac.newInstance( 'Arnet::IpGenAddr' )
         if i.peerIp != defaultIp:
            ha.peerIp = str( i.peerIp )
         ha.sourceInterface = i.bfdConfig.intf
         ha.failoverRecoveryTime = i.recoveryWaitTime
      ha.enabled = config.enable
      if cloudStatus:
         ha.configValidationStatus = cloudStatus.status.lower()
      else:
         ha.configValidationStatus = 'invalid'
      ha.haState = str.lower( cloudStatus.state ) if cloudStatus else 'disabled'
      ha.lastFailoverTime = cloudStatus.lastFailoverTime
      ha.lastRecoveryTime = cloudStatus.lastRecoveryTime
      ha.lastValidationStartTime = cloudStatus.lastValidationStartTime
      ha.lastValidationEndTime = cloudStatus.lastValidationEndTime
      ha.failovers = cloudStatus.failovers
      return ha

BasicCli.addShowCommandClass( CloudHighAvailabilityCmd )

#--------------------------------------------------------------------------------
# show cloud provider aws
#--------------------------------------------------------------------------------
class CloudProviderAwsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cloud provider aws'
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'provider' : CloudHaCliLib.matcherProvider,
      'aws' : CloudHaCliLib.nodeAws,
   }
   cliModel = CloudHaModelAws.CloudAws

   @staticmethod
   def handler( mode, args ):
      aws = CloudHaModelAws.CloudAws()
      # Fill from status
      awsConfig = cloudStatus.config
      if not awsConfig:
         # This could be when Cloud HA agent is not running
         # due to disabled or in test environ
         aws.region = ''
         aws.accessKey = ''
         aws.secretKey = ''
         return aws
      accessConfig = awsConfig.accessConfig
      aws.region = accessConfig.region
      aws.accessKey = accessConfig.accessKey
      aws.secretKey = accessConfig.secretAccess
      aws.proxy = accessConfig.proxyName
      return aws

BasicCli.addShowCommandClass( CloudProviderAwsCmd )

#--------------------------------------------------------------------------------
# show cloud provider azure
#--------------------------------------------------------------------------------
class CloudProviderAzureCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cloud provider azure'
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'provider' : CloudHaCliLib.matcherProvider,
      'azure' : CloudHaCliLib.nodeAzure,
   }
   cliModel = CloudHaModelAzure.CloudAzure

   @staticmethod
   def handler( mode, args ):
      azure = CloudHaModelAzure.CloudAzure()
      config = cloudStatus.config
      if not config:
         # This could be when Cloud HA agent is not running
         # due to disabled or in test environ
         return azure
      azureConfig = config.accessConfig
      azure.email = azureConfig.adEmail
      azure.subscriptionId = azureConfig.subscriptionId
      azure.password = azureConfig.password
      azure.sdkAuthCredentialFile = azureConfig.sdkcredFileLocation
      azure.activeCredentialsInUse =  'SDK authentication credential file' \
         if azure.sdkAuthCredentialFile else 'active directory credentials' \
         if azure.email else 'MSI'
      azure.proxy = azureConfig.proxyName
      return azure

BasicCli.addShowCommandClass( CloudProviderAzureCmd )

#-------------------------------------------------------------------------------
# show cloud provider gcp
#-------------------------------------------------------------------------------
class ShowCloudProviderGcpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cloud provider gcp'
   data = {
         'cloud' : CloudHaCliLib.nodeCloud,
         'provider' : CloudHaCliLib.matcherProvider,
         'gcp' : CloudHaCliLib.nodeGcp,
   }
   cliModel = CloudHaModelGcp.CloudGcp

   @staticmethod
   def handler( mode, args ):
      gcp = CloudHaModelGcp.CloudGcp()
      # Fill from status
      gcpConfig = cloudStatus.config
      if not gcpConfig:
         # This could be when Cloud HA agent is not running
         # due to disabled or in test environ
         gcp.project = ''
         gcp.authenticationMode = ''
         return gcp
      accessConfig = gcpConfig.accessConfig
      gcp.project = accessConfig.project
      gcp.serviceAccountFile = accessConfig.serviceFileLocation
      if gcp.serviceAccountFile:
         gcp.authenticationMode = 'Service Account'
      else:
         gcp.authenticationMode = 'Default account'
      gcp.proxy = accessConfig.proxyName
      return gcp

BasicCli.addShowCommandClass( ShowCloudProviderGcpCmd )

#--------------------------------------------------------------------------------
# show cloud proxy [ PROXY ]
#--------------------------------------------------------------------------------
class CloudProxyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cloud proxy [ PROXY ]'
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'proxy' : 'Proxy details',
      'PROXY' : CloudHaCliLib.matcherProxyValue,
   }
   cliModel = CloudHaModel.CloudProxy

   @staticmethod
   def handler( mode, args ):
      def proxyModel( proxy ):
         model = CloudHaModel.CloudProxy.proxy( name=proxy.name )

         http = proxy.httpUsername
         if http:
            http += ':'
         http += proxy.httpPassword
         if http:
            http += '@'
         model.http = http + proxy.httpProxy + ':' + str( proxy.httpPort )

         https = proxy.httpsUsername
         if https:
            https += ':'
         https += proxy.httpsPassword
         if https:
            https += '@'
         model.https = https + proxy.httpsProxy + ':' + str( proxy.httpsPort )

         return model

      proxyName = args.get( 'PROXY' )
      proxies = proxyConfig.proxy.values()
      if proxyName is not None:
         proxies = ( proxy for proxy in proxies if proxy.name == proxyName )

      model = CloudHaModel.CloudProxy()
      model.proxies = [ proxyModel( proxy ) for proxy in proxies ]

      return model

BasicCli.addShowCommandClass( CloudProxyCmd )

#--------------------------------------------------------------------------------
# show cloud high-availability routes
#--------------------------------------------------------------------------------
def showHaAwsRoutes( mode, args ):
   t0( 'showHaRoutes' )
   peer = None
   routes = CloudHaModelAws.AwsCloudHaRoutes()
   localRoutes = routes.localRoutes
   peerRoutes = routes.peerRoutes
   config = None
   # Fill from status
   config = cloudStatus.config
   if config:
      if config.peer:
         assert len( config.peer ) == 1
         peer = config.peer[ config.peer.keys()[ 0 ] ]
      for i, val in config.localRoutes.items():
         rtbId = i.routeTableId
         dest = i.destination
         route = CloudHaModelAws.AwsRoute()
         route.nextHop = str( val.nextHopRouteTarget )
         if not localRoutes.get( rtbId ):
            localRoutes[ rtbId ] = CloudHaModelAws.AwsRouteTable()
         localRoutes[ rtbId ].routes[ str( dest ) ] = route
      for i, val in config.peerRoutes.items():
         rtbId = i.routeTableId
         dest = i.destination
         route = CloudHaModelAws.AwsRoute()
         route.nextHop = str( val.nextHopRouteTarget )
         if not peerRoutes.get( rtbId ):
            peerRoutes[ rtbId ] = CloudHaModelAws.AwsRouteTable()
         peerRoutes[ rtbId ].routes[ str(dest) ] = route
   if peer:
      routes.peerName = peer.name
   return routes

def showHaAzureRoutes( mode, args ):
   t0( 'showHaAzureRoutes' )
   peer = None
   routes = CloudHaModelAzure.AzureCloudHaRoutes()
   localRoutes = routes.localRoutes
   peerRoutes = routes.peerRoutes
   config = None
   # Fill from status
   config = cloudStatus.config
   if config:
      if config.peer:
         assert len( config.peer ) == 1
         peer = config.peer[ config.peer.keys()[ 0 ] ]
      for i, val in config.localRoutes.items():
         rtbId = i.routeTableId
         dest = i.destination
         route = CloudHaModelAzure.AzureRoute()
         route.nextHop = str( val.nextHopRouteTarget )
         route.resourceGroup = str( val.resourceGroup )
         if not localRoutes.get( rtbId ):
            localRoutes[ rtbId ] = CloudHaModelAzure.AzureRouteTable()
         localRoutes[ rtbId ].routes[ str( dest ) ] = route
      for i, val in config.peerRoutes.items():
         rtbId = i.routeTableId
         dest = i.destination
         route = CloudHaModelAzure.AzureRoute()
         route.nextHop = str( val.nextHopRouteTarget )
         route.resourceGroup = str( val.resourceGroup )
         if not peerRoutes.get( rtbId ):
            peerRoutes[ rtbId ] = CloudHaModelAzure.AzureRouteTable()
         peerRoutes[ rtbId ].routes[ str(dest) ] = route
   if peer:
      routes.peerName = peer.name
   return routes

def showHaGcpRoutes( mode, args ):
   def populateRoutes( routes, model ):
      for i in routes:
         gcpRoute = CloudHaModelGcp.GcpRoute()
         gcpRoute.interface = i.interface
         gcpRoute.vpc = i.vpc
         gcpRoute.destination = i.destination.stringValue
         gcpRoute.tag = i.tag
         model.append( gcpRoute )
   t0( 'showHaRoutes' )
   peer = None
   routes = CloudHaModelGcp.GcpCloudHaRoutes()
   localRoutes = routes.localRoutes
   peerRoutes = routes.peerRoutes
   config = None
   # Fill from status
   config = cloudStatus.config
   if config:
      if config.peer:
         if len( config.peer ) != 1:
            mode.addError( 'Active Cloud HA config cannot have multiple peers' )
            return None
         peer = config.peer[ config.peer.keys()[ 0 ] ]
      populateRoutes( config.localRoutes, localRoutes )
      populateRoutes( config.peerRoutes, peerRoutes )
   if peer:
      routes.peerName = peer.name
   return routes

# This is bad. Defining it conditionally with a conditional model
# results in this command not showing up in the documentation.
if cloudType in ( 'AWS', 'Azure', 'GCP' ):
   class ShowHaRoutesCmd( ShowCommand.ShowCliCommandClass ):
      syntax = 'show cloud high-availability routes'
      data = {
            'cloud': CloudHaCliLib.nodeCloud,
            'high-availability': CloudHaCliLib.matcherHighAvailability,
            'routes': 'Cloud High Availability Routes',
      }

      if cloudType == 'AWS':
         handler = showHaAwsRoutes
         cliModel = CloudHaModelAws.AwsCloudHaRoutes
      elif cloudType == 'Azure':
         cliModel = CloudHaModelAzure.AzureCloudHaRoutes
         handler = showHaAzureRoutes
      elif cloudType == 'GCP':
         cliModel = CloudHaModelGcp.GcpCloudHaRoutes
         handler = showHaGcpRoutes

   BasicCli.addShowCommandClass( ShowHaRoutesCmd )

def Plugin( entityManager ):
   global cloudStatus
   global proxyConfig

   if cloudType == 'AWS':
      cloudStatus = LazyMount.mount( entityManager, 'cloudha/awsstatus',
                                     'Cloud::HA::AwsStatus', "r" )
   elif cloudType == 'Azure':
      cloudStatus = LazyMount.mount( entityManager, 'cloudha/azurestatus',
                                     'Cloud::HA::AzureStatus', "r" )
   elif cloudType == 'GCP':
      cloudStatus = LazyMount.mount( entityManager, 'cloudha/gcpstatus',
                                     'Cloud::HA::GcpStatus', "r" )

   proxyConfig = LazyMount.mount( entityManager, 'cloudha/proxyConfig',
                                  'Cloud::HA::ProxyConfig', "r" )
