# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddrAndMask
from CliModel import  Model, Str, List
from IntfModels import Interface
from TableOutput import Format, FormattedCell, createTable
from CloudHaModel import _getStr, showCloudTableFormat

# Cli Model for "show cloud provider gcp"
class CloudGcp( Model ):
   # Configuration
   project = Str( help="GCP project" )
   serviceAccountFile = Str( help="Service account file location", optional=True )
   authenticationMode = Str( help="Authentication mode" )
   proxy = Str( help="Proxy name", optional=True )

   def render( self ):
      t = showCloudTableFormat()
      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )
      t.newRow( FormattedCell( content="Cloud GCP Configuration:", nCols=1,
                               format=headerFormat ) )

      for ( label, value ) in (
            ( "Project", self.project ),
            ( "Authentication mode", self.authenticationMode ),
            ( "Service account file", self.serviceAccountFile ),
            ( "Proxy", self.proxy ) ):
         t.newRow( label, " : ", _getStr( value ) )

      print t.output()

class GcpRoute( Model ):
   interface = Interface( help='Interface name', optional=True )
   vpc = Str( help='VPC name', optional=True )
   destination = IpGenericAddrAndMask( help='Destination IP address and mask' )
   tag = Str( help='Route tag', optional=True )

# Cli Model for "show cloud ha routes"
class GcpCloudHaRoutes( Model ):
   peerName = Str( help='Peer name', optional=True )
   localRoutes = List( valueType=GcpRoute, help='List of local GCP routes' )
   peerRoutes = List( valueType=GcpRoute, help='List of peer GCP routes' )

   def render( self ):
      def renderRoutes( t, routes, routeType ):
         def routeKey( route ):
            return ( route.interface.stringValue, route.vpc, route.destination,
                  route.tag )
         for route in sorted( routes, key=routeKey ):
            t.newRow( self.peerName, routeType, route.destination,
                  route.interface.stringValue, route.vpc, route.tag )

      f1 = Format( justify="left" )
      f1.padLimitIs( True )
      f2 = Format( justify="right" )
      f2.padLimitIs( True )
      t = createTable( ( "Peer", "Route Type", "Destination", "Interface", "VPC",
         "Tag" ) )
      t.formatColumns( f1, f1, f2, f1, f1, f1 )
      renderRoutes( t, self.localRoutes, 'primary' )
      renderRoutes( t, self.peerRoutes, 'backup' )
      print t.output()
