# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Str, Dict
from TableOutput import Format, FormattedCell, createTable
import operator
from CloudHaModel import _getStr, showCloudTableFormat

# Cli Model for "show cloud azure"
class CloudAzure( Model ):
   # Configuration
   email = Str( help='Active Directory Email', optional=True )
   subscriptionId = Str( help='Azure subscription ID', optional=True )
   password = Str( help='Active Directory Password', optional=True )
   activeCredentialsInUse = Str( help='Credentials currently in use',
         optional=True )
   sdkAuthCredentialFile = Str( help="SDK Auth Credentials file", optional=True )
   proxy = Str( help="Proxy name", optional=True )

   def render( self ):
      t = showCloudTableFormat()
      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )
      t.newRow( FormattedCell( content="Cloud Azure Configuration:", nCols=1,
                               format=headerFormat ) )
      for( label, value ) in (
         ( "Active credentials", self.activeCredentialsInUse ),
         ( "SDK auth credentials file", _getStr( self.sdkAuthCredentialFile ) ),
         ( "Proxy name", _getStr( self.proxy ) ),
         ):
         t.newRow( label, " : ", value )
      t.newRow( "Active directory credentials", " : " )
      opStr = t.output()
      if self.email:
         for i in [
            '\tEmail : %s' % _getStr( self.email ),
            '\n\tPassword : %s' % _getStr( self.password ),
            '\n\tSubscription ID : %s' %  _getStr( self.subscriptionId ),
            '\n' ]:
            opStr +=  i
      print opStr


class AzureRoute( Model ):
   nextHop = Str( help='Next hop local IP address' )
   resourceGroup = Str( help='Route Resource Group' )

# Key type should be IpAddrAndMask but it is not supported by CliModel.
# Use str for now.
class AzureRouteTable(Model ):
   routes = Dict( valueType=AzureRoute,
            help='str(IpAddAndMask) to Route' )

# Cli Model for "show cloud ha routes"
class AzureCloudHaRoutes( Model ):
   peerName = Str( help='PeerName', optional=True )
   localRoutes = Dict( valueType=AzureRouteTable, keyType=str,
                     help='Route table ID to routeTable' )
   peerRoutes = Dict( valueType=AzureRouteTable, keyType=str,
                     help='Route Table ID to routeTable' )

   def render( self ):
      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )
      t = createTable( ( "Peer", "Route Type", "Resource Group",
         "Route ID", "Destination", "Next Hop Interface") )
      # Seems like pylint gets confused as Dict simulates member function items()
      # pylint: disable=E1101
      sortedDict = sorted( self.localRoutes.items(), key=operator.itemgetter( 0 ) )
      for rt, val in sortedDict:
         sortedRoutes = sorted( val.routes.items(), key=operator.itemgetter( 0 ) )
         for dest, route in sortedRoutes:
            t.newRow( self.peerName, 'primary', route.resourceGroup, rt, dest,
               route.nextHop )

      sortedDict = sorted( self.peerRoutes.items(), key=operator.itemgetter( 0 ) )
      for rt, val in sortedDict:
         sortedRoutes = sorted( val.routes.items(), key=operator.itemgetter( 0 ) )
         for dest, route in sortedRoutes:
            t.newRow( self.peerName, 'backup', route.resourceGroup, rt, dest,
               route.nextHop )
      print t.output()

