# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import List, Model, Str, Enum, Int, Bool, Float
from IntfModel import Interface
from ArnetModel import Ip4Address
from TableOutput import TableFormatter, Format, FormattedCell
import Tac
from Ark import timestampToStr

def _getStr( val, default="" ):
   return str( val ) if val else default

def _getCapStr( val, default="" ):
   return str( val ).capitalize() if val else default

def _getStringValue( val, default="" ):
   return val.stringValue if val else default

def showCloudTableFormat():
   t = TableFormatter( tableWidth=120 )
   f = Format( justify="left", noBreak=True, terminateRow=False )
   f2 = Format( justify='left' )
   f3 = Format( justify='left' )
   f4 = Format( justify='left' )
   f.noPadLeftIs( True )
   f.padLimitIs( True ) # makes the table compact
   f.noTrailingSpaceIs( True )
   f2.noPadLeftIs( True )
   f2.padLimitIs( True )
   f2.noTrailingSpaceIs( True )
   f3.noPadLeftIs( True )
   f3.padLimitIs( True )
   f3.noTrailingSpaceIs( True )
   f4.noTrailingSpaceIs( True )
   t.formatColumns( f, f2, f3, f4 )
   return t

# Cli Model for "show cloud proxy
class CloudProxy( Model ):
   # Configuration
   class proxy( Model ):
      name = Str( help="Proxy name" )
      http = Str( help="HTTP proxy IP addres", optional=True )
      https = Str( help="HTTPS proxy IP address", optional=True )

   proxies = List( valueType=proxy, help='configured proxies', optional=True )

   def render( self ):
      if not self.proxies:
         return
      t = showCloudTableFormat()
      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )
      t.newRow( FormattedCell( content="Cloud Proxy Configuration:", nCols=1,
                               format=headerFormat ) )
      for i in self.proxies:
         for ( label, value ) in (
            ( "Proxy name", _getStr( i.name ) ),
            ( "HTTP proxy", i.http ),
            ( "HTTPS proxy", i.https ),
            ):
            if value is not None:
               t.newRow( label, " : ", value )
      print t.output()


# Cli Model for "show cloud high-availability"
class CloudHa( Model ):
   # Configuration related attributes
   peerIp = Ip4Address( help="Peer IP address", optional=True )
   sourceInterface = Interface(
         help="Source interface for the peer BFD session", optional=True )
   enabled = Bool( help="Cloud HA enable" )
   failoverRecoveryTime = Int( help="Failover recovery wait time in seconds",
         optional=True )

   # status related attributes
   configValidationStatus = Enum( values=( 'valid', 'invalid', 'validating' ),
                  help="Cloud HA Config validation Status", optional=True )

   haState = Enum( values=( 'init', 'disabled', 'enabled', 'waiting', 'connected',
                           'ready', 'failover' ),
                     help="Cloud HA State", optional=True )
   lastFailoverTime = Float( help="Last failover time", optional=True )
   lastRecoveryTime = Float( help="Last recovery time", optional=True )
   lastValidationStartTime = Float( help="Last config validation start time",
      optional=True )
   lastValidationEndTime = Float( help="Last config validation end time",
      optional=True )
   failovers = Int( help="Failovers since enabled", optional=True )

   def render( self ):
      t = showCloudTableFormat()
      headerFormat = Format( justify="left", noBreak=True, terminateRow=True )
      headerFormat.noPadLeftIs( True )
      headerFormat.padLimitIs( True )
      headerFormat.noTrailingSpaceIs( True )
      t.newRow( FormattedCell( content="Cloud HA Configuration: ", nCols=1,
                               format=headerFormat ) )
      for ( label, value ) in (
         ( "Peer address", _getStr( self.peerIp ) ),
         # stringValue is not known to pylint as this is _TacAttributeType.
         # pylint: disable=E1101
         ( "Source interface", ( self.sourceInterface.stringValue \
            if self.sourceInterface else "" ) ),
         ( "Enabled", _getStr( self.enabled, 'false' ) ),
         ( "Failover recovery time", self.failoverRecoveryTime ),
         ( "Status", _getStr( self.configValidationStatus ) ),
         ( "State", _getStr( self.haState) ),
         ( "Last failover time", timestampToStr( self.lastFailoverTime ) ),
         ( "Last recovery time", timestampToStr( self.lastRecoveryTime ) ),
         ( "Last config validation start time",
            timestampToStr( self.lastValidationStartTime ) ),
         ( "Last config validation end time",
            timestampToStr( self.lastValidationEndTime ) ),
         ( "Failovers", self.failovers ),
         ):
         if True or value is not None:
            t.newRow( label, " : ", value )
      print t.output()


