#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliMode.CloudHaModes
import CliPlugin.CloudHaCliLib as CloudHaCliLib
import ConfigMount
import Tac
from Tracing import t0

awsCloudConfig = None
gcpCloudConfig = None

@CloudHaCliLib.providerProxyHandler
def updateAwsCloudProxy( proxyName ):
   t0( 'updateAwsCloudProxy: proxy', proxyName )
   if awsCloudConfig and awsCloudConfig.accessConfig.proxyName == proxyName:
      t0( "Changed aws proxy specific config, updating it" )
      awsCloudConfig.lastChangeTime = Tac.now()
      return True
   return False

#--------------------------------------------------------------------------------
# cloud high-availability
#--------------------------------------------------------------------------------
class CloudHighAvailabilityCmd( CliCommand.CliCommandClass ):
   syntax = 'cloud high-availability'
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'high-availability' : CloudHaCliLib.matcherHighAvailability,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliMode.CloudHaModes.ConfigCloudHaMode )
      mode.session_.gotoChildMode( childMode )

BasicCliModes.GlobalConfigMode.addCommandClass( CloudHighAvailabilityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] cloud provider aws
#--------------------------------------------------------------------------------
class CloudProviderAwsCmd( CliCommand.CliCommandClass ):
   syntax = 'cloud provider aws'
   noOrDefaultSyntax = syntax
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'provider' : CloudHaCliLib.matcherProvider,
      'aws' : CloudHaCliLib.nodeAws,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliMode.CloudHaModes.ConfigCloudAwsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "no cloud-aws mode" )
      defaultAwsConfig = Tac.newInstance( 'Cloud::HA::AwsAccessConfig' )
      t0( "Removed AWS specific config " )
      # Test case only w/o guard
      if not awsCloudConfig:
         return
      if awsCloudConfig.accessConfig != defaultAwsConfig:
         t0( "Changed AWS specific config,updating to default" )
         awsCloudConfig.accessConfig = defaultAwsConfig
         awsCloudConfig.lastChangeTime = Tac.now()

BasicCliModes.GlobalConfigMode.addCommandClass( CloudProviderAwsCmd )

#--------------------------------------------------------------------------------
# cloud provider azure
#--------------------------------------------------------------------------------
class CloudProviderAzureCmd( CliCommand.CliCommandClass ):
   syntax = 'cloud provider azure'
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'provider' : CloudHaCliLib.matcherProvider,
      'azure' : CloudHaCliLib.nodeAzure,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliMode.CloudHaModes.ConfigCloudAzureMode )
      mode.session_.gotoChildMode( childMode )

   # FIXME: The other two cloud providers have no/def form,
   # but this one does not. Is this right?

BasicCliModes.GlobalConfigMode.addCommandClass( CloudProviderAzureCmd )

#--------------------------------------------------------------------------------
# cloud provider gcp
#--------------------------------------------------------------------------------
class ConfigCloudGcpModeCmd( CliCommand.CliCommandClass ):
   syntax = 'cloud provider gcp'
   noOrDefaultSyntax = syntax
   data = {
         'cloud': CloudHaCliLib.nodeCloud,
         'provider': CloudHaCliLib.matcherProvider,
         'gcp': CloudHaCliLib.nodeGcp,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliMode.CloudHaModes.ConfigCloudGcpMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'no cloud-gcp mode' )
      defaultGcpConfig = Tac.newInstance( 'Cloud::HA::GcpAccessConfig' )
      t0( 'Removed GCP specific config' )
      if gcpCloudConfig.accessConfig != defaultGcpConfig:
         t0( 'Changed GCP specific config, updating to default' )
         gcpCloudConfig.accessConfig = defaultGcpConfig
         gcpCloudConfig.lastChangeTime = Tac.now()

BasicCliModes.GlobalConfigMode.addCommandClass( ConfigCloudGcpModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] cloud proxy PROXY
#--------------------------------------------------------------------------------
class CloudProxyProxynameCmd( CliCommand.CliCommandClass ):
   syntax = 'cloud proxy PROXY'
   noOrDefaultSyntax = syntax
   data = {
      'cloud' : CloudHaCliLib.nodeCloud,
      'proxy' : 'Cloud proxy configuration',
      'PROXY' : CloudHaCliLib.matcherProxyValue,
   }

   @staticmethod
   def handler( mode, args) :
      proxyName = args[ 'PROXY' ]
      t0( 'Creating cloud proxy mode for proxy', proxyName  )
      childMode = mode.childMode( CliMode.CloudHaModes.ConfigCloudProxyMode,
                                  proxyName=proxyName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      proxyName = args[ 'PROXY' ]
      if proxyName not in CloudHaCliLib.proxyConfig.proxy:
         return
      del CloudHaCliLib.proxyConfig.proxy[ proxyName ]
      # Check if the aws/azure/gcp config do use these.
      # if yes, notify the agent.
      if CloudHaCliLib.updateCloudHaProxy( proxyName ):
         t0( 'Proxy being referenced. CloudHa agent notified' )
      t0( "Removed proxy config" )

BasicCliModes.GlobalConfigMode.addCommandClass( CloudProxyProxynameCmd )

def Plugin( entityManager ):
   global awsCloudConfig
   global gcpCloudConfig

   # Mount unconditionally as CLI test calls with no flag set.
   awsCloudConfig = ConfigMount.mount( entityManager, "cloudha/awsconfig",
                                       "Cloud::HA::AwsHaConfig", "w" )

   # Mount unconditionally as CLI test calls with no flag set.
   gcpCloudConfig = ConfigMount.mount( entityManager, 'cloudha/gcpconfig',
                                       'Cloud::HA::GcpHaConfig', 'w' )

   CliMode.CloudHaModes.awsCloudConfig = awsCloudConfig
   CliMode.CloudHaModes.gcpCloudConfig = gcpCloudConfig
