#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''CLI commands for configuring Cloud Ha.'''

import CliCommand
import CliMatcher
import CliMode.CloudHaModes
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.CloudHaCliLib import passwordExpr, lettersThenAnything
import ConfigMount
import Tac
import Tracing
from CloudUtil import getHypervisor

t0 = Tracing.trace0

cloudType = getHypervisor()
cloudConfig = None

#-------------------------------------------------------------------------------
# config-cloud-proxy# [ no | default ] ( http | https ) IP PORT
#                     username USERNAME password PASSWORD
#-------------------------------------------------------------------------------
class ProxyCommand( CliCommand.CliCommandClass ):
   syntax = '( http | https ) IP PORT [ username USERNAME [ password PASSWORD ] ]'
   noOrDefaultSyntax = '( http | https ) ...'
   data = {
      'http' : 'HTTP proxy',
      'https' : 'HTTPS proxy',
      'IP' : IpAddrMatcher.ipAddrMatcher,
      'PORT' : CliMatcher.IntegerMatcher( 1, 65535,
                                          helpdesc='Proxy port' ),
      'username' : 'Proxy username',
      'USERNAME' : CliMatcher.PatternMatcher( pattern=lettersThenAnything,
                                              helpdesc="Proxy user name for "
                                                       "authentication",
                                              helpname='WORD'),
      'password' : 'Proxy password',
      'PASSWORD' : passwordExpr,
   }

   @staticmethod
   def handler( mode, args ):
      ipAddr = args.get( 'IP', '' )
      port = args.get( 'PORT', 0 )
      username = args.get( 'USERNAME', '' )
      password = args.get( 'PASSWORD', '' )
      t0( 'Got token', args, 'in mode', mode )

      proxy = Tac.nonConst( mode.proxy )
      if 'http' in args:
         proxy.httpProxy = ipAddr
         proxy.httpPort = port
         proxy.httpUsername = username
         proxy.httpPassword = password
      elif 'https' in args:
         proxy.httpsProxy = ipAddr
         proxy.httpsPort = port
         proxy.httpsUsername = username
         proxy.httpsPassword = password

      mode.proxy = proxy

   noOrDefaultHandler = handler

CliMode.CloudHaModes.ConfigCloudProxyMode.addCommandClass( ProxyCommand )

def Plugin( entityManager ):
   global cloudConfig

   if cloudType == 'AWS':
      cloudConfig = ConfigMount.mount( entityManager, "cloudha/awsconfig",
                                       "Cloud::HA::AwsHaConfig", "w" )
   elif cloudType == 'Azure':
      cloudConfig = ConfigMount.mount( entityManager, "cloudha/azureconfig",
                                       "Cloud::HA::AzureHaConfig", "w" )
   elif cloudType == 'GCP':
      cloudConfig = ConfigMount.mount( entityManager, 'cloudha/gcpconfig',
                                       'Cloud::HA::GcpHaConfig', 'w' )
   else:
      # Use GCP as default since EosImage Cli tests call this code with no flags set.
      cloudConfig = ConfigMount.mount( entityManager, 'cloudha/gcpconfig',
                                       'Cloud::HA::GcpHaConfig', 'w' )

   CliMode.CloudHaModes.cloudConfig = cloudConfig
