#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliMode.CloudHaModes as CloudHaModes
import CliPlugin.CloudHaCliLib as CloudHaCliLib
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.IpAddrMatcher import ipAddrMatcher
from CliPlugin.LoopbackIntfCli import LoopbackIntf
from CliPlugin.TunnelIntfCli import TunnelIntf
import Tac
from CloudUtil import defaultRecoveryWaitTime
from Tracing import t0

#--------------------------------------------------------------------------------
# aws
#--------------------------------------------------------------------------------
class AwsCmd( CliCommand.CliCommandClass ):
   syntax = 'aws'
   data = {
      'aws' : CliCommand.guardedKeyword( 'aws',
                  helpdesc='AWS specific config for Cloud HA',
                  guard=CloudHaCliLib.cloudAwsSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CloudHaModes.ConfigCloudHaAwsMode )
      mode.session_.gotoChildMode( childMode )

CloudHaModes.ConfigCloudHaPeerMode.addCommandClass( AwsCmd )

#--------------------------------------------------------------------------------
# azure
#--------------------------------------------------------------------------------
class AzureCmd( CliCommand.CliCommandClass ):
   syntax = 'azure'
   data = {
      'azure' : CliCommand.guardedKeyword( 'azure',
                    helpdesc='Azure specific config for Cloud HA',
                    guard=CloudHaCliLib.cloudAzureSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      t0( "Entering Cloud ha peer azure submode: " )
      childMode = mode.childMode( CloudHaModes.ConfigCloudHaAzureMode )
      mode.session_.gotoChildMode( childMode )

CloudHaModes.ConfigCloudHaPeerMode.addCommandClass( AzureCmd )

#--------------------------------------------------------------------------------
# gcp
#--------------------------------------------------------------------------------
class ConfigCloudHaGcpModeCmd( CliCommand.CliCommandClass ):
   syntax = 'gcp'
   data = {
         'gcp': CliCommand.guardedKeyword( 'gcp',
                   helpdesc='GCP specific config for Cloud HA',
                   guard=CloudHaCliLib.cloudGcpSupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      t0( "Entering Cloud ha GCP submode" )
      childMode = mode.childMode( CloudHaModes.ConfigCloudHaGcpMode )
      mode.session_.gotoChildMode( childMode )

CloudHaModes.ConfigCloudHaPeerMode.addCommandClass( ConfigCloudHaGcpModeCmd )


#--------------------------------------------------------------------------------
# [ no | defualt ] bfd source-interface ( ETH | LOOP | TUN ) [ single-hop ]
#--------------------------------------------------------------------------------
class BfdSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd source-interface ( ETH | LOOP | TUN ) [ single-hop ]'
   noOrDefaultSyntax = 'bfd ...'
   data = {
      'bfd' : 'Configure BFD parameters',
      'source-interface' : 'BFD source address',
      'ETH' : EthPhyIntf.ethMatcher,
      'LOOP' : LoopbackIntf.matcher,
      'TUN' : TunnelIntf.matcher,
      'single-hop' : 'Single hop BFD',
   }

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'ETH' ) or args.get( 'LOOP' ) or args.get( 'TUN' )
      singlehop = 'single-hop' in args
      t0( "Setting Cloud ha BFD  Peer ", intf, singlehop )

      bfdConfig = Tac.nonConst( mode.peer.bfdConfig)
      if intf:
         intf = Tac.Value( 'Arnet::IntfId', intf.name )
      bfdConfig.intf = intf
      bfdConfig.multihop = not singlehop
      mode.peer.bfdConfig = bfdConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( "Disabling Cloud ha Bfd Peer " )
      defaultBfdConfig = Tac.newInstance( 'Cloud::HA::BfdConfig' )
      mode.peer.bfdConfig = defaultBfdConfig

CloudHaModes.ConfigCloudHaPeerMode.addCommandClass( BfdSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] peer address ADDR
#--------------------------------------------------------------------------------
class PeerAddressIpCmd( CliCommand.CliCommandClass ):
   syntax = 'peer address ADDR'
   noOrDefaultSyntax = 'peer address ...'
   data = {
      'peer' : 'Peer',
      'address' : 'Peer address',
      'ADDR' : ipAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      ip = args[ 'ADDR' ]
      addr = Tac.Value( 'Arnet::IpGenAddr', ip )
      mode.peer.peerIp = addr

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.peer.peerIp = Tac.Value( 'Arnet::IpGenAddr', '' )

CloudHaModes.ConfigCloudHaPeerMode.addCommandClass( PeerAddressIpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] recovery wait-time SECONDS
#--------------------------------------------------------------------------------
class RecoveryWaitTimeDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'recovery wait-time SECONDS'
   noOrDefaultSyntax = 'recovery wait-time ...'
   data = {
      'recovery' : 'Recovery related config',
      'wait-time' : 'Delay before taking control of local traffic on recovery',
      'SECONDS' : CliMatcher.IntegerMatcher( 1, 1000,
                                             helpdesc='Specify delay in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      delay = args.get( 'SECONDS', defaultRecoveryWaitTime )
      t0( "Setting Cloud ha recovery wait time ", delay )
      mode.peer.recoveryWaitTime = delay

   noOrDefaultHandler = handler

CloudHaModes.ConfigCloudHaPeerMode.addCommandClass( RecoveryWaitTimeDelayCmd )
