#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliMode.CloudHaModes
import CliPlugin.CloudHaCliLib as CloudHaCliLib
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import ConfigMount
import Tac
from Tracing import t0

azureCloudConfig = None

@CloudHaCliLib.providerProxyHandler
def updateAzureCloudProxy( proxyName ):
   t0( 'updateAzureCloudProxy: proxy', proxyName )
   #pylint: disable=E1103
   if azureCloudConfig and azureCloudConfig.accessConfig.proxyName == proxyName:
      t0( "Changed azure proxy specific config, updating it" )
      azureCloudConfig.lastChangeTime = Tac.now()
      return True
   return False

#--------------------------------------------------------------------------------
# [ no | default ] ( primary-gateway | backup-gateway )
#                  TABLE ROUTE ADDR resource-group GROUP
#--------------------------------------------------------------------------------
class BackupGatewayRtidCmd( CliCommand.CliCommandClass ):
   syntax = ( '( primary-gateway | backup-gateway ) '
              'TABLE ROUTE ADDR resource-group GROUP' )
   noOrDefaultSyntax = '( primary-gateway | backup-gateway ) TABLE ROUTE ...'
   data = {
      'primary-gateway' : CloudHaCliLib.matcherPrimaryGateway,
      'backup-gateway' : CloudHaCliLib.matcherBackupGateway,
      'TABLE' : CliMatcher.PatternMatcher( pattern=CloudHaCliLib.lettersThenAnything,
                                           helpdesc='Azure route table ID',
                                           helpname='WORD' ),
      'ROUTE' : CloudHaCliLib.matcherRoute,
      'ADDR' : IpAddrMatcher.IpAddrMatcher( helpdesc='Next hop IP address' ),
      'resource-group' : 'Azure resource group name',
      'GROUP' : CliMatcher.PatternMatcher( pattern=CloudHaCliLib.lettersThenAnything,
                                           helpdesc='Azure resource group name',
                                           helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      rtId = args[ 'TABLE' ]
      ipAddrWithMask = args[ 'ROUTE' ]
      ipAddr = args[ 'ADDR' ]
      rg = args[ 'GROUP' ]
      primary = 'primary-gateway' in args

      if primary:
         routes = azureCloudConfig.localRoutes
      else:
         routes = azureCloudConfig.peerRoutes

      rtIdAndDestKey = Tac.newInstance( "Cloud::HA::RouteTableDestKey",
                                        rtId, ipAddrWithMask )
      rtIdAndDest = Tac.newInstance( "Cloud::HA::AzureRoute", rtIdAndDestKey )
      rtIdAndDest.nextHopRouteTarget = ipAddr
      rtIdAndDest.resourceGroup = rg
      # Make sure it is not there or update change flag.
      val = routes.get( rtIdAndDestKey )
      if not val or val.nextHopRouteTarget != ipAddr:
         t0( " Updating local =", primary, "RouteTable", ipAddr, val )
         mode.changed = True

      routes.addMember( rtIdAndDest )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rtId = args[ 'TABLE' ]
      ipAddrWithMask = args[ 'ROUTE' ]
      primary = 'primary-gateway' in args

      if primary:
         routes = azureCloudConfig.localRoutes
      else:
         routes = azureCloudConfig.peerRoutes

      t0( "Reseting local =", primary, "route for ",
          rtId, ipAddrWithMask.__class__ )
      rtIdAndDestKey = Tac.newInstance( "Cloud::HA::RouteTableDestKey",
            rtId, ipAddrWithMask )
      if routes.get( rtIdAndDestKey ):
         t0(" Removed entry from peerRouteTable" )
         mode.changed = True

      del routes[ rtIdAndDestKey ]

CliMode.CloudHaModes.ConfigCloudHaAzureMode.addCommandClass( BackupGatewayRtidCmd )

def Plugin( entityManager ):
   global azureCloudConfig

   # Mount unconditionally as CLI test calls with no flag set.
   azureCloudConfig = ConfigMount.mount( entityManager, "cloudha/azureconfig",
                                         "Cloud::HA::AzureHaConfig", "w" )

   CliMode.CloudHaModes.azureCloudConfig = azureCloudConfig
