#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''CLI commands for configuring Cloud Ha.'''

import CliCommand
import CliMatcher
import CliMode.CloudHaModes
import CliPlugin.CloudHaCliLib as CloudHaCliLib
import Tac
import Url
from Tracing import t0

@CloudHaCliLib.providerProxyHandler
def updateGcpCloudProxy( proxyName ):
   t0( 'updateGcpCloudProxy: proxy', proxyName )
   gcpCloudConfig = CliMode.CloudHaModes.gcpCloudConfig
   if gcpCloudConfig and gcpCloudConfig.accessConfig.proxyName == proxyName:
      t0( "Changed gcp proxy specific config, updating it" )
      gcpCloudConfig.lastChangeTime = Tac.now()
      return True
   return False

#-------------------------------------------------------------------------------
# config-cloud-gcp# [ no | default ] project PROJECT
#-------------------------------------------------------------------------------
class ProjectCmd( CliCommand.CliCommandClass ):
   syntax = 'project PROJECT'
   noOrDefaultSyntax = 'project ...'
   data = {
      'project': 'GCP project',
      'PROJECT': CliMatcher.PatternMatcher(
         pattern=r'[a-z][a-z0-9\-]{4,28}[a-z0-9]', helpdesc='GCP project name',
         helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      project = args.get( 'PROJECT' )
      t0( 'Got token project', project, 'in mode', mode )
      mode.gcpConfig.project = project

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'Got token no project in mode',  mode )
      mode.gcpConfig.project = ''

CliMode.CloudHaModes.ConfigCloudGcpMode.addCommandClass( ProjectCmd )

#-------------------------------------------------------------------------------
# config-cloud-gcp# [ no | default ] service-account file FILE
#-------------------------------------------------------------------------------
class ServiceAccountFileCmd( CliCommand.CliCommandClass ):
   syntax = 'service-account file FILE'
   noOrDefaultSyntax = 'service-account file ...'
   data = {
      'service-account': 'GCP service account configuration',
      'file': 'Service account file location',
      'FILE': Url.UrlMatcher( fsFunc=lambda fs: fs.fsType == 'flash',
                              helpdesc='Service account file URL' ),
   }

   @staticmethod
   def handler( mode, args ):
      fileLoc = args.get( 'FILE' )
      t0( 'Got token service-account file', fileLoc, 'in mode', mode )
      mode.gcpConfig.serviceFileLocation = fileLoc.url

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'Got token no service-account file in mode', mode )
      mode.gcpConfig.serviceFileLocation = ''

CliMode.CloudHaModes.ConfigCloudGcpMode.addCommandClass( ServiceAccountFileCmd )

#-------------------------------------------------------------------------------
# config-cloud-gcp# [ no | default ] proxy PROXY
#-------------------------------------------------------------------------------
class ProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'proxy PROXY'
   noOrDefaultSyntax = 'proxy ...'
   data = {
      'proxy': CloudHaCliLib.matcherProxy,
      'PROXY': CloudHaCliLib.matcherProxyValue,
   }

   @staticmethod
   def handler( mode, args ):
      CloudHaCliLib.setProxy( mode, mode.gcpConfig, args[ 'PROXY' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      CloudHaCliLib.noProxy( mode, mode.gcpConfig )

CliMode.CloudHaModes.ConfigCloudGcpMode.addCommandClass( ProxyCmd )
