# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliModel import (
      Model,
      Str,
      Int,
      Float,
      List,
      Dict,
      Bool )
import CliSchedulerLib
import errno
from FileCliUtil import sizeFmt
import os
from TableOutput import (
      createTable,
      Format )
import time
import Url
from urlparse import urlparse

class ScheduleJobModels( Model ):
   """
   Models for Schedule Summaries.
   """
   nextRun = Float( help="Next execution start time (in secs)." )
   cliCommand = Str( help="Scheduled Cli Command." )
   interval = Int( help="Cli command execution interval (in mins)." )
   logfileDirectory = Str( help="Logfile directory location." )
   maxLogFiles = Int( help="Maximum number of logfiles stored." )
   timeout = Float( help="Cli command execution timeout (in secs)." )
   verbose = Bool( help="Verbose logging state." )
   jobInProgress = Bool( help="Job in progress.", optional=True )
   _logfiles = List( valueType=str, help="List of logfile paths.", optional=True )

   jobInProgressStartTime = Float( help="Cli commmand in progress execution start "
                                   "time (in secs)", optional=True )
   lastRunStatus = Int( help="Cli command last execution status.",
                        optional=True )
   lastRun = Float( help="Time the job last executed (in secs).",
                    optional=True )
   _scheduledCliJobStatus = Bool( help="Job name is scheduled.", optional=True )

   def renderSummary( self, name, table ):
      lastTime = time.strftime( "%H:%M", time.localtime(
                 self.lastRun ) ) if self.lastRun else "-"
      if self.lastRun and self.maxLogFiles > 0:
         if self.logfileDirectory != CliSchedulerLib.logPrefixDefault:
            logfileLoc = Url.filenameToUrl( self.logfileDirectory )
         else:
            logfileLoc = self.logfileDirectory

         if not logfileLoc.endswith( "/" ):
            logfileLoc += "/"

         logfileLoc += ( "%s/" % name )
      else:
         logfileLoc = "-"
      interval = CliSchedulerLib.scheduleOnceStr
      if self.interval != CliSchedulerLib.scheduleOnce:
         interval = str( self.interval )
      status = "Waiting"
      if self.jobInProgress or self.lastRun:
         if self.jobInProgress:
            status = "Job under progress"
         elif self.lastRunStatus == 0:
            status = "Success"
         elif self.lastRunStatus == -1:
            status = "Timedout"
         else:
            status = "Fail"
      table.newRow( name, CliSchedulerLib.extractAtFromDateTime(
                    self.nextRun ), lastTime, interval,
                    int( self.timeout / 60 ), self.maxLogFiles,
                    logfileLoc, status )
      return table

   def renderName( self, name ):
      if self._scheduledCliJobStatus:
         if self.jobInProgress:
            print( "The CLI command execution is under progress for "
                   "last %d seconds" % ( time.time() -
                   self.jobInProgressStartTime ) )
         elif self.lastRun:
            if self.lastRunStatus == 0:
               print( "The last CLI command execution was successful" )
            elif self.lastRunStatus == -1:
               print( "The last CLI command execution timed out" )
            else:
               print( "The last CLI command failed with exit status %d" %
                      self.lastRunStatus )
            if self.interval != CliSchedulerLib.scheduleOnce:
               atTime = time.localtime( self.lastRun +
                                        self.interval*60 )
               print( "CLI command \"%s\" is scheduled next at \"%s\", "
                      "interval is %d minutes" % (
                      self.cliCommand, "%02d:%02d:%02d %02d/%02d/%04d"
                      % ( atTime.tm_hour, atTime.tm_min,
                          atTime.tm_sec, atTime.tm_mon, atTime.tm_mday,
                          atTime.tm_year ), self.interval ) )
            else:
               print( "CLI command \"%s\" was scheduled at \"%s\" for "
                      "single execution" % (
                      self.cliCommand,
                      CliSchedulerLib.extractAtFromDateTime( self.nextRun ) ) )
         else:
            if int( self.nextRun ) == CliSchedulerLib.scheduleNow:
               print( "CLI command \"%s\" is scheduled now, interval is %d "
                      "minutes" % ( self.cliCommand, self.interval ) )
            else:
               if self.interval != CliSchedulerLib.scheduleOnce:
                  print( "CLI command \"%s\" is scheduled at \"%s\", "
                         "interval is %d minutes" % (
                         self.cliCommand,
                         CliSchedulerLib.extractAtFromDateTime( self.nextRun ),
                         self.interval ) )
               else:
                  print( "CLI command \"%s\" is scheduled at \"%s\" for "
                         "single execution" % (
                         self.cliCommand,
                         CliSchedulerLib.extractAtFromDateTime( self.nextRun ) ) )

      else:
         print( "CLI command \"%s\" is currently not scheduled" % name )

      print( "Timeout is %u minutes" % int( self.timeout / 60 ) )

      print( "Maximum of %d log files will be stored" % self.maxLogFiles )

      print( 'Verbose logging is %s' % ( 'on' if self.verbose else 'off' ) )

      if self.logfileDirectory != CliSchedulerLib.logPrefixDefault:
         logLocation = Url.filenameToUrl( self.logfileDirectory )
      else:
         logLocation = self.logfileDirectory

      if not logLocation.endswith( "/" ):
         logLocation += "/"

      logLocation += ( "%s/" % name )
      if not self._logfiles:
         print( "No log files are stored in %s" % logLocation )
      else:
         print( "%d log files currently stored in %s\n" % ( len( self._logfiles ),
                logLocation ) )
         table = createTable( ( "Start time", "Size", "Filename" ) )
         leftFormat = Format( justify='left' )
         leftFormat.noPadLeftIs( True )
         centerFormat = Format( justify='center' )
         centerFormat.noPadLeftIs( True )
         table.formatColumns( leftFormat, centerFormat, leftFormat )
         for u in self._logfiles:
            filename = os.path.basename( u )
            tm = time.localtime( CliSchedulerLib.extractTsFromFilename(
                                 filename ) )
            # If a logFile <fname>.log extension is being compressed using
            # gzip when show CLI command is executed u.size() may fail with
            # OSError exception due to gzip renaming it to <fname>.log.gz
            try:
               url = urlparse( u )
               filePath = '/mnt/' + url.scheme + url.path
               size = os.path.getsize( filePath )
               table.newRow( time.strftime( "%b %d %Y %H:%M", tm ),
                             sizeFmt( size ), filename )
            except OSError, e:
               if e.errno != errno.ENOENT:
                  raise
         print( table.output().rstrip() )

class ShowScheduleJobModels( Model ):
   jobs = Dict( keyType=str, valueType=ScheduleJobModels,
                help="Maps job names to their respective details." )
   maxJobs = Int( help="Maximum number of jobs in progress at the same time.",
                  optional=True )
   prependHostname = Bool( help="Prepend host name to logfile.", optional=True)
   _showScheduleAction = Bool(
                           help="Defines which show schedule command was called." )

   def render( self ):

      if not self.jobs:
         print( "No CLI commands are currently scheduled" )
      else:
         # renders summary action if _showScheduleAction == True, else NAME action
         if self._showScheduleAction:
            table = createTable( ( "Name", "At time", "Last time", "Interval (mins)",
                                   "Timeout (mins)", "Max log files",
                                   "Logfile Location", "Status" ) )
            nameFormat = Format( justify='left', maxWidth=15, wrap=True )
            nameFormat.noPadLeftIs( True )
            nameFormat.noTrailingSpaceIs( True )
            leftFormat = Format( justify='left', maxWidth=8, wrap=True )
            leftFormat.noPadLeftIs( True )
            leftFormat.noTrailingSpaceIs( True )
            atTimeFormat = Format( justify='center', maxWidth=12, wrap=True )
            intervalFormat = Format( justify='center', maxWidth=8,
                                     wrap=True )
            timeoutFormat = Format( justify='center', maxWidth=8, wrap=True )
            lastTimeFormat = Format( justify='center', maxWidth=5, wrap=True )
            centerFormat = Format( justify='center' )
            centerFormat.noPadLeftIs( True )
            centerFormat.noTrailingSpaceIs( True )
            maxLogFileSizeFormat = Format( justify='center', maxWidth=5, wrap=True )
            loc = Format( justify='left', maxWidth=30, wrap=True )
            loc.noPadLeftIs( True )
            loc.noTrailingSpaceIs( True )
            table.formatColumns( nameFormat, atTimeFormat, lastTimeFormat,
                                 intervalFormat, timeoutFormat, maxLogFileSizeFormat,
                                 loc, leftFormat )
            # Jobs are sorted by the dictionary key, i.e. each individual job name
            for key in sorted( self.jobs.iterkeys() ):
               table = self.jobs[key].renderSummary( key, table )
            print( "Maximum concurrent jobs ", str( self.maxJobs ) )
            print( "Prepend host name to logfile: %s" % ( "Yes"
                   if self.prependHostname else "No" ) )
            print( table.output().rstrip() )
         else:
            self.jobs.values()[0].renderName( self.jobs.keys()[0] )

