# Copyright (c) 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import BasicCli
import CliCommand
import CliMatcher
import CliParserCommon
import CliSchedulerLib
from CliSchedulerLib import (
      showTechJobName,
      showTechCliCommand,
      showTechIntervalDefault,
      showTechMaxLogFilesDefault )
import ConfigMount
import DateTimeRule
import LazyMount
import ShowCommand
import Tac
import Url
import UrlPlugin.FlashUrl
from CliSchedulerCliModels import (
      ShowScheduleJobModels,
      ScheduleJobModels )

config = None
status = None

gconfigMode = BasicCli.GlobalConfigMode

scheduleKwMatcher = CliMatcher.KeywordMatcher(
   'schedule',
   helpdesc='Configure a CLI command to be run periodically' )

def allSchduledJobs( mode ):
   return { name : 'CLI command %s' % value.cliCommand
            for name, value in config.scheduledCli.iteritems() }

setSchedErrorMessage = {
      CliSchedulerLib.DATEFMT_ERROR[ "INVALID" ] : "Invalid at argument",
      CliSchedulerLib.DATEFMT_ERROR[ "PAST" ] : "Cannot schedule a command in past",
}

def setSchedConfig( mode, jobName, atDateTime, interval, timeout, maxLogFiles, 
                    loggingVerbose, logSavePath, command ):

   schedConfigType = Tac.Type( "System::CliScheduler::ScheduledCli" )

   dateTimeArr = CliSchedulerLib.createDateTimefromAt( atDateTime, interval )
   if dateTimeArr[ 0 ] < 0:
      mode.addError( setSchedErrorMessage[ dateTimeArr[ 0 ] ] )
      return

   if CliSchedulerLib.inPast( dateTimeArr[ 0 ] ):
      if interval == 0:
         mode.addError( setSchedErrorMessage[
               CliSchedulerLib.DATEFMT_ERROR[ "PAST" ] ] )
         return
      mode.addWarning( "Schedule a command starting in past" )

   verbose = True if loggingVerbose else False
   startAt = dateTimeArr[ 0 ]
   if len( dateTimeArr ) == 2:
      startAt = dateTimeArr[ 1 ]

   jobName = jobName.lower()
   if jobName == "summary":
      mode.addError( "Job name cannot match keyword %s " % jobName ) 
      return

   if logSavePath != CliSchedulerLib.logPrefixDefault:
      if isinstance( logSavePath, UrlPlugin.FlashUrl.FlashUrl ) and \
            logSavePath.localFilename() is not None and \
            ( not logSavePath.exists() or logSavePath.isdir() ):

         logSavePath = logSavePath.localFilename()
         if not logSavePath.endswith("/"):
            logSavePath += "/"

      elif isinstance( logSavePath, Url.Url ):
         mode.addError( "Invalid scheduled job output location" )
         return

      else:
         logSavePath = CliSchedulerLib.logPrefixDefault

   timeout = timeout if timeout else CliSchedulerLib.timeoutDefault
   # interval should be more than the timeout, otherwise the job (if it has a
   # long running time) will run continuously
   if interval > 0 and interval <= timeout:
      mode.addError( "For job %s, Interval %u should be greater than the timeout %u"\
                     % ( jobName, interval, timeout ) )
      return

   schedConfig = schedConfigType( jobName,
                                  command,
                                  dateTimeArr[ 0 ], startAt,
                                  interval, maxLogFiles,
                                  verbose,
                                  timeout * 60,
                                  logDir=logSavePath )

   config.scheduledCli.addMember( schedConfig )

def delSchedConfig( mode, jobName ):
   jobName = jobName.lower()
   del config.scheduledCli[ jobName ]

def setSchedConfigDefault( mode, jobName ):
   jobName = jobName.lower()
   if jobName != showTechJobName:
      delSchedConfig( mode, jobName )
      return

   logSavePath = CliSchedulerLib.logPrefixDefault
   schedConfigType = Tac.Type( "System::CliScheduler::ScheduledCli" )
   schedConfig = schedConfigType( showTechJobName,
                                  showTechCliCommand,
                                  CliSchedulerLib.scheduleNow,
                                  CliSchedulerLib.scheduleNow,
                                  showTechIntervalDefault,
                                  showTechMaxLogFilesDefault,
                                  False,
                                  CliSchedulerLib.timeoutDefault * 60,
                                  logDir=logSavePath )

   config.scheduledCli.addMember( schedConfig )

class ScheduleCommand( CliCommand.CliCommandClass ):
   syntax = "schedule NAME " \
            "( [ %s ] interval INT_MIN ) | " \
            "( at TIME [ DATE ] ( interval INT_MIN ) | %s ) " \
            "[ timeout TIMEOUT_MIN ] max-log-files MAXLOGS " \
            "[ logging verbose ] [ loglocation LOGPATH ] " \
            "command COMMAND" % ( CliSchedulerLib.scheduleNowStr,
                                  CliSchedulerLib.scheduleOnceStr )
   noOrDefaultSyntax = "schedule NAME ..."
   data = {
      'schedule' : scheduleKwMatcher,
      # config being a keyword should not be allowed as job name
      'NAME' : CliMatcher.PatternMatcher( r'^(?!config)[\w_-]+',
                                          helpname='WORD',
                                          helpdesc='Scheduled job name' ),
      CliSchedulerLib.scheduleNowStr :
      'Set the start time of the schedule to now',
      'interval' : 'Set interval for CLI command execution',
      'INT_MIN' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.intervalMin,
         CliSchedulerLib.intervalMax,
         helpdesc='Interval in minutes for CLI command execution' ),
      'at' : 'Set the start time of the schedule',
      'DATE' : DateTimeRule.dateExpression( 'DATE', helpdesc="Start date" ),
      'TIME' : DateTimeRule.ValidTimeMatcher( helpdesc="Start time" ),
      CliSchedulerLib.scheduleOnceStr :
      'Schedule the command for single execution',
      'timeout' : 'Set timeout for CLI command execution',
      'TIMEOUT_MIN' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.timeoutMin,
         CliSchedulerLib.timeoutMax,
         helpdesc='Timeout in minutes for CLI command execution' ),
      'max-log-files' : 'Set maximum number of logfiles to be stored',
      'MAXLOGS' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.maxLogFilesMin,
         CliSchedulerLib.maxLogFilesMax,
         helpdesc='Number of logfiles to be stored' ),
      'logging' : 'Logging level for scheduler',
      'verbose' : 'Detailed logging',
      'loglocation' : 'Choose which path scheduler log files can be saved to',
      'LOGPATH' : Url.UrlMatcher(
         lambda fs: fs.supportsWrite() and
         fs.fsType == 'flash' and not 'file:' in fs.scheme,
         'Destination for scheduled command output log files',
         notAllowed=[ 'all-filesystems', 'file' ] ),
      'command' : 'CLI command',
      'COMMAND' : CliMatcher.StringMatcher( helpname='WORD',
                                            helpdesc='Argument' )
      }
   @staticmethod
   def handler( mode, args ):
      jobName = args[ 'NAME' ]
      timeVal = args.get( 'TIME' )
      if timeVal is not None:
         dateVal = args.get( 'DATE', [] )
         atDateTime = list( timeVal ) + list( dateVal )
      else:
         atDateTime = CliSchedulerLib.scheduleNow

      if CliSchedulerLib.scheduleOnceStr in args:
         interval = CliSchedulerLib.scheduleOnce
      else:
         interval = args.get( 'INT_MIN', 0 )

      timeout = args.get( 'TIMEOUT_MIN', 0 )
      maxLogFiles = args[ 'MAXLOGS' ]
      loggingVerbose = 'verbose' in args
      logSavePath = args.get( 'LOGPATH' )
      command = args[ 'COMMAND' ]
      setSchedConfig( mode, jobName, atDateTime, interval, timeout, maxLogFiles,
                      loggingVerbose, logSavePath, command )

   @staticmethod
   def noHandler( mode, args ):
      delSchedConfig( mode, args[ 'NAME' ] )

   @staticmethod
   def defaultHandler( mode, args ):
      setSchedConfigDefault( mode, args[ 'NAME' ] )

gconfigMode.addCommandClass( ScheduleCommand )

configKwMatcher = CliMatcher.KeywordMatcher(
   'config',
   helpdesc='Set CLI scheduler configuration parameters' )

class ScheduleMaxJobsCommand( CliCommand.CliCommandClass ):
   syntax = "schedule config max-concurrent-jobs JOBS"
   noOrDefaultSyntax = "schedule config max-concurrent-jobs ..."
   data = {
      'schedule' : scheduleKwMatcher,
      'config' : configKwMatcher,
      'max-concurrent-jobs' : 'Set limit for maximum number of concurrent jobs',
      'JOBS' : CliMatcher.IntegerMatcher(
         CliSchedulerLib.jobsInProgressMin,
         CliSchedulerLib.jobsInProgressMax,
         helpdesc='Maximum number of concurrent jobs' )
      }
   @staticmethod
   def handler( mode, args ):
      config.jobsInProgress = args[ 'JOBS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.jobsInProgress = config.jobsInProgressDefault

gconfigMode.addCommandClass( ScheduleMaxJobsCommand )

class ScheduleHostnameCommand( CliCommand.CliCommandClass ):
   syntax = "schedule config prepend-hostname-logfile"
   noSyntax = syntax
   defaultSyntax = syntax
   data = {
      'schedule' : scheduleKwMatcher,
      'config' : configKwMatcher,
      'prepend-hostname-logfile' : 'Prepend hostname to logfile name',
      }
   @staticmethod
   def handler( mode, args ):
      config.prependHostname = config.prependHostnameDefault

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      config.prependHostname = False

gconfigMode.addCommandClass( ScheduleHostnameCommand )

#pylint: disable=protected-access
def showSchedulePopulateModel( key, jobType, mode ):
   job = ScheduleJobModels()
   cliConfig = config.scheduledCli[ key ]
   lastTime = None
   if key in status.scheduledCliJobStatus:
      job._scheduledCliJobStatus = True
      cliStatus = status.scheduledCliJobStatus[ key ]
      job.jobInProgress = cliStatus.jobInProgress
      # jobType is a bool which differentiates between 'summary'|'NAME' cli commands
      # for this specific case
      if jobType:
         if cliStatus.jobInProgress or cliStatus.lastExecutionTime:
            lastTime = cliStatus.lastExecutionTime
            job.lastRunStatus = cliStatus.lastExecutionStatus
      else:
         if cliStatus.jobInProgress:
            job.jobInProgressStartTime = cliStatus.lastExecutionStartTime
         elif cliStatus.lastExecutionTime:
            lastTime = cliStatus.lastExecutionTime
            job.lastRunStatus = cliStatus.lastExecutionStatus
   else:
      job._scheduledCliJobStatus = False

   if cliConfig.logDir != CliSchedulerLib.logPrefixDefault:
      logLocation = Url.filenameToUrl( cliConfig.logDir )
   else:
      logLocation = cliConfig.logDir

   if not logLocation.endswith( "/" ):
      logLocation += "/"

   logLocation += ( "%s/" % cliConfig.name )
   context = Url.Context( mode.entityManager, mode.session_.disableAaa_ )
   schedUrl = Url.parseUrl( logLocation, context )
   logfiles = []
   try:
      logfiles = [ str( schedUrl.child( f ) ) for f in sorted(
                   schedUrl.listdir(), reverse=True,
                   key=CliSchedulerLib.extractTsFromFilename )
                   if CliSchedulerLib.extractTsFromFilename( f ) ]
   except OSError, e:
      if e.errno != errno.ENOENT:
         raise

   if not logfiles:
      job._logfiles = None
   else:
      job._logfiles = logfiles

   job.verbose = cliConfig.verbose
   job.nextRun = cliConfig.at
   job.lastRun = lastTime
   job.interval = cliConfig.interval
   job.timeout = cliConfig.timeout
   job.maxLogFiles = cliConfig.maxLogFiles
   job.logfileDirectory = cliConfig.logDir
   job.cliCommand = cliConfig.cliCommand
   return job

def showSchedule( mode, args ):
   model = ShowScheduleJobModels()
   jobName = args.get( 'NAME', 'summary' ).lower()
   if not config.scheduledCli:
      return model
   else:
      if jobName == "summary":
         model._showScheduleAction = True
         for key in config.scheduledCli:
            model.jobs[ key ] = showSchedulePopulateModel( key, True, mode )
         model.maxJobs = config.jobsInProgress
         model.prependHostname = config.prependHostname
      else:
         model._showScheduleAction = False
         if jobName not in config.scheduledCli:
            mode.addError( 'Please enter job name of a scheduled CLI command' )
            return model
         model.jobs[ jobName ] = showSchedulePopulateModel( jobName, False, mode )
         model.maxJobs = config.jobsInProgress
         model.prependHostname = config.prependHostname
      return model

class ShowScheduleCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show schedule NAME | summary"
   data = {
      "schedule" : "Show CLI Scheduler information",
      "NAME" : CliMatcher.DynamicNameMatcher( allSchduledJobs,
                                              "Scheduled job name",
                                              priority=CliParserCommon.PRIO_LOW ),
      "summary" : "Summarized output"
      }
   handler = showSchedule
   cliModel = ShowScheduleJobModels
   privileged = True

BasicCli.addShowCommandClass( ShowScheduleCommand )

# --------------------------------------------------------------------------
def Plugin( entityManager ):
   global config
   global status
   config = ConfigMount.mount( entityManager, "sys/clischeduler/config",
                               "System::CliScheduler::Config", "w" )
   status = LazyMount.mount( entityManager, "sys/clischeduler/status",
                             "System::CliScheduler::Status", "r" )
