#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import BasicCli
import CliCommand
import CliMatcher
import HostnameCli
import LazyMount
import MacAddr
import Plugins
import ShowCommand
import CliPlugin.ControllerCli as ControllerCli
from CliPlugin.ControllerCli import serviceAfterShowKwMatcher
from CliPlugin.ControllerdbLib import registerNotifiee, controllerGuard
from CliRelayModels import Switch
from CliRelayModels import Switches
from CliRelayModels import PythonExpressionWithException
from CliRelayModels import PythonExpressionWithExceptions
from CliRelayModels import IdsToCmds
from CliRelayModels import Result
from CliRelayModels import Command
from CliRelayModels import Summary
from CliRelayModels import SwitchCommand
from CliRelayModels import SwitchCommands
from Ethernet import convertMacAddrToCanonical
import cPickle

# Unused import just to build the dependency on Topology package
# as topology controllerdb mount, 'topology/version3/global/status' is used
# in this plugin.
import CliPlugin.NetworkTopology # pylint: disable-msg=W0611

toSwitch = None
fromSwitch = None
topologyStatus = None
serviceAgentStatus = None

def parseStuffForDebuggingOnly( stuffForDebuggingOnly ):
   return cPickle.loads( stuffForDebuggingOnly ) if \
         len( cPickle.loads( stuffForDebuggingOnly ) ) > 1 else \
         ( cPickle.loads( stuffForDebuggingOnly )[ 0 ], None )

def topologyRegisteredHosts():
   if topologyStatus:
      return [ host.replace( ':', '-') for host in topologyStatus.host.keys() ]
   else:
      return []

def topologyRegisteredHostnames():
   return topologyStatus.hostsByHostname.keys() if topologyStatus else None

#Returns list of all mac addresses mapped to hostname
def getTopologyRegisteredHost( hostname ):
   if topologyStatus and hostname in topologyStatus.hostsByHostname:
      hosts = topologyStatus.hostsByHostname[ hostname ].host.keys()
      return [ host.replace( ':', '-' ) for host in hosts ]
   else:
      return []

#Returns hostname mapped to mac address
def getTopologyRegisteredHostname( host ):
   macAddr = host.replace( '-', ':' )
   if topologyStatus and topologyStatus.host.get( macAddr ):
      return topologyStatus.host[ macAddr ].hostname
   else:
      return None

cliRelayKwMatcher = CliMatcher.KeywordMatcher( 'cli-relay',
                                             helpdesc='configure Cli Relay Service' )

#----------------------------------
# show service cli-relay summary
#---------------------------------
def requestResponseSummary( mode, args ):
   numCommands = 0
   numResults = 0
   numValid = 0
   numExc = 0
   switches = topologyRegisteredHosts()
   if fromSwitch and toSwitch:
      for switchName in switches:
         switchDir = toSwitch.get( switchName )
         if switchDir:
            for app in switchDir:
               appDir = switchDir[ app ]
               numCommands += len( appDir )
      for switchName in fromSwitch: 
         switchDir = fromSwitch.get( switchName )
         for app in switchDir: 
            appDir = switchDir[ app ] 
            numResults += len( appDir ) 
            for cmdId in appDir: 
               cmdResult = appDir[ cmdId ] 
               if ( len( cPickle.loads( cmdResult.stuffForDebuggingOnly ) ) > 1 ):
                  numExc += 1
               else: 
                  numValid += 1
   return Summary( apps=serviceAgentStatus.numClient,
                   commands=numCommands, resultsAll=numResults,
                   resultsValid=numValid, resultsExc=numExc )

class ShowCliRelaySummary( ShowCommand.ShowCliCommandClass ):
   syntax = "show service cli-relay summary"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'cli-relay' : cliRelayKwMatcher,
      'summary' : 'Overall summary information'
      }
   cliModel = Summary
   handler = requestResponseSummary

BasicCli.addShowCommandClass( ShowCliRelaySummary )

#----------------------------------------------------------------------
# show service cli-relay switch { detail, H.H.H, WORD, >, >>, |, <cr> }
#----------------------------------------------------------------------
def requestResponseForSwitch( mode, macAddr, detail=True ): 
   numCmdRequests = 0
   numCmdResponse = 0
   latestTime = None
   latestCmd = None
   exceptions = []
   switchAgentRunning = False

   canonicalMac = convertMacAddrToCanonical( macAddr ).replace( ':', '-' )
   if fromSwitch and ( canonicalMac in fromSwitch and \
         len( fromSwitch[ canonicalMac ].showCommandResultDir.keys() ) > 0):
      switchAgentRunning = True

   if not fromSwitch or not toSwitch or not switchAgentRunning:
      return Switch( switchMac=canonicalMac.replace( '-', ':' ),
                     hostname = getTopologyRegisteredHostname( canonicalMac ), 
                     switchAgentRunning = switchAgentRunning,  
                     commandRequests = numCmdRequests,
                     comandResponses = numCmdResponse,
                     detail = detail )

   for app in toSwitch[ canonicalMac ]: 
      numCmdRequests += len( toSwitch[ canonicalMac ][ app ] )
   latestTime = 0
   latestCmd = None
   if canonicalMac in fromSwitch: 
      for app in fromSwitch[ canonicalMac ]: 
         numCmdResponse += len( fromSwitch[ canonicalMac ][ app ] )
         for cmdId in fromSwitch[ canonicalMac ][ app ]: 
            request = toSwitch[ canonicalMac ][ app ][ cmdId ]
            result = fromSwitch[ canonicalMac ][ app ][ cmdId ]
            ( timeStamp, exc ) = parseStuffForDebuggingOnly( 
                  result.stuffForDebuggingOnly )
            if ( float( timeStamp ) >= latestTime ): 
               latestTime  = float( timeStamp )
               latestCmd = request.showCommand
            if ( exc ):
               pythonExpr = None if request.pythonExpression == '' \
                     else request.pythonExpression
               exceptions += [ PythonExpressionWithException( 
                                    command=request.showCommand, 
                                    pythonExpression=pythonExpr,
                                    exception = cPickle.dumps( exc ) ) ]

   return Switch( switchMac = canonicalMac.replace( '-', ':' ), 
                  hostname = getTopologyRegisteredHostname( canonicalMac ), 
                  switchAgentRunning = switchAgentRunning ,  
                  commandRequests = numCmdRequests,
                  comandResponses = numCmdResponse,
                  latestRunTimeStamp = latestTime, 
                  latestRunCommand = latestCmd, 
                  pythonExpressionWithExceptions =
                  PythonExpressionWithExceptions( exceptions = exceptions ), 
                  detail = detail )

def detailedSwitchResponses( mode ): 
   switches = topologyRegisteredHosts()
   switchCliModels = {}
   for switchName in switches: 
      switchModel = requestResponseForSwitch( mode, switchName )
      switchCliModels[ switchName ] = switchModel
   return Switches( switches=switchCliModels, detail=True )

def briefSwitchResponses( mode ):
   switchCliModels = {}
   if toSwitch:
      for macaddr in toSwitch:
         switchModel = requestResponseForSwitch( mode, macaddr, detail=False )
         switchCliModels[ macaddr ] = switchModel
   return Switches( switches=switchCliModels, detail=False )

switchKwMatcher = CliMatcher.KeywordMatcher( 'switch',
                                             helpdesc='Show information per switch' )

class ShowCliRelaySwitch( ShowCommand.ShowCliCommandClass ):
   syntax = "show service cli-relay switch [ detail ]"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'cli-relay' : cliRelayKwMatcher,
      'switch' : switchKwMatcher,
      'detail' : 'More comprehensive output'
      }
   cliModel = Switches

   @staticmethod
   def handler( mode, args ):
      if 'detail' in args:
         return detailedSwitchResponses( mode )
      else:
         return briefSwitchResponses( mode )

BasicCli.addShowCommandClass( ShowCliRelaySwitch )

#---------------------------------------
# show service cli-relay switch mac <H.H.H>
#---------------------------------------

#Format MAC address from H:H:H:H to H.H.H  and return a dict[ H.H.H ] = H-H-H-H
#for dynamic keyword matchiing

def registeredHostsDisplayFormatMap(): 
   formattedMacAddrs = {}
   if topologyStatus:
      for macAddr in topologyStatus.host.keys(): 
         macAddrHexDigits = macAddr.split(':')
         macAddrFormatted = ('').join( macAddrHexDigits[0:2] ) + '.' + \
                            ('').join( macAddrHexDigits[2:4] ) + '.' + \
                            ('').join( macAddrHexDigits[4:] )

         formattedMacAddrs[ macAddrFormatted ] = macAddr.replace( ':', '-')
   return formattedMacAddrs

macKwMatcher = CliMatcher.KeywordMatcher( 'mac', helpdesc='Display for mac address' )

class ShowCliRelayMac( ShowCommand.ShowCliCommandClass ):
   syntax = "show service cli-relay switch mac MAC"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'cli-relay' : cliRelayKwMatcher,
      'switch' : switchKwMatcher,
      'mac' : macKwMatcher,
      'MAC' : MacAddr.macAddrMatcher
      }
   cliModel = Switch

   @staticmethod
   def handler( mode, args ):
      return requestResponseForSwitch( mode, args[ 'MAC' ] )

BasicCli.addShowCommandClass( ShowCliRelayMac )

#------------------------------------------
# show service cli-relay switch hostname <hostname>
#------------------------------------------
class ShowCliRelayHostname( ShowCommand.ShowCliCommandClass ):
   syntax = "show service cli-relay switch hostname HOSTNAME"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'cli-relay' : cliRelayKwMatcher,
      'switch' : switchKwMatcher,
      'hostname' : CliCommand.guardedKeyword( 'hostname',
                                              'Hostname of switch',
                                              controllerGuard ),
      'HOSTNAME' : HostnameCli.HostnameMatcher( helpname="WORD",
                                                helpdesc="switch hostname" )
      }
   cliModel = Switches

   @staticmethod
   def handler( mode, args ):
      hostname = args[ 'HOSTNAME' ]
      switchCliModels = {}
      for switchName in getTopologyRegisteredHost( hostname ):
         switchCliModel = requestResponseForSwitch( mode, switchName )
         switchCliModels[ switchName ] = switchCliModel
      return Switches( switches=switchCliModels, detail=True )

BasicCli.addShowCommandClass( ShowCliRelayHostname )

#-----------------------------------------------------------------------------------
# show service cli-relay command [ switch mac H.H.H [ APP ] ]
#-----------------------------------------------------------------------------------
def dumpShowCommandRequests( mode, switchName=None, appName=None ): 
   if not toSwitch:
      return SwitchCommands()
   apps = []
   if ( switchName ): 
      switchName = convertMacAddrToCanonical( switchName ).replace( ':', '-' )
      switches = [ switchName ]
   else: 
      switches = toSwitch.keys()

   switchToCommands = {}
   for switchMac in switches:
      #Throw error if provided appName is not registered for switch 
      if appName: 
         if ( appName not in toSwitch[ switchName ] ): 
            mode.addError( 'Unrecognized app for switch %s' % switchName )
            return
         apps = [ appName ]
      else: 
         apps = toSwitch.get( switchMac, default=[] )

      appToCmds = {}
      for app in apps:
         idsToCmds = {}
         for cmdId in toSwitch[ switchMac ][ app ]: 
            cmdRequest = toSwitch[ switchMac ][ app ][ cmdId ] 
            
            pythonExpr = None if cmdRequest.pythonExpression == '' \
                  else cmdRequest.pythonExpression
            
            #Check if acmdId does not exist in fromSwitch, in the case that 
            #the switch is disconnected. Return only command request information
            if not fromSwitch or not fromSwitch.get( switchMac ) or \
                   ( app not in fromSwitch[ switchMac ] ) or \
                  ( cmdId not in fromSwitch[ switchMac ][ app ] ): 
               command = Command( appname=app, cmdId=cmdId,
                                  command = cmdRequest.showCommand, 
                                  revision = cmdRequest.revision, 
                                  pythonExpr = pythonExpr, 
                                  forSwitch= ( switchName != None ) )
            else:  
               cmdResult = fromSwitch[ switchMac ][ app ][ cmdId ]
               ( timeStamp, exception ) = parseStuffForDebuggingOnly( 
                     cmdResult.stuffForDebuggingOnly )
               if exception:
                  exception = cPickle.dumps( exception )

               result = Result( pythonExpressionResult = 
                                 cmdResult.pythonExpressionResult.rstrip( '\n' ), 
                                 exception = exception, 
                                 lastRun = timeStamp )
               
               command = Command( appname=app, cmdId=cmdId,
                                  command = cmdRequest.showCommand, 
                                  revision = cmdRequest.revision, 
                                  pythonExpr = pythonExpr, 
                                  result = result, 
                                  forSwitch= ( switchName != None ) )
            
            idsToCmds[ str( cmdId ) ] = command
         if ( idsToCmds ): 
            appToCmds[ app ] = IdsToCmds( idsToCmds=idsToCmds )
      switchToCommands[ switchMac ] = SwitchCommand( appToCmds=appToCmds, 
                                       switchName=convertMacAddrToCanonical( \
            switchMac ).replace( '-', ':' ) )

   return SwitchCommands( switchToCommands=switchToCommands )


class ShowCliRelayCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show service cli-relay command [ switch mac MAC [ APP ] ]"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'cli-relay' : cliRelayKwMatcher,
      'command' : 'Commands configured to run on connected switches',
      'switch': switchKwMatcher,
      'mac' : macKwMatcher,
      'MAC' : MacAddr.macAddrMatcher,
      'APP' : CliMatcher.PatternMatcher( r'\w+', helpname='WORD',
                                         helpdesc='App name' )
      }
   cliModel = SwitchCommands

   @staticmethod
   def handler( mode, args ):
      switchName = args.get( 'MAC' )
      appName = args.get( 'APP' )
      return dumpShowCommandRequests( mode, switchName, appName )

BasicCli.addShowCommandClass( ShowCliRelayCommand )

def handleCvxShutdown( mode, no=False ):
   enabled = bool( no )
   global toSwitch
   global fromSwitch
   global topologyStatus
   if not enabled:
      toSwitch = None
      fromSwitch = None
      topologyStatus = None

ControllerCli.addCvxShutdownCallback( handleCvxShutdown )
ControllerCli.addNoCvxCallback( handleCvxShutdown )

def doControllerdbMounts( controllerdbEntMan ):
   global toSwitch, fromSwitch, topologyStatus
   toSwitch = LazyMount.mount( controllerdbEntMan,
               'cliRelay/version1/toSwitch', 'Tac::Dir', 'ri' )
 
   fromSwitch = LazyMount.mount( controllerdbEntMan,
               'cliRelay/version1/fromSwitch', 'Tac::Dir', 'ri' )

   topologyStatus = LazyMount.mount( controllerdbEntMan,
         'topology/version3/global/status',
         'NetworkTopologyAggregatorV3::Status', 'ri' )


@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global serviceAgentStatus
   registerNotifiee( doControllerdbMounts )
   serviceAgentStatus = LazyMount.mount( entityManager,
                                         'cliRelay/serviceAgentStatus',
                                         'CliRelay::CliRelayCvxAgentStatus', 'r' )

