#!/usr/bin/env pythtab = ' ' * numSpaceson
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Bool
from CliModel import Int
from CliModel import Float
from CliModel import Submodel

#Used to format show command results exceptions such that each line
#in the exception is tabbed to the correct level
def formatLines( out, numSpaces ):
   lineArr = out.split( '\n' )
   formattedOut = ''
   tab = ' ' * numSpaces
   for ( i, line ) in enumerate( lineArr ):
      if ( i == len( lineArr ) - 1 ):
         formattedOut += ( tab + line )
      else:
         formattedOut += ( tab + line + '\n' )

   return formattedOut

class PythonExpressionWithException( Model ):
   command = Str( help = 'Show command for which exception occured' )
   pythonExpression = Str( help = 'Python expression for which exception occured',
                           optional=True )

   exception = Str( help = 'Exception for show command/python expression' )


class PythonExpressionWithExceptions( Model ):
   exceptions = List( valueType=PythonExpressionWithException,
               help= 'List of python expresion exception details' )


class Switch( Model ):
   switchMac = Str( help = 'Mac address of switch')
   hostname = Str( help = 'Hostname of switch', optional=True )
   switchAgentRunning = Bool( help = 'Check if switch agent is responding' )
   commandRequests = Int( help = 'Number of command requests' )
   comandResponses = Int( help = 'Number of command responses' )
   latestRunTimeStamp = Float( help = 'Timestamp of latest received update',
                               optional=True)

   latestRunCommand = Str( help = 'Command with latest response', optional=True )
   pythonExpressionWithExceptions = \
                           Submodel( valueType=PythonExpressionWithExceptions,
                           help= 'Python expression evaluation exception',
                           optional=True )

   detail = Bool( help = 'Print switch detail' )

   def render( self ):
      tab = ' ' * 2
      print '\nSwitch %s' % self.switchMac
      print tab + 'Hostname: %s' % self.hostname
      print tab + 'Switch Agent Running: %s' % self.switchAgentRunning
      print tab + 'Command Requests: %d' % self.commandRequests
      print tab + 'Command Responses: %d' % self.comandResponses
      if ( self.latestRunTimeStamp and self.latestRunCommand ):
         print tab + 'Latest Run Timestamp: %f' % self.latestRunTimeStamp
         print tab + 'Latest Run Command: %s' % self.latestRunCommand
         if ( self.detail ):
            print tab + 'Python Expression with Exception:'
            for result  in self.pythonExpressionWithExceptions.exceptions:
               tab = ' ' * 4
               print tab + '%s:' % result.command
               tab = ' ' * 6
               print tab + '%s' % result.pythonExpression
               print tab + 'Exception: '
               print formatLines( result.exception, 8 )

class Switches( Model ):
   switches = Dict( valueType=Switch, help='Maps MAC to corresponding switches' )
   detail = Bool( help = 'Print switch detail' )

   def render( self ):
      for macAddr in sorted(self.switches.keys() ):
         self.switches[ macAddr ].render()

class Result( Model ):
   pythonExpressionResult = Str( help = 'Python expression evaluation result',
                                 optional=True )

   exception = Str( help = 'Exception for show command/python expression',
                     optional=True )

   lastRun = Float( help = 'Timestamp of latest show command result', optional=True )

   def render( self ):
      pass

class Command( Model ):
   appname = Str( help = 'App client requesting command' )
   cmdId = Int( help = 'Id of show command request' )
   command = Str( help = 'Show command requested' )
   revision = Int( help = 'Capi-revision for command' )
   pythonExpr = Str( help = 'Python expression applied to command output',
                     optional=True )

   result = Submodel( valueType=Result, help = 'Show command result', optional=True )
   forSwitch = Bool( help = 'For specific switch?' )

   def renderAll( self ):
      tab = ' ' * 2
      print tab + 'Command %s/%d' % ( self.appname, self.cmdId )
      tab = ' ' * 4
      print tab + 'Command: %s' % self.command
      print tab + 'Revision: %d' % self.revision
      print tab + 'Expression: %s' % self.pythonExpr
      if ( self.result ):
         print tab + 'Result: %s' % self.result.pythonExpressionResult
      else:
         print tab + 'Result: None'

   def renderForSwitch( self ):
      tab = ' ' * 2
      print tab + 'Command %s/%d' % ( self.appname, self.cmdId )
      tab = ' ' * 4
      print tab + 'Command: %s' % self.command
      print tab + 'Expression: %s' % self.pythonExpr
      print tab + 'Result: '
      tab = ' ' * 6
      if ( self.result ):
         print tab + 'Expression Result: %s' % self.result.pythonExpressionResult
         if ( self.result.exception ):
            print tab + 'Exception: '
            print formatLines( self.result.exception, 8 )
         print tab + 'Last Run: %s' % self.result.lastRun
      else:
         print tab + 'None'

   def render( self ):
      if ( self.forSwitch ):
         self.renderForSwitch()
      else:
         self.renderAll()

#IdsToCmds.idsToCmds = { cmdId : show command }
class IdsToCmds( Model ):
   idsToCmds = Dict( valueType=Command, help= 'Map from id to commands' )

   def render( self ):
      pass

#SwitchCommand.appsToCmds = { appName : IdsToCmds }
class SwitchCommand( Model ):
   switchName = Str( help= 'Mac address of switch' )
   appToCmds = Dict( valueType=IdsToCmds, help='Map from app to command dict' )

   def render( self ):
      if ( self.appToCmds ):
         print '\nSwitch: %s' % self.switchName
         for app in sorted( self.appToCmds.keys() ):
            for cmdId in sorted( self.appToCmds[ app ].idsToCmds.keys() ):
               command = self.appToCmds[ app].idsToCmds[ cmdId ]
               command.render()

#SwitchCommands.switchCommands= { macAddr : SwitchCommand }
class SwitchCommands( Model ):
   switchToCommands = Dict( valueType=SwitchCommand,
                            help='Map from switch to show commands' )

   def render( self ):
      for switch in sorted( self.switchToCommands.keys() ):
         switchCommand = self.switchToCommands[ switch ]
         switchCommand.render()

class Summary( Model ):
   apps = Int( help= 'Total number of apps' )
   commands = Int( help = 'Total number of commands' )
   resultsAll = Int( help= 'Total number of results' )
   resultsValid = Int( help= 'Number of valid resutls' )
   resultsExc = Int( help = 'Number of results with exceptions' )

   def render( self ):
      print 'Apps: %d' % self.apps
      print 'Commands: %d' % self.commands
      print 'Results (all): %d' % self.resultsAll
      print 'Results (valid): %d' % self.resultsValid
      print 'Results (exc): %d' % self.resultsExc
