#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys

import CliParser
import CliMatcher
import CliCommand
import BasicCli
import BasicCliUtil
import ConfigMount
import LazyMount
from CliMode.CliRelay import CliRelayConfigModeBase
from CliMode.CliRelay import ShowCommandConfigModeBase
import CliPlugin.ControllerCli as ControllerCli
import CliPlugin.CliRelayShow as CliRelayShow
from CliPlugin.ControllerCli import CvxConfigMode
from CliPlugin.ControllerCli import addNoCvxCallback
import Ethernet
import Tac

showCommandConfig = None
cliRelayClients = None
cliRelayClientName = 'static'

def getSwitchMacEntry( switchMac ):
   switchMac = switchMac.replace( '.', '' )
   switchMac = '-'.join( [ switchMac[i:i+2]
      for i in range( 0, len(switchMac), 2 ) ] )
   return switchMac

#--------------------
#  service cli-relay
#--------------------

class CliRelayConfigMode( CliRelayConfigModeBase, BasicCli.ConfigModeBase ):
   name = 'CliRelay configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliRelayConfigModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class EnterCliRelayMode( CliCommand.CliCommandClass ):
   syntax = """service cli-relay"""
   data = { "service" : ControllerCli.serviceKwMatcher,
            "cli-relay" : CliRelayShow.cliRelayKwMatcher }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CliRelayConfigMode )
      mode.session_.gotoChildMode( childMode )

CvxConfigMode.addCommandClass( EnterCliRelayMode )

#---------------------------------------
#  switch mac <switch mac addr> <cmd id>
#---------------------------------------

class ShowCommandConfigMode( ShowCommandConfigModeBase, BasicCli.ConfigModeBase ):

   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, switchMac, hostName, cmdId ):
      ShowCommandConfigModeBase.__init__( self, ( switchMac, hostName, cmdId ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.showCommandKey = Tac.Value( "CliRelay::ShowCommandKey",
            switchMac=switchMac, hostName=hostName, cmdId=cmdId )
      self.cmd = None
      self.rev = 1
      self.responseFormat = 'json'
      self.pyExp = None

      # Check if entry already exists and initialize with previous values
      if self.showCommandKey in showCommandConfig.cliRelayShowCommand:
         cliRelayShowCommand = showCommandConfig.cliRelayShowCommand[
               self.showCommandKey ]
         self.cmd = cliRelayShowCommand.showCommand
         self.rev = cliRelayShowCommand.revision
         self.responseFormat = cliRelayShowCommand.responseFormat
         self.pyExp = cliRelayShowCommand.pythonExpression

   def commit( self ):
      if not self.cmd:
         self.addError( "commit error: Command not configured" )
         return

      if not self.pyExp:
         # Empty python expression means return everything
         self.pyExp = self.responseFormat

      #print "CmdId ", self.cmdId, "Cmd ", self.cmd, "revision ", \
            #self.rev, "Python expression ", self.pyExp, "switch name ",\
            #self.switchMac, "hostname ", self.hostName, "response format ", \
            #self.responseFormat

      # If the entry already exists, delete and create new one
      if self.showCommandKey in showCommandConfig.cliRelayShowCommand:
         del showCommandConfig.cliRelayShowCommand[ self.showCommandKey ]

      showCommandConfig.newCliRelayShowCommand(
               self.showCommandKey, self.cmd,
               self.rev, self.responseFormat,
               self.pyExp )

      # Create static client for CliRelay if needed
      if cliRelayClientName not in cliRelayClients:
         cliRelayClients.newEntity( 'CliRelay::CliRelayClient', cliRelayClientName )

   def onExit( self ):
      self.commit()
      BasicCli.ConfigModeBase.onExit( self )

class EnterShowCommandConfig( CliCommand.CliCommandClass ):
   syntax = """switch mac <switchMac> <cmdId>"""
   noOrDefaultSyntax = """switch mac <switchMac> <cmdId>"""
   data = { 'switch': CliRelayShow.switchKwMatcher,
            'mac': 'Configure switch MAC address',
            '<switchMac>': CliMatcher.PatternMatcher( Ethernet.macAddrPattern,
                                                      helpname='H.H.H',
                                                     helpdesc='Switch MAC address' ),
            '<cmdId>': CliMatcher.IntegerMatcher( 0, sys.maxint,
                                                  helpname='1-%s' % sys.maxint,
                                                  helpdesc= 'show command id' )
          }

   @staticmethod
   def handler( mode, args ):
      if "<switchMac>" in args:
         switchMac = getSwitchMacEntry( args[ "<switchMac>" ] )
      else:
         switchMac = ''
      hostName = ''
      childMode = mode.childMode( ShowCommandConfigMode,
            switchMac=switchMac,
            hostName=hostName,
            cmdId=int( args[ "<cmdId>" ] ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if "<switchMac>" in args:
         switchMac = getSwitchMacEntry( args[ "<switchMac>" ] )
      else:
         switchMac = ''
      hostName = ''
      cmdId = int( args[ "<cmdId>" ] )
      showCommandKey = Tac.Value( "CliRelay::ShowCommandKey", switchMac=switchMac,
            hostName=hostName, cmdId=cmdId )
      if showCommandKey in showCommandConfig.cliRelayShowCommand:
         del showCommandConfig.cliRelayShowCommand[ showCommandKey ]
         # Check if we need to stop CliRelay agent
         if not showCommandConfig.cliRelayShowCommand:
            cliRelayClients.deleteEntity( cliRelayClientName )

CliRelayConfigMode.addCommandClass( EnterShowCommandConfig )

#-------------------------------------------------
# command <show command>
# revision <CAPI revision number>
# python-expression <multi word python expression>
#-------------------------------------------------
class ConfigureShowCommand( CliCommand.CliCommandClass ):
   syntax = """command <showCmd>"""
   noOrDefaultSyntax = """command <showCmd>"""
   data = { "command" : "Configure show command",
            "<showCmd>" : CliMatcher.StringMatcher( helpname='showCommand',
                                    helpdesc='"show command" to send to switches' ) }

   @staticmethod
   def handler(  mode, args ):
      mode.cmd = args[ "<showCmd>" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.cmd = None

class ConfigureShowCommandRevision( CliCommand.CliCommandClass ):
   syntax = """revision <revision>"""
   noOrDefaultSyntax = """revision <revision>"""
   data = { "revision": "Configure CAPI revision",
            "<revision>": CliMatcher.PatternMatcher( '[0-9]+',
                                                     helpname='CAPI revision',
                                       helpdesc='CAPI revision of show command' ) }

   @staticmethod
   def handler( mode, args ):
      mode.rev = int( args[ "<revision>" ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.rev = 1

class ConfigureShowCommandResponseFormat( CliCommand.CliCommandClass ):
   syntax = """response-format ( json | text )"""
   noOrDefaultSyntax = """response-format ( json | text )"""
   data = { "response-format" : "Response format ( json or text )",
            "json": "json format output",
            "text": "text output" }

   @staticmethod
   def handler( mode, args ):
      mode.responseFormat = 'json'
      if "text" in args:
         mode.responseFormat = 'text'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.responseFormat = 'json'

class ConfigureShowCommandPyExp( CliCommand.CliCommandClass ):
   syntax = """python-expression """
   noOrDefaultSyntax = """python-expression """

   data = { "python-expression":
            "Python expression to be evaluated on the show command output"
            " 'json' returns complete output for query in json format."
            " 'text' returns complete output for query in text format" }

   @staticmethod
   def handler( mode, args ):
      pyExp = BasicCliUtil.getMultiLineInput( mode, cmd='python expression',
            prompt='Enter python expression' )
      # We allow indented python script as input. This creates an issue when
      # we load config saved from 'show run' as 'show run' adds it's own
      # indentation based on the config mode depth
      # If first line of script starts with 9 spaces, we assume that it's
      # loaded thru saved config than input thru Cli
      indent = " " * 9
      if pyExp.split( '\n' )[0].startswith( indent ):
         mode.pyExp = '\n'.join( [ line[9:] if line.startswith(
            indent ) else line for line in pyExp.split( '\n' ) ] )
      else:
         mode.pyExp = pyExp

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.pyExp = None

ShowCommandConfigMode.addCommandClass( ConfigureShowCommand )
ShowCommandConfigMode.addCommandClass( ConfigureShowCommandRevision )
ShowCommandConfigMode.addCommandClass( ConfigureShowCommandResponseFormat )
ShowCommandConfigMode.addCommandClass( ConfigureShowCommandPyExp )

def noCvxCallback( mode, shutdown=True ):
   if shutdown:
      if cliRelayClientName in cliRelayClients.keys():
         showCommandConfig.cliRelayShowCommand.clear()
         cliRelayClients.deleteEntity( cliRelayClientName )

addNoCvxCallback( noCvxCallback )

def Plugin( entityManager ):
   global showCommandConfig
   global cliRelayClients
   showCommandConfig = ConfigMount.mount( entityManager,
                           'cliRelay/serviceAgentConfig',
                            'CliRelay::CliRelayCvxAgentConfig', 'w' )
   cliRelayClients = LazyMount.mount( entityManager,
                           'cliRelay/clients',
                            'Tac::Dir', 'wi' )

