# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from operator import itemgetter

from CliModel import Bool, Dict, Int, Model, Str, List
from TableOutput import createTable

class Aliases( Model ):
   class Alias( Model ):
      commands = Dict( keyType=int, valueType=str,
                       help="Map a sequence number to a command" )

   class RegexAlias( Alias ):
      regex = Str( help="A Python-compatible regular expression " )

   commandsPath = Str( help="Directory of scripts to be converted to aliases" )
   aliases = Dict( valueType=Alias,
                   help="Maps alias names to their corresponding commands" )
   regexAliases = List( valueType=RegexAlias,
                        help="Regular expression aliases" )
   dynamicAliases = Dict( valueType=Alias,
                          help="Maps aliases of scripts in commandsPath "
                               "to their corresponding commands." )

   def renderAliases( self, aliases ):
      for alias, definition in aliases:
         commands = definition.commands
         if len( commands ) == 1:
            print( "%s\t%s" % ( alias, commands.itervalues().next() ) )
         else:
            print( alias )
            for seq, command in sorted( commands.items() ):
               print( "   %d\t%s" % ( seq, command ) )

   def render( self ):
      if self.commandsPath:
         print( 'Commands path: ' + self.commandsPath )

      self.renderAliases( sorted( self.aliases.items() ) )
      self.renderAliases( sorted( self.dynamicAliases.items() ) )
      # For RegEx aliases, we want to use the same function, but preserve order.
      if self.regexAliases:
         quote = '"{}"'.format
         print( 'RegEx aliases:' )
         self.renderAliases( ( quote( a.regex ), a ) for a in self.regexAliases )

class AgentMemoryUsage( Model ):
   memory = Int( help='Memory usage in kB' )
   threads = Int( help='Number of threads in agent' )
   mostCommonObjects = Dict( keyType=str, valueType=int,
                 help='Counts for the 100 most common objects, keyed by their type' )

   def render( self ):
      template = 'Private memory size: {:,}kB in {} threads.\n'
      print( template.format( self.memory, self.threads ) )

      header = ( ( 'Garbage Collected Object Type', 'l' ), ( 'Count', 'r' ) )
      table = createTable( header )

      objects = sorted( self.mostCommonObjects.items(), key=itemgetter( 1 ) )
      for typeAndCount in reversed( objects ):
         table.newRow( *typeAndCount )
      print( table.output() )

class CliPid( Model ):
   pid = Int( help='PID of the current CLI' )

   def render( self ):
      print( 'CLI PID: %s' % self.pid )

class PrivilegeLevel( Model ):
   privilegeLevel = Int( help="The current privilege level" )
   _secureMonitor = Bool( help="Secure-monitor user" )

   def render( self ):
      print( "Current privilege level is %d" % ( self.privilegeLevel ) )
      if self._secureMonitor:
         print( "* This is a secure-monitor session *" )

class ShowHistory( Model ):
   # BUG68678: Currently not ready for API consumption as the return
   # will always be an empty list, because history is integrated with
   # the readline history.
   __public__ = False
   historyItems = List( valueType=str,
                                 help="List of previously run CLI commands" )

   def render( self ):
      for i in self.historyItems:
         print( '  %s' % i )
