# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import BasicCliModes
import CliCommand
from CliPlugin import CliCliModel
import CliCommon
import CliMatcher
import CliToken.Clear
import CliToken.Cli
import ConfigMount
from cStringIO import StringIO
from MainCli import loadConfigFromFile
import ShowCommand
from Syscall import gettid

import Tac

cliConfig = None

#------------------------------------------------------------------------------------
# show cli pid
#------------------------------------------------------------------------------------
class ShowCliPid( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cli pid'
   data = {
      'cli': 'Show CLI commands',
      'pid': 'PID of the current CLI'
   }
   cliModel = CliCliModel.CliPid

   @staticmethod
   def handler( mode, args ):
      return CliCliModel.CliPid( pid=gettid() )

BasicCli.addShowCommandClass( ShowCliPid )

#------------------------------------------------------------------------------------
# [ no | default ] prompt PROMPT
#------------------------------------------------------------------------------------
class PromptCmd( CliCommand.CliCommandClass ):
   syntax = 'prompt PROMPT'
   noOrDefaultSyntax = 'prompt ...'
   data = {
      'prompt': 'Configure the CLI prompt',
      'PROMPT': CliMatcher.PatternMatcher( '.+', helpname='WORD',
         helpdesc='Prompt format string' )
   }

   @staticmethod
   def handler( mode, args ):
      cliConfig.prompt = args.get( 'PROMPT', cliConfig.promptDefault )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( PromptCmd )

#-----------------------------------------------------------------------------------
# show history [ current ]
#-----------------------------------------------------------------------------------
class ShowHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show history [ current ]'
   data = {
      'history': 'CLI shell history of entered commands',
      'current': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'current',
            helpdesc='Display history for the current mode' ),
         hidden=True ),
   }
   cliModel = CliCliModel.ShowHistory

   @staticmethod
   def handler( mode, args ):
      current = 'current' in args
      # We might be running from a config mode - find it instead of the current mode
      # which is always EXEC mode.
      mode = mode.session_.modeOfLastPrompt() or mode
      items = ( mode.session_.cliInput.history( mode.historyKey() ) if current else
                mode.session_.cliInput.history() )
      return CliCliModel.ShowHistory( historyItems=items )

BasicCli.addShowCommandClass( ShowHistoryCmd )

#------------------------------------------------------------------------------------
# clear history
#------------------------------------------------------------------------------------
class ClearHistory( CliCommand.CliCommandClass ):
   syntax = 'clear history'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'history': 'Clear the session command history',
   }

   @staticmethod
   def handler( mode, args ):
      mode.session_.cliInput.clearHistory()

BasicCli.UnprivMode.addCommandClass( ClearHistory )
BasicCli.EnableMode.addCommandClass( ClearHistory )

#-----------------------------------------------------------------------------------
# show privilege
#-----------------------------------------------------------------------------------
class ShowPrivilege( ShowCommand.ShowCliCommandClass ):
   syntax = 'show privilege'
   data = {
      'privilege': 'Display the current privilege level',
   }
   cliModel = CliCliModel.PrivilegeLevel

   @staticmethod
   def handler( mode, args ):
      return CliCliModel.PrivilegeLevel(
         privilegeLevel=mode.session.privLevel_,
         _secureMonitor=mode.session.secureMonitor() )

BasicCli.addShowCommandClass( ShowPrivilege )

#------------------------------------------------------------------------------------
# [ no | default ] activity-lock-monitor
# In ConfigAgent there is an ActivityLock monitor. If this command is enabled
# it will log a syslog if the ActivityLock is held for too long in ConfigAgent
#------------------------------------------------------------------------------------
class ActivityLockMonitor( CliCommand.CliCommandClass ):
   syntax = 'activity-lock-monitor'
   noOrDefaultSyntax = syntax
   data = {
      'activity-lock-monitor': 'Enable the activity lock monitor syslog message'
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      cliConfig.activityLockSyslogMsg = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.activityLockSyslogMsg = False

BasicCli.GlobalConfigMode.addCommandClass( ActivityLockMonitor )

cliDebugNodeHidden = CliCommand.Node( CliMatcher.KeywordMatcher( 'debug',
                                          helpdesc='Cli debug commands' ),
                                      hidden=True )

#------------------------------------------------------------------------------------
# cli debug parser result
#------------------------------------------------------------------------------------
class CliDebugParserResultCmd( CliCommand.CliCommandClass ):
   syntax = 'cli debug parser result'
   noOrDefaultSyntax = syntax
   data = {
      'cli': CliToken.Cli.cliForExecMatcher,
      'debug': cliDebugNodeHidden,
      'parser': 'Parser Debugs',
      'result': 'Parser Results displayed',
   }

   @staticmethod
   def handler( mode, args ):
      mode.session.parserDebugIs( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.session.parserDebugIs( False )

BasicCli.EnableMode.addCommandClass( CliDebugParserResultCmd )

#------------------------------------------------------------------------------------
# cli debug parser guard
#------------------------------------------------------------------------------------
class CliDebugParserGuardCmd( CliCommand.CliCommandClass ):
   syntax = 'cli debug parser guard'
   noSyntax = syntax
   data = {
      'cli': CliToken.Cli.cliForExecMatcher,
      'debug': cliDebugNodeHidden,
      'parser': 'Parser Debugs',
      'guard': 'Disable parser guards',
   }

   @staticmethod
   def handler( mode, args ):
      mode.session.guardsEnabledIs( True )

   @staticmethod
   def noHandler( mode, args ):
      mode.session.guardsEnabledIs( False )

BasicCli.EnableMode.addCommandClass( CliDebugParserGuardCmd )

#------------------------------------------------------------------------------------
# cli debug memory handler { heapcheck | mallinfo | rss | gc | log }
#------------------------------------------------------------------------------------
class CliDebugMemoryHandlerCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'cli debug memory handler ...'
   syntax = 'cli debug memory handler [ OPTIONS ]'
   data = {
      'cli': CliToken.Cli.cliForExecMatcher,
      'debug': cliDebugNodeHidden,
      'memory': 'Analyze memory usage',
      'handler': 'Analyze memory usage during cli command handler execution',
      'OPTIONS': CliCommand.setCliExpression( {
         'heapcheck': ( 'Run libtcmalloc\'s heapcheck, which needs to be '
            '\'preloaded\' (LD_PRELOAD=/usr/lib/libtcmalloc.so HEAPCHECK=local), '
            'which will invalidate the mallinfo results '
            '(returns free=allocated=0)' ),
         'mallinfo': 'Display stats on free/available bytes using mallinfo',
         'rss': 'Display number of rss pages (4k) used by ConfigAgent',
         'py-objects': ( 'Display the count of python objects known to the '
                         'garbage collector' ),
         'gc': ( 'Invoke garbage collector before protocolling mem usage (will be '
                 'slow)' ),
         'log': 'Don\'t display the memory stats in the cli prompt (instead log it)',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      mode.session_.memoryDebugCfgIs( args.get( 'heapcheck' ),
          args.get( 'mallinfo' ), args.get( 'rss' ), args.get( 'py-objects' ),
          args.get( 'gc' ), args.get( 'log' ) )

   noOrDefaultHandler = handler

BasicCli.EnableMode.addCommandClass( CliDebugMemoryHandlerCmd )

#-----------------------------------------------------------------------------------
# run [ ignore-errors ] COMMAND
#-----------------------------------------------------------------------------------
def runCommandsHandler( mode, args ):
   modeClass = ( BasicCliModes.UnprivMode if
                 isinstance( mode, BasicCliModes.UnprivMode ) else
                 BasicCliModes.EnableMode )
   cmdList = ( x.strip() for x in args[ 'COMMAND' ].split( ';' ) )
   errors = 0
   ignoreErrors = 'ignore-errors' in args
   try:
      errors = loadConfigFromFile( StringIO( '\n'.join( cmdList ) ),
                                   mode.session_.entityManager_,
                                   disableAaa=mode.session_.disableAaa_,
                                   initialModeClass=modeClass,
                                   echo='e',
                                   privLevel=mode.session_.privLevel_,
                                   disableGuards=mode.session_.disableGuards_,
                                   startupConfig=mode.session_.startupConfig_,
                                   secureMonitor=mode.session_.secureMonitor_,
                                   skipConfigCheck=mode.session_.skipConfigCheck_,
                                   raiseOnException=not ignoreErrors,
                                   abortOnError=not ignoreErrors,
                                   isEapiClient=mode.session_.isEapiClient_,
                                   aaaUser=mode.session_.aaaUser_,
                                   autoComplete=mode.session_.autoComplete_ )
   except ( CliCommon.LoadConfigError ) as e:
      mode.addError( 'Error running commands (%s)' % e.msg )

   if errors:
      mode.addError( '%d command(s) failed during run.' % errors )

class runCommands( CliCommand.CliCommandClass ):
   syntax = 'run [ ignore-errors ] COMMAND'
   data = {
            'run': 'Run multiple commands in one line',
            'ignore-errors': 'Do not stop on errors',
            'COMMAND': CliMatcher.StringMatcher(
               helpname='COMMAND',
               helpdesc="Commands separated by ';'" ),
          }
   handler = runCommandsHandler

BasicCli.EnableMode.addCommandClass( runCommands )
BasicCli.UnprivMode.addCommandClass( runCommands )

#------------------------------------------------------------------------------------
# Plugin Func
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, 'cli/config', 'Cli::Config', 'w' )
   BasicCliModes.cliConfig_ = cliConfig
