#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliParser
from CliMode.Classification import ( AppTrafficRecConfigMode,
                                     AppProfileConfigMode,
                                     AppConfigMode,
                                     FieldSetL4PortConfigMode,
                                     FieldSetIpPrefixConfigMode )

from ClassificationCliLib import ( AppRecognitionContext,
                                   AppProfileContext,
                                   AppContext,
                                   CommitAbortModelet,
                                   FieldSetL4PortBaseConfigCmd,
                                   FieldSetL4PortConfigCmds,
                                   FieldSetL4PortExceptConfigCmds,
                                   FieldSetIpPrefixBaseConfigCmd,
                                   FieldSetIpPrefixConfigCmds,
                                   PrefixFieldSetCmdBase,
                                   generateFieldSetExpression,
                                   ProtocolFieldSetBaseCmd,
                                   generateTcpFlagExpression,
                                   ipv4ProtoExpr, invalidPortConflictMsg,
                                   invalidProtocolConflictMsg,
                                   protectedFieldSetNamesRegex )

appRecognitionConfig = None
appProfileIdMap = None
fieldSetConfig = None

#---------------------------------------------------
# The "application traffic recognition" mode command
#---------------------------------------------------
class AppTrafficRecConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = '''application traffic recognition'''
   noOrDefaultSyntax = syntax
   data = {
      'application': 'Configure application',
      'traffic': 'Application traffic',
      'recognition': 'Traffic recognition',
   }

   @staticmethod
   def handler( mode, args ):
      context = AppRecognitionContext( appRecognitionConfig,
                                       appProfileIdMap,
                                       fieldSetConfig )
      context.copyEditAppRecognitionConfig()
      childMode = mode.childMode( AppTrafficRecConfigMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      fieldSetConfig.fieldSetL4Port.clear()
      fieldSetConfig.fieldSetIpPrefix.clear()
      appRecognitionConfig.app.clear()
      appRecognitionConfig.appProfile.clear()

# --------------------------------------------------------------------------
# The "field-set l4-port PORT_SET_NAME" command
# --------------------------------------------------------------------------
def getL4PortFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetL4Port

l4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer-4 port field-set name",
   pattern=protectedFieldSetNamesRegex( 'port' ),
   priority=CliParser.PRIO_LOW )

class FieldSetL4PortConfigCmd( FieldSetL4PortBaseConfigCmd ):
   syntax = 'field-set l4-port FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'FIELD_SET_NAME': l4PortFieldSetNameMatcher,
   }
   data.update( FieldSetL4PortBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      parentContext = mode.getContext()
      config = parentContext.fieldSetEditConfig
      return {
         'fieldSetL4PortName': fieldSetL4PortName,
         'fieldSetConfig': config,
         'childMode': FieldSetL4PortConfigMode
      }

# --------------------------------------------------------------------------
# The "field-set ipv4 prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
def getIpPrefixFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return [ key for key in fieldSetConfig.fieldSetIpPrefix ]

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   "IP prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpPrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   # XXX ipv6 is not supported yet
   syntax = 'field-set ipv4 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'FIELD_SET_NAME': ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      assert setType == 'ipv4' # only ipv4 prefix field-set is supported
      parentContext = mode.getContext()
      config = parentContext.fieldSetEditConfig
      return {
         'fieldSetIpPrefixName': fieldSetIpPrefixName,
         'fieldSetConfig': config,
         'setType': setType,
         'childMode': FieldSetIpPrefixConfigMode
      }

# --------------------------------------------------------------------------
# The "application-profile <name>" command
# --------------------------------------------------------------------------
def appProfileNames( mode ):
   if appRecognitionConfig is None:
      return []
   return appRecognitionConfig.appProfile

appProfileNameMatcher = CliMatcher.DynamicNameMatcher(
   appProfileNames,
   "Application profile name",
   priority=CliParser.PRIO_LOW )

def appNames( mode ):
   if appRecognitionConfig is None:
      return []
   return appRecognitionConfig.app

appNameMatcher = CliMatcher.DynamicNameMatcher( appNames,
                                                "Application name",
                                                priority=CliParser.PRIO_LOW )

class AppProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'application-profile PROFILE'
   noOrDefaultSyntax = syntax
   data = {
      'application-profile': 'Configure application profile',
      'PROFILE': appProfileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'PROFILE' ]
      parentContext = mode.getContext()
      context = AppProfileContext( name, parentContext )

      if context.hasAppProfile( name ):
         context.copyEditAppProfile()
      else:
         context.newEditAppProfile()

      childMode = mode.childMode( context.childMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'PROFILE' ]
      parentContext = mode.getContext()
      context = AppProfileContext( name, parentContext )
      if context.hasAppProfile( name ):
         context.delAppProfile( name )

class AppProfileConfigCmds( CliCommand.CliCommandClass ):
   syntax = 'application APP'
   noOrDefaultSyntax = syntax
   data = {
      'application': 'Add application to application profile',
      'APP': appNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      appName = args.get( 'APP' )
      context = mode.getContext()
      context.updateApp( appName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      appName = args.get( 'APP' )
      context = mode.getContext()
      context.updateApp( appName, add=False )

# --------------------------------------------------------------------------
# The "application ipv4 <name>" command
# --------------------------------------------------------------------------
class AppConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'application ipv4 APP'
   noOrDefaultSyntax = syntax
   data = {
      'application': 'Configure application',
      'ipv4': 'IPv4',
      'APP': appNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      name = args[ 'APP' ]
      parentContext = mode.getContext()
      context = AppContext( name, parentContext )

      if context.hasApp( name ) and context.app[ name ].readonly:
         return

      if context.hasApp( name ):
         context.copyEditApp()
      else:
         context.newEditApp()

      childMode = mode.childMode( context.childMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      name = args[ 'APP' ]
      parentContext = mode.getContext()
      context = AppContext( name, parentContext )
      if context.hasApp( name ):
         context.delApp( name )

prefixFieldSetExpr = generateFieldSetExpression( ipPrefixFieldSetNameMatcher,
                                                 'FIELD_SET',
                                                 allowMultiple=False )

class AppConfigFieldSetCmd( PrefixFieldSetCmdBase ):
   data = {
      'FIELD_SET': prefixFieldSetExpr,
   }
   data.update( PrefixFieldSetCmdBase._baseData )

#------------------------------------------------------------------------------------
# "protocol PROTOCOL | TCP_UDP [ (source|destination) port field-set NAME ]"
#  where PROTOCOL is:
#  ahp      Authentication Header Protocol
#  icmp     Internet Control Message Protocol
#  igmp     Internet Group Management Protocol (IGMP)
#  ospf     OSPF routing protocol
#  pim      Protocol Independent Multicast (PIM)
#  rsvp     Resource Reservation Protocol (RSVP)
#  tcp      Transmission Control Protocol
#  udp      User Datagram Protocol
#  vrrp     Virtual Router Redundancy Protocol
#  <1-255>  IP protocol number
#------------------------------------------------------------------------------------
portKwMatcher = CliMatcher.KeywordMatcher( 'port', helpdesc='Port' )
fieldSetKwMatcher = CliMatcher.KeywordMatcher( 'field-set',
                                               helpdesc='field-set' )
tcpUdpL4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer-4 port field-set name for UDP and TCP",
   priority=CliParser.PRIO_LOW )

class ProtocolListIPv4ConfigCmd( CliCommand.CliCommandClass ):
   syntax = ( 'protocol PROTOCOL' )
   noOrDefaultSyntax = ( 'protocol [ PROTOCOL ]' )

   data = {
      'protocol': 'Protocol',
      'PROTOCOL': ipv4ProtoExpr,
   }

   @staticmethod
   def handler( mode, args ):
      protoList = args.get( 'PROTOCOL' )
      context = mode.getContext()
      if ProtocolListIPv4ConfigCmd._hasConflict( mode, protoList ):
         return

      rangeSet = protoList
      context.updateRangeAttr( 'proto', rangeSet, "Classification::ProtocolRange",
                               add=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rangeSet = args.get( 'PROTOCOL' )
      context = mode.getContext()
      if not rangeSet:
         # no protoccol
         rangeSet = set()
      context.updateRangeAttr( 'proto', rangeSet, "Classification::ProtocolRange",
                               add=False )

   @staticmethod
   def _hasConflict( mode, protoList ):
      rangeSet = protoList
      context = mode.getContext()
      ( currentProtoSet, hasPort ) = context.portAndProtoConfigured()
      updatedSet = rangeSet | currentProtoSet
      tcpUdpSet = { 6, 17 } # tcp and udp
      nonTcpUdpSet = updatedSet - tcpUdpSet

      if nonTcpUdpSet and hasPort:
         mode.addError( invalidProtocolConflictMsg )
         return True
      return False

l4PortSrcFieldSetExpr = generateFieldSetExpression( tcpUdpL4PortFieldSetNameMatcher,
                                                   'SRC_FIELD_SET_NAME',
                                                    allowMultiple=False )
l4PortDstFieldSetExpr = generateFieldSetExpression( tcpUdpL4PortFieldSetNameMatcher,
                                                   'DST_FIELD_SET_NAME',
                                                    allowMultiple=False )

class ProtocolFieldSetIPv4ConfigCmd( ProtocolFieldSetBaseCmd ):
   _sportFieldSetAttr = "srcPortFieldSet"
   _dportFieldSetAttr = "dstPortFieldSet"
   data = {
      'FLAGS_EXPR': generateTcpFlagExpression( tcpFlagsSupported=False ),
      'SRC_FIELD_SET_NAME': l4PortSrcFieldSetExpr,
      'DST_FIELD_SET_NAME': l4PortDstFieldSetExpr
   }
   data.update( ProtocolFieldSetBaseCmd._baseData )

   @classmethod
   def _updatePort( cls, mode, args, attrName, argListName, add=True ):
      pass

   @classmethod
   def _maybeHandleErrors( cls, mode, args, proto, source=False, destination=False,
                           fieldSet=False ):
      hasError = False
      context = mode.getContext()
      ( currentProtoSet, hasPort ) = context.portAndProtoConfigured()
      tcpUdpSet = { 6, 17 } # tcp and udp
      nonTcpUdpSet = currentProtoSet - tcpUdpSet
      if nonTcpUdpSet and hasPort:
         if proto and nonTcpUdpSet:
            # we already have non-tcp/Udp proto configured
            mode.addError( invalidPortConflictMsg % 'source|destination port' )
         else:
            mode.addError( invalidProtocolConflictMsg )
         hasError = True
      if not hasError:
         return
      cls._updateProtoAndPort( mode, args, proto, source, destination,
                               fieldSet=fieldSet, add=False )


BasicCli.GlobalConfigMode.addCommandClass( AppTrafficRecConfigModeCmd )
AppTrafficRecConfigMode.addModelet( CommitAbortModelet )

AppTrafficRecConfigMode.addCommandClass( FieldSetL4PortConfigCmd )
FieldSetL4PortConfigMode.addModelet( CommitAbortModelet )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortConfigCmds )
FieldSetL4PortConfigMode.addCommandClass( FieldSetL4PortExceptConfigCmds )

AppTrafficRecConfigMode.addCommandClass( FieldSetIpPrefixConfigCmd )
FieldSetIpPrefixConfigMode.addModelet( CommitAbortModelet )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixConfigCmds )

AppTrafficRecConfigMode.addCommandClass( AppProfileConfigCmd )
AppProfileConfigMode.addModelet( CommitAbortModelet )
AppProfileConfigMode.addCommandClass( AppProfileConfigCmds )

AppTrafficRecConfigMode.addCommandClass( AppConfigCmd )
AppConfigMode.addModelet( CommitAbortModelet )
AppConfigMode.addCommandClass( AppConfigFieldSetCmd )
AppConfigMode.addCommandClass( ProtocolListIPv4ConfigCmd )
AppConfigMode.addCommandClass( ProtocolFieldSetIPv4ConfigCmd )

def Plugin( entityManager ):
   global appRecognitionConfig, appProfileIdMap, fieldSetConfig

   appRecognitionConfig = ConfigMount.mount( entityManager,
                                             'classification/app-recognition/config',
                                             'Classification::AppRecognitionConfig',
                                             'w' )
   appProfileIdMap = ConfigMount.mount(
      entityManager, 'classification/app-profile-id',
      'Classification::AppProfileIdMap', 'w' )
   fieldSetConfig = ConfigMount.mount( entityManager,
                                       'classification/app-recognition/fieldset',
                                       'Classification::FieldSetConfig', 'w' )
