# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import datetime
import re
import ConfigMount
import FileCliUtil
import Url

# Get the saved checkpoints list
def getCheckpoints( mode ):
   # pylint: disable=maybe-no-member
   baseUrl = Url.parseUrl( 'checkpoint:/', 
                           Url.Context( *Url.urlArgsFromMode( mode ) ) )
   urls = [ baseUrl.child( f ) for f in baseUrl.listdir() ]
   urls.sort( key=lambda url: url.date() )
   
   return urls

def makeCheckpoint( mode, maxCheckpoints, checkpointName=None ):
   """
   Create a checkpoint of current running config
   """
   # get the current checkpoints files in dir
   urls = getCheckpoints( mode )
   # if the current checkpoints number reaches maxCheckpoints
   # delete the oldest ones
   if len( urls ) >= maxCheckpoints:
      delCheckpoint = len( urls ) - maxCheckpoints + 1
      
      try:
         for i in range( 0, delCheckpoint ):
            urls[ i ].delete( False )
      except EnvironmentError as e:
         mode.addError( "Error deleting url (%s)" % e.strerror )
         return

   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   
   # if checkpointName is not defined by the user, 
   # use the default format 'ckp-[date]-[version]'.
   # [date]: YYYYMMDD to indicate the creating date
   # [version]: an integer indicates the files created in the same day
   checkpointList = [ url.basename() for url in getCheckpoints( mode ) ]
   if checkpointName is None:
      prefix = 'ckp-' + datetime.datetime.now().strftime( '%Y%m%d' )
      # find the latest checkpoint file with the default format to check prefix
      lastCheckpoint = ''
      for c in checkpointList[ ::-1 ]:
         if c.startswith( 'ckp' ):
            lastCheckpoint = c
            break
      # if the prefix is in lastCheckpoint, meaning the files created 
      # in the same second, inceasing version number to distinguish the files
      if prefix in lastCheckpoint:
         # get the latest version number
         # pylint: disable=anomalous-backslash-in-string
         versionStr = re.match( '%s-(\d+)' % prefix, lastCheckpoint ).group( 1 )
         version = int( versionStr ) + 1
      else:
         version = 0
      checkpointName = prefix + ( '-%d' % version )

   checkpointUrl = Url.parseUrl( 'checkpoint:%s' % checkpointName, context )
   srcConfigUrl = Url.parseUrl( 'system:/running-config', context )

   # copy the running-config to checkpoint file
   FileCliUtil.copyFile( None, mode,
                         surl=srcConfigUrl,
                         durl=checkpointUrl,
                         commandSource="checkpointing at %s" % checkpointName )
   return checkpointUrl

def saveCheckpoint( mode, maxCheckpoints, checkpointName=None ):
   if maxCheckpoints != 0:
      try:
         with ConfigMount.ConfigMountDisabler():
            return makeCheckpoint( mode, maxCheckpoints,
                                   checkpointName=checkpointName )
      except OSError as e:
         mode.addWarning( 'Cannot make checkpoint file: %s' % e.strerror )
