# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CheckpointCliLib
import CliCommand
import CliMatcher
import CliToken.Configure
from CliToken.Service import configMatcherForAfterService, serviceMatcherForConfig
import ConfigMount
import ConfigSessionModel
import ConfigSessionCli
import ConfigSessionShowCli
import FileCliUtil
import SessionCli
import ShowCommand
import Url
import CliPlugin.TechSupportCli as TechSupportCli

cliConfig = None

matcherCheckpoint = CliMatcher.KeywordMatcher( 'checkpoint',
      helpdesc='Creating a new checkpoint' )

#--------------------------------------------------------------------------------
# [ no | default ] service configuration checkpoint max saved MAX_CHECKPOINTS
#--------------------------------------------------------------------------------
# set max checkpoint file number in configure mode
def setMaxCheckpointNum( mode, args ):
   maxCheckpoints = args.get( 'MAX_CHECKPOINTS' )
   if maxCheckpoints is None:
      maxCheckpoints = cliConfig.defaultMaxCheckpoints

   cliConfig.maxCheckpoints = maxCheckpoints
   
   # sync the checkpoint dir
   # delete the oldest files beyond the max
   try:
      urls = CheckpointCliLib.getCheckpoints( mode )
   except OSError: 
      return
   if len( urls ) > maxCheckpoints:
      deleteCheckpoints = len( urls ) - maxCheckpoints
      urls.sort( key=lambda url: url.date() )
   
      try:
         for i in range( 0, deleteCheckpoints ):
            urls[ i ].delete( False )
      except EnvironmentError, e:
         mode.addError( "Error deleting url (%s)" % e.strerror )

class GlobalMaxCheckpointNumCmd( CliCommand.CliCommandClass ):
   syntax = 'service configuration checkpoint max saved MAX_CHECKPOINTS'
   noOrDefaultSyntax = 'service configuration checkpoint max saved ...'
   data = {
      'service': serviceMatcherForConfig,
      'configuration': configMatcherForAfterService,
      'checkpoint': 'Creating a new checkpoint',
      'max': 'Set a max limit',
      'saved': 'Maximum number of saved checkpoints',
      'MAX_CHECKPOINTS': CliMatcher.IntegerMatcher( 0, 100,
         helpdesc='Maximum number of saved checkpoints' ),
   }
   handler = setMaxCheckpointNum
   noOrDefaultHandler = setMaxCheckpointNum

BasicCliModes.GlobalConfigMode.addCommandClass( GlobalMaxCheckpointNumCmd )

#--------------------------------------------------------------------------------
# configure checkpoint restore CHECKPOINT_NAME
#--------------------------------------------------------------------------------
def getCheckpointNames( mode ):
   return [ ckp.pathname[ 1: ] for ckp 
               in CheckpointCliLib.getCheckpoints( mode) ]

def restoreCheckpoint( mode, args ):
   # get the url from checkpoint name
   restoreCheckpointName = args[ 'CHECKPOINT_NAME' ]
   url = Url.parseUrl( 'checkpoint:%s' % restoreCheckpointName,
                       Url.Context( *Url.urlArgsFromMode( mode ) ) )
   FileCliUtil.checkUrl( url )
   # replace the running-config with the checkpoint file
   ConfigSessionCli.configureReplace( mode, url )

class ConfigureCheckpointRestoreCmd( CliCommand.CliCommandClass ):
   syntax = 'configure checkpoint restore CHECKPOINT_NAME'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'checkpoint': matcherCheckpoint,
      'restore': 'Restore running-config from a previous checkpoint file',
      'CHECKPOINT_NAME': CliMatcher.DynamicNameMatcher(
         getCheckpointNames, 'Restore checkpoint name',
         pattern='\\S+' ),
   }
   handler = restoreCheckpoint

BasicCliModes.EnableMode.addCommandClass( ConfigureCheckpointRestoreCmd )

#--------------------------------------------------------------------------------
# configure checkpoint save CHECKPOINT_NAME
#--------------------------------------------------------------------------------
def configureCheckpoint( mode, args ):
   checkpointName = args.get( 'CHECKPOINT_NAME' )
   if cliConfig.maxCheckpoints == 0:
      return
   try:
      CheckpointCliLib.makeCheckpoint( mode, cliConfig.maxCheckpoints, 
                                       checkpointName )
   except IOError as e:
      mode.addError( 'Cannot make checkpoint file due to %s.' % e.strerror )

class ConfigureCheckpointSaveCmd( CliCommand.CliCommandClass ):
   syntax = 'configure checkpoint save [ CHECKPOINT_NAME ]'
   data = {
      'configure': CliToken.Configure.configureParseNode,
      'checkpoint': matcherCheckpoint,
      'save': 'Save running-config to a checkpoint file',
      'CHECKPOINT_NAME': CliMatcher.PatternMatcher( pattern='\\S+',
         helpdesc='Checkpoint name', helpname='WORD' ),
   }
   handler = configureCheckpoint

BasicCliModes.EnableMode.addCommandClass( ConfigureCheckpointSaveCmd )

#-----------------------------------------------------------------------------------
# show configuration checkpoints
# show the checkpoint file name, creating date and creating user
#-----------------------------------------------------------------------------------
def showConfigureCheckpoint( mode, args ):
   try:
      urls = CheckpointCliLib.getCheckpoints( mode )
   except OSError as e:
      mode.addError( 'Cannot show checkpoint due to %s' % str( e ) )
      return None

   checkpoints = {}
   for url in urls:
      fileName = url.basename() 
      fileUser = url.getUserName()
      fileDate = url.date() 
      checkpoints[ fileName ] = ConfigSessionModel.Checkpoint( 
                                 fileDate = fileDate, fileUser = fileUser )

   return ConfigSessionModel.Checkpoints( checkpoints=checkpoints, 
                                          maxCheckpoints=cliConfig.maxCheckpoints )

class ShowConfigCheckpoints( ShowCommand.ShowCliCommandClass ):
   syntax = 'show configuration checkpoints'
   data = {
            'configuration': ConfigSessionShowCli.configKwMatcher,
            'checkpoints': 'List all saved checkpoints',
          }
   cliModel = ConfigSessionModel.Checkpoints
   handler = showConfigureCheckpoint

BasicCli.addShowCommandClass( ShowConfigCheckpoints )

TechSupportCli.registerShowTechSupportCmdCallback(
    '2020-05-12 01:22:30', lambda: [],
    summaryCmdCallback=lambda:[ 'show configuration checkpoints' ] )

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager, 
                                  "cli/session/input/config",
                                  "Cli::Session::CliConfig", "w" )
   
   #Register the callback function makeCheckpoint to Cli
   SessionCli.setCheckpointFunc( callback=CheckpointCliLib.saveCheckpoint )
