#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import IntfModels
import CliModel
import Arnet
import TableOutput
import IntfCli

class CfmIntfCounter( CliModel.Model ):
   inPkts = CliModel.Int( help="Number of CFM packets received on the interface" )
   outPkts = CliModel.Int( help="Number of CFM packets sent from the interface" )
   errorPkts = CliModel.Int(
      help="Number of invalid CFM packets received on the interface" )

class CfmCounter( CliModel.Model ):
   interfaces = CliModel.Dict( keyType=IntfModels.Interface,
      valueType=CfmIntfCounter,
      help="Maps CFM counters to the corresponding interfaces" )

   def render( self ):
      headings = ( "Interface", "InPkts", "OutPkts", "ErrorPkts" )
      table = TableOutput.createTable( headings )
      fmtName = TableOutput.Format( justify="left" )
      fmtName.noPadLeftIs( True )
      fmtName.padLimitIs( True )
      fmtPkts = TableOutput.Format( justify="right" )
      fmtPkts.noPadLeftIs( True )
      fmtPkts.padLimitIs( True )
      table.formatColumns( fmtName, fmtPkts, fmtPkts, fmtPkts )

      for intf in Arnet.sortIntf( self.interfaces ):
         model = self.interfaces[ intf ]
         table.newRow( IntfCli.Intf.getShortname( intf ),
                       model.inPkts, model.outPkts, model.errorPkts )
      print table.output()

class CfmMaintenanceDomain( CliModel.Model ):
   intermediatePoint = CliModel.Dict( keyType=int, valueType=bool,
                              help='Maps Maintenance Domains to Intermediate Point' )

   def render( self ):
      table = TableOutput.createTable( ( 'Domain', 'Intermediate Point' ) )
      fmtName = TableOutput.Format( justify='left' )
      fmtName.noPadLeftIs( True )
      fmtName.padLimitIs( True )
      fmtMip = TableOutput.Format( justify='right' )
      fmtMip.noPadLeftIs( True )
      fmtMip.padLimitIs( True )
      table.formatColumns( fmtName, fmtMip )

      for ( domain, defaultMip ) in sorted( self.intermediatePoint.items() ):
         table.newRow( domain, 'enabled' if defaultMip else 'disabled' )
      print table.output()
