#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliParser
import CliToken
import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.IntfCli as IntfCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.VirtualIntfRule as VirtualIntfRule
import ShowCommand
import ConfigMount
import LazyMount
import Ethernet
import Arnet
import TableOutput
import MultiRangeRule
import Intf.IntfRange
import AgentCommandRequest
import CfmAgent
from Ark import timestampToStr
from CfmLib import ( ccmTxIntervalToCliToken, defaultCcmTxIntervalWithUnit,
                     getCcmTxInterval )
from CfmTypes import ( tacMdName, defaultCfmProfileName, tacMepDirection,
                       defaultMepDirection, defaultPrimaryVlanId )
from CliMode.Cfm import ( CfmModeBase, MaintenanceDomainModeBase, CfmProfileModeBase,
                          MaintenanceAssociationModeBase,
                          LocalMaintenanceEndPointModeBase )

from CfmAgentTypes import tacCounterClass
from CfmCliModel import CfmIntfCounter, CfmCounter, CfmMaintenanceDomain
from TypeFuture import TacLazyType

#-----------------------------------------------------------
# The Cfm globals
#-----------------------------------------------------------
cfmConfig = None
cfmStatus = None
aleCfmConfig = None
aleCfmStatus = None
cfmHwSupportStatus = None
trapConfig = None

disableMIPMsg = 'Disabling the MIP on MD Level %s'

TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )

def waitForAgent():
   return cfmConfig.mdConfig or cfmStatus.mdStatus or \
      aleCfmConfig.mdConfig or aleCfmStatus.mdStatus

def getCfmProfile( mode ):
   return cfmConfig.cfmProfile.get( mode.cfmProfileName )

def getMdConfig( mode ):
   return cfmConfig.mdConfig.get( mode.mdLevel )

def getDefaultCfmProfile():
   defaultCfmProfile = cfmConfig.cfmProfile.get( defaultCfmProfileName )
   assert defaultCfmProfile is not None
   return defaultCfmProfile

def localMepExist( mdLevel ):
   mdConfig = cfmConfig.mdConfig.get( mdLevel )
   if mdConfig is None:
      return False
   for maNameId in mdConfig.maConfig:
      maConfig = mdConfig.maConfig.get( maNameId )
      if maConfig is None:
         continue
      if maConfig.localMepConfig:
         return True
   return False

def getMaConfig( mode ):
   mdConfig = getMdConfig( mode )
   return mdConfig.maConfig.get( mode.maNameId ) if mdConfig else None

def getLocalMepConfig( mode ):
   maConfig = getMaConfig( mode )
   return maConfig.localMepConfig.get( mode.localMepId ) if maConfig else None

#-----------------------------------------------------------
# Modes
#-----------------------------------------------------------
class CfmMode( CfmModeBase, BasicCli.ConfigModeBase ):
   name = 'CFM configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CfmModeBase.__init__( self, param=None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MaintenanceDomainMode( MaintenanceDomainModeBase,
                             BasicCli.ConfigModeBase ):
   name = 'CFM Maintenance Domain configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      MaintenanceDomainModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class CfmProfileMode( CfmProfileModeBase, BasicCli.ConfigModeBase ):
   name = 'CFM Profile configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      CfmProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MaintenanceAssociationMode( MaintenanceAssociationModeBase,
                                  BasicCli.ConfigModeBase ):
   name = 'CFM Maintenance Association configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      MaintenanceAssociationModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class LocalMaintenanceEndPointMode( LocalMaintenanceEndPointModeBase,
                                    BasicCli.ConfigModeBase ):
   name = 'CFM Local Maintenance EndPoint configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      LocalMaintenanceEndPointModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------
# Cli Guard
#-----------------------------------------------------------
def cfmSupported( mode, token ):
   if not cfmHwSupportStatus.hwSupported():
      return CliParser.guardNotThisPlatform
   return None

def defaultMipSupported( mode, token ):
   if not cfmHwSupportStatus.defaultMipHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def mepSupported( mode, token ):
   if not cfmHwSupportStatus.mepHwSupported():
      return CliParser.guardNotThisPlatform
   return None

def upMepSupported( mode, token ):
   if not cfmHwSupportStatus.upMepHwSupported:
      return CliParser.guardNotThisPlatform
   return None

def downMepSupported( mode, token ):
   if not cfmHwSupportStatus.downMepHwSupported:
      return CliParser.guardNotThisPlatform
   return None

#-----------------------------------------------------------
# (config)# cfm
#-----------------------------------------------------------
class CfmCommand( CliCommand.CliCommandClass ):
   syntax = 'cfm'

   data = { 'cfm':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'cfm',
                     helpdesc='Configure Connectivity Fault Management parameters' ),
                  guard=cfmSupported )
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CfmMode )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( CfmCommand )

#-----------------------------------------------------------
# (config-cfm)# [no|default] domain <0-7>
#-----------------------------------------------------------
class DomainCommand( CliCommand.CliCommandClass ):
   syntax = 'domain DOMAIN'
   noOrDefaultSyntax = 'domain DOMAIN'

   data = { 'domain':
               'Configure a Maintenance Domain',
            'DOMAIN':
                CliMatcher.IntegerMatcher( 0, 7,
                   helpdesc='Maintenance Domain level' )
          }

   @staticmethod
   def handler( mode, args ):
      mdLevel = args[ 'DOMAIN' ]
      mdConfig = cfmConfig.mdConfig.newMember( mdLevel )
      mdConfig.config = cfmConfig
      trapConfig.features.add( TrapFeatureName.cfm )
      childMode = mode.childMode( MaintenanceDomainMode, param=mdLevel )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mdLevel = args[ 'DOMAIN' ]
      del cfmConfig.mdConfig[ mdLevel ]
      if not cfmConfig.mdConfig and \
            TrapFeatureName.cfm in trapConfig.features:
         trapConfig.features.remove( TrapFeatureName.cfm )

CfmMode.addCommandClass( DomainCommand )

#------------------------------------------------------------------------------------
# (config-cfm)# [no|default] profile <cfmProfileName>
#------------------------------------------------------------------------------------
class ProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE'
   noOrDefaultSyntax = 'profile PROFILE'

   data = { 'profile':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'profile',
                     helpdesc='Configure Connectivity Fault Management profile' ),
                  guard=mepSupported ),
            'PROFILE':
               CliMatcher.StringMatcher(
                  helpname='STRING',
                  helpdesc='Connectivity Fault Management profile name' )
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfileName = args[ 'PROFILE' ]
      cfmConfig.cfmProfile.newMember( cfmProfileName )
      childMode = mode.childMode( CfmProfileMode, param=cfmProfileName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfileName = args[ 'PROFILE' ]
      del cfmConfig.cfmProfile[ cfmProfileName ]

CfmMode.addCommandClass( ProfileCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)# [no|default] continuity-check
#------------------------------------------------------------------------------------
class ContinuityCheckCommand( CliCommand.CliCommandClass ):
   syntax = 'continuity-check'
   noOrDefaultSyntax = 'continuity-check'

   data = { 'continuity-check':
               'Configure Continuity Check protocol parameters'
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.ccmEnable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.ccmEnable = False

CfmProfileMode.addCommandClass( ContinuityCheckCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] continuity-check tx-interval <INTERVAL>
#------------------------------------------------------------------------------------
def intervalSyntax():
   choices = []
   for ccmTxIntervalWithUnit in ccmTxIntervalToCliToken.values():
      if ccmTxIntervalWithUnit.ccmTxInterval == 0:
         continue
      choices.append( '( %s %s )' % ( ccmTxIntervalWithUnit.ccmTxInterval,
                                      ccmTxIntervalWithUnit.unit ) )

   return ' | '.join( choices )

def intervalData():
   data = {}
   for interval in intervals():
      data[ interval ] = 'Set Continuity Check transmission interval to ' + interval
   for unit in units():
      data[ unit ] = 'Continuity Check transmission interval in ' + unit
   return data

def intervals():
   result = set()
   for ccmTxIntervalWithUnit in ccmTxIntervalToCliToken.values():
      if ccmTxIntervalWithUnit.ccmTxInterval == 0:
         continue
      result.add( str( ccmTxIntervalWithUnit.ccmTxInterval ) )
   return result

def units():
   result = set()
   for ccmTxIntervalWithUnit in ccmTxIntervalToCliToken.values():
      if ccmTxIntervalWithUnit.ccmTxInterval == 0:
         continue
      result.add( ccmTxIntervalWithUnit.unit )
   return result

class ContinuityCheckTxIntervalCommand( CliCommand.CliCommandClass ):
   syntax = 'continuity-check tx-interval ( %s )' % intervalSyntax()
   noOrDefaultSyntax = 'continuity-check tx-interval ...'

   data = { 'continuity-check':
               'Configure Continuity Check protocol parameters',
            'tx-interval':
               'Set Continuity Check transmission interval'
          }
   data.update( intervalData() )

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      interval = set( args.iterkeys() ) & intervals()
      assert len( interval ) == 1
      interval = interval.pop()
      unit = set( args.iterkeys() ) & units()
      assert len( unit ) == 1
      unit = unit.pop()
      cfmProfile.ccmTxInterval = getCcmTxInterval( interval, unit )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.ccmTxInterval = getCcmTxInterval(
                                    defaultCcmTxIntervalWithUnit.ccmTxInterval,
                                    defaultCcmTxIntervalWithUnit.unit )

CfmProfileMode.addCommandClass( ContinuityCheckTxIntervalCommand )

#-----------------------------------------------------------
# (config-cfm-mdX)# [no|default] name <string>
#-----------------------------------------------------------
class DomainNameCommand( CliCommand.CliCommandClass ):
   syntax = 'name NAME'
   noOrDefaultSyntax = 'name ...'

   data = { 'name':
               'Configure the Maintenance Domain name',
            'NAME':
               CliMatcher.PatternMatcher( r'.{1,43}',
                  helpname='STRING',
                  helpdesc='Maintenance Domain name' )
          }

   @staticmethod
   def handler( mode, args ):
      mdName = args[ 'NAME' ]
      mdConfig = getMdConfig( mode )
      assert mdConfig
      mdConfig.mdName = tacMdName( mdName, 'mdNameFormatCharString' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mdConfig = getMdConfig( mode )
      assert mdConfig
      mdConfig.mdName = tacMdName( '', 'mdNameFormatCharString' )

MaintenanceDomainMode.addCommandClass( DomainNameCommand )

#----------------------------------------------------------------------
#  (config-cfm-mdX)# [no|default] intermediate-point
#----------------------------------------------------------------------
class IntermediatePointCommand( CliCommand.CliCommandClass ):
   syntax = 'intermediate-point'
   noOrDefaultSyntax = 'intermediate-point'

   data = { 'intermediate-point':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'intermediate-point',
                     helpdesc='Configure the switch as '
                              'Maintenance Intermediate Point in this domain' ),
                  guard=defaultMipSupported )
          }

   @staticmethod
   def handler( mode, args ):
      mdConfig = getMdConfig( mode )
      assert mdConfig
      if localMepExist( mode.mdLevel ):
         mode.addWarning( disableMIPMsg % mode.mdLevel )
      mdConfig.defaultMip = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mdConfig = getMdConfig( mode )
      assert mdConfig
      mdConfig.defaultMip = False

MaintenanceDomainMode.addCommandClass( IntermediatePointCommand )

#------------------------------------------------------------------------------------
# (config-md<mdLevel>)# [no|default] association <maName>
#------------------------------------------------------------------------------------
class AssociationCommand( CliCommand.CliCommandClass ):
   syntax = 'association NAME'
   noOrDefaultSyntax = 'association NAME'

   data = { 'association':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'association',
                     helpdesc='Configure Maintenance Association' ),
                  guard=mepSupported ),
            'NAME':
               CliMatcher.PatternMatcher( r'.{1,45}',
                  helpname='STRING',
                  helpdesc='Maintenance Association name' )
          }

   @staticmethod
   def handler( mode, args ):
      maName = args[ 'NAME' ]
      mdConfig = getMdConfig( mode )
      assert mdConfig
      maConfig = mdConfig.maConfig.newMember( maName )
      maConfig.mdConfig = mdConfig
      childMode = mode.childMode( MaintenanceAssociationMode,
                                  param=( mode.mdLevel, maName ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maName = args[ 'NAME' ]
      mdConfig = getMdConfig( mode )
      assert mdConfig
      del mdConfig.maConfig[ maName ]

MaintenanceDomainMode.addCommandClass( AssociationCommand )

#------------------------------------------------------------------------------------
# (config-md<mdLevel>-ma-<maNameId>)# [no|default] direction <up|down>
# Default direction is 'down'
#------------------------------------------------------------------------------------
class DirectionCommand( CliCommand.CliCommandClass ):
   syntax = 'direction ( up | down )'
   noOrDefaultSyntax = 'direction ...'

   data = { 'direction':
               'Set Local Maintenance Endpoint direction',
            'up':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'up',
                     helpdesc='Set direction as Up' ),
                  guard=upMepSupported ),
            'down':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'down',
                     helpdesc='Set direction as Down' ),
                  guard=downMepSupported ),
          }

   @staticmethod
   def handler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      if 'down' in args:
         maConfig.direction = tacMepDirection.MepDirectionDown
      elif 'up' in args:
         maConfig.direction = tacMepDirection.MepDirectionUp
      else:
         assert False, 'Invalid direction'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.direction = defaultMepDirection

MaintenanceAssociationMode.addCommandClass( DirectionCommand )

#------------------------------------------------------------------------------------
# (config-md<mdLevel>-ma-<maNameId>)#[ no|default ] profile <cfmProfileName>
#------------------------------------------------------------------------------------
class MaProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE'
   noOrDefaultSyntax = 'profile ...'

   data = { 'profile':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'profile',
                     helpdesc='Apply Connectivity Fault Management profile' ),
                  guard=mepSupported ),
            'PROFILE':
               CliMatcher.StringMatcher(
                  helpname='STRING',
                  helpdesc='Connectivity Fault Management profile name' )
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfileName = args[ 'PROFILE' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      cfmProfile = cfmConfig.cfmProfile.get( cfmProfileName )
      if not cfmProfile:
         mode.addWarning( 'Connectivity Fault Management profile %s doesn\'t exist' %
                          cfmProfileName )
      maConfig.cfmProfileName = cfmProfileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.cfmProfileName = defaultCfmProfileName

MaintenanceAssociationMode.addCommandClass( MaProfileCommand )

#------------------------------------------------------------------------------------
# (config-md<mdLevel>-ma-<maNameId>)# [ no|default ] vlan <1-4094>
# By default, vlanId will be 0 which means MA is not associated with any vlan
#------------------------------------------------------------------------------------
class VlanCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN'
   noOrDefaultSyntax = 'vlan ...'

   data = { 'vlan':
               'Set VLAN in the Maintenance Association',
            'VLAN':
               CliMatcher.IntegerMatcher( 1, 4094,
                  helpdesc='Set VLAN ID in the given range' )
          }

   @staticmethod
   def handler( mode, args ):
      vlanId = args[ 'VLAN' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.primaryVlanId = vlanId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.primaryVlanId = defaultPrimaryVlanId

MaintenanceAssociationMode.addCommandClass( VlanCommand )

#------------------------------------------------------------------------------------
# (config-md<mdLevel>-ma-<maNameId>)# [no|default] endpoint <1-8191>
#-----------------------------------------------------------------------------------
class EndpointCommand( CliCommand.CliCommandClass ):
   syntax = 'endpoint ENDPOINT'
   noOrDefaultSyntax = 'endpoint ENDPOINT'

   data = { 'endpoint':
               'Configure Local Maintenance EndPoint',
            'ENDPOINT':
               CliMatcher.IntegerMatcher( 1, 8191,
                  helpdesc='Set Local Maintenance EndPoint ID' )
          }

   @staticmethod
   def handler( mode, args ):
      mepId = args[ 'ENDPOINT' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      if getMdConfig( mode ).defaultMip:
         mode.addWarning( disableMIPMsg % mode.mdLevel )
      localMepConfig = maConfig.localMepConfig.newMember( mepId )
      localMepConfig.maConfig = maConfig
      childMode = mode.childMode( LocalMaintenanceEndPointMode,
                                  param=( mode.mdLevel, mode.maNameId, mepId ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mepId = args[ 'ENDPOINT' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      del maConfig.localMepConfig[ mepId ]

MaintenanceAssociationMode.addCommandClass( EndpointCommand )

#--------------------------------------------------------------------------------
# [ no | default ] interface INTF
#--------------------------------------------------------------------------------
intfMatcher = VirtualIntfRule.IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher

class InterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTF'
   noOrDefaultSyntax = 'interface ...'
   data = {
      'interface' : 'Configure Local Maintenance Endpoint on interface',
      'INTF' : intfMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      localMepConfig = getLocalMepConfig( mode )
      assert localMepConfig
      intf = args.get( 'INTF' )
      intf = '' if intf is None else intf.name
      localMepConfig.intfId = Tac.newInstance( 'Arnet::IntfId', intf )

   noOrDefaultHandler = handler

LocalMaintenanceEndPointMode.addCommandClass( InterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] remote endpoint [ ACTION ] REMOTE_MEP_IDS
#--------------------------------------------------------------------------------
class RemoteMaintenanceEndPointCmd( CliCommand.CliCommandClass ):
   syntax = 'remote endpoint [ ACTION ] REMOTE_MEP_IDS'
   noOrDefaultSyntax = 'remote endpoint ...'
   data = {
      'remote' : 'Configure Remote Maintenance Endpoint',
      'endpoint' : 'Configure Remote Bridge parameters',
      'ACTION' : CliMatcher.EnumMatcher( {
         'add' : 'Add remote endpoint(s) to the current list',
         'remove' : 'Remove remote endpoint(s) from the current list',
      } ),
      'REMOTE_MEP_IDS' : MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, 8191 ),
         noSingletons=False,
         helpdesc=( 'Remote maintenance endpoint id(s) or range(s) of '
                    'remote maintenance endpoint ids' ) )
   }

   @staticmethod
   def handler( mode, args ):
      remoteMepIds = args[ 'REMOTE_MEP_IDS' ].values()
      action = args.get( 'ACTION' )
      localMepConfig = getLocalMepConfig( mode )
      if localMepConfig.localMepId in remoteMepIds:
         mode.addError( 'Remote MEP ID must be different from Local MEP ID' )
         return
      if action == 'add': # Addition of remote MEP IDs
         for remoteMepId in remoteMepIds:
            localMepConfig.remoteMepConfig[ remoteMepId ] = True
      elif action == 'remove': # Deletion of remote MEP IDs
         for remoteMepId in remoteMepIds:
            del localMepConfig.remoteMepConfig[ remoteMepId ]
      else: # Overwrite the existing remote MEP IDs with the new ones
         for remoteMepId in localMepConfig.remoteMepConfig:
            if remoteMepId not in remoteMepIds:
               del localMepConfig.remoteMepConfig[ remoteMepId ]
         for remoteMepId in remoteMepIds:
            localMepConfig.remoteMepConfig[ remoteMepId ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      localMepConfig = getLocalMepConfig( mode )
      for remoteMepId in localMepConfig.remoteMepConfig:
         del localMepConfig.remoteMepConfig[ remoteMepId ]

LocalMaintenanceEndPointMode.addCommandClass( RemoteMaintenanceEndPointCmd )

# Cfm global tokens
nodeCfm = CliCommand.guardedKeyword( 'cfm',
      helpdesc='Connectivity Fault Management information', guard=cfmSupported )
intfTypes = ( LagIntfCli.LagAutoIntfType, EthIntfCli.EthPhyAutoIntfType )
intfRangeMatcher = Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes )
matcherCfmInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='CFM interface level information' )
matcherCfmCounters = CliMatcher.KeywordMatcher( 'counters',
                                          helpdesc='CFM counters' )

#--------------------------------------------------------------------------------
# show cfm counters [ interface INTF ]
#--------------------------------------------------------------------------------
def showInterfaceCounters( mode, args ):
   intfs = args.get( 'INTF' )
   cfmCounter = CfmCounter()
   intfNames = []
   if intfs:
      intfNames += list( intfs.intfNames() )
   else:
      intfNames = IntfCli.Intf.getAll( mode, intf=intfs,
            intfType=[ LagIntfCli.EthLagIntf, EthIntfCli.EthPhyIntf ] )
      intfNames = [ intf.name for intf in intfNames ]
   if not intfNames:
      return cfmCounter
   for intf in intfNames:
      if not ( Tac.Type( 'Arnet::EthIntfId' ).isEthIntfId( intf ) or
               Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( intf ) ):
         continue
      cfmCounter.interfaces[ intf ] = getCfmIntfCounter( intf )
   return cfmCounter

def getCfmIntfCounter( intfId ):
   cfmIntfCounter = CfmIntfCounter()
   counterHelper = Tac.newInstance( "CfmAgent::CounterHelper", cfmStatus.counter )
   counterFilter = Tac.newInstance( "CfmAgent::CounterFilter" )
   counterFilter.intf[ intfId ] = True
   counterFilter.counterClass[ tacCounterClass.counterClassIn ] = True
   counterFilter.counterClass[ tacCounterClass.counterClassInError ] = True
   counterFilter.counterClass[ tacCounterClass.counterClassOut ] = True

   aggregateCounterClass = counterHelper.aggregateCounterClass( counterFilter )
   cfmIntfCounter.inPkts = aggregateCounterClass.counter.get(
      tacCounterClass.counterClassIn, 0 )
   cfmIntfCounter.errorPkts = aggregateCounterClass.counter.get(
      tacCounterClass.counterClassInError, 0 )
   cfmIntfCounter.outPkts = aggregateCounterClass.counter.get(
      tacCounterClass.counterClassOut, 0 )
   return cfmIntfCounter

class CfmCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm counters [ interface INTF ]'
   data = {
      'cfm' : nodeCfm,
      'counters' : matcherCfmCounters,
      'interface' : matcherCfmInterface,
      'INTF' : intfRangeMatcher,
   }

   cliModel = CfmCounter
   handler = showInterfaceCounters

BasicCli.addShowCommandClass( CfmCountersCmd )

#--------------------------------------------------------------------------------
# clear cfm counters [ interface INTF ]
#--------------------------------------------------------------------------------
class ClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cfm counters [ interface INTF ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfm,
      'counters' : matcherCfmCounters,
      'interface' : matcherCfmInterface,
      'INTF' : intfRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # command format : protocol, domain, interface1, .., interfaceN
      # value of protocol, domain, list of interfaces are used by counterFilter to
      # clear the corresponding protocol, mdLevel and interfaces counters
      # empty value for each field implies to clear counters over all possible
      # values for the field
      protocol = ''
      intfs = args.get( 'INTF' )
      mdLevel = ''
      if not waitForAgent():
         return
      command = []
      command.append( protocol )
      command.append( str( mdLevel ) )
      if intfs:
         command += list( intfs.intfNames() )
      else:
         command.append( '' )
      command = ','.join( command )
      AgentCommandRequest.runSocketCommand( mode.entityManager, CfmAgent.name,
                                            "CfmCliCallback",
                                            command, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearCountersCmd )

#--------------------------------------------------------------------------------
# show cfm continuity-check intermediate-point database [ count ]
#--------------------------------------------------------------------------------
nodeIntermediatePoint = CliCommand.guardedKeyword( 'intermediate-point',
      helpdesc='Maintenance Intermediate Point information',
      guard=defaultMipSupported )
matcherDatabase = CliMatcher.KeywordMatcher( 'database',
      helpdesc='Database entries' )
matcherContinuityCheck = CliMatcher.KeywordMatcher( 'continuity-check',
      helpdesc='Continuity Check information' )

def intfThenMacKey( keyValuePair ):
   '''We sort by interface, then MAC address.
   '''
   mac, intf = keyValuePair
   return ( Arnet.intfNameKey( intf ), mac )

class ShowMipCcmDbCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check intermediate-point database [ count ]'
   data = {
      'cfm' : nodeCfm,
      'continuity-check' : matcherContinuityCheck,
      'intermediate-point' : nodeIntermediatePoint,
      'database' : matcherDatabase,
      'count' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'count', helpdesc='Count keyword' ),
         hidden=True ),
   }

   @staticmethod
   def handler( mode, args ):
      count = 'count' in args
      mipCcmDb = cfmStatus.mipCcmDb

      if count:
         count = 0
         if mipCcmDb:
            count = sum( [ len( mipCcmDbVlan.entry )
                           for mipCcmDbVlan in mipCcmDb.vlan.values() ] )
         print 'Total entries in database:', count
         return
      print '%-4s    %-14s    %-8s' % ( 'Vlan', 'Mac Address', 'Interface' )
      print '%-4s    %-14s    %-8s' % ( '----', '-----------', '---------' )

      if not mipCcmDb:
         return
      for vlanId, mipCcmDbVlan in sorted( mipCcmDb.vlan.items() ):
         for mac, intf in sorted( mipCcmDbVlan.entry.items(), key=intfThenMacKey ):
            print '%4d    %-14s    %-8s' % ( vlanId,
                  Ethernet.convertMacAddrCanonicalToDisplay( mac ),
                  IntfCli.Intf.getShortname( intf ) )

BasicCli.addShowCommandClass( ShowMipCcmDbCmd )

#--------------------------------------------------------------------------------
# show cfm continuity-check database [ domain DOMAIN ]
#  Database for Maintenance Domain level: <mdLevel>
#  Maintenance Association Name : <maName1>
#  --------------------------------------------
#  MEP Id  Mac Address     Interface  Vlan  Last Received
#  ------  -------------   ---------  ----  -------------
#  666     0012.2222.1212  Et1/3      300  0:00:02 ago
#  777     00aa.aaaa.aaaa  Et1/1      100  0:00:02 ago
#
#  Database for Maintenance Domain level: <mdLevel>
#  Maintenance Association Name : <maName2>
#  --------------------------------------------
#  MEP Id  Mac Address     Interface  Vlan Last Received
#  ------  -------------   ---------  ---- -------------
#--------------------------------------------------------------------------------
class CfmContinuityCheckDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check database [ domain DOMAIN ]'
   data = {
      'cfm' : nodeCfm,
      'continuity-check' : matcherContinuityCheck,
      'database' : matcherDatabase,
      'domain' : 'Identify the database by its Maintenance Domain level',
      'DOMAIN' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Maintenance Domain level' ),
   }

   @staticmethod
   def handler( mode, args ):
      domain = args.get( 'DOMAIN' )
      ccmDb = cfmStatus.ccmDb
      hdr = ' Database for Maintenance Domain level:'
      hdrMa = ' Maintenance Association Name :'
      hdrLine = '----------------------------------------'
      hdrTable = [ 'MEP Id', 'Mac Address', 'Interface', 'VLAN',
                   'Last Received' ]

      fl = TableOutput.Format( justify='left' )
      fl.padLimitIs( True )

      def printHeaderAndCreateTable( domain, maName ):
         if domain is not None:
            print hdr, domain
         else:
            print hdr
         if maName is not None:
            print hdrMa, maName
         else:
            print hdrMa
         print hdrLine
         table = TableOutput.createTable( hdrTable )
         table.formatColumns( fl, fl, fl, fl, fl )
         return table

      printed = False
      if ccmDb and ccmDb.ccmMd:
         for mdLevel, ccmMd in sorted( ccmDb.ccmMd.items() ):
            if domain is None or domain == mdLevel:
               for maName, ccmMa in sorted( ccmMd.ccmMa.items() ):
                  table = printHeaderAndCreateTable( mdLevel, maName )
                  for mepId, info in sorted( ccmMa.ccmMepInfo.items() ):
                     table.newRow( mepId,
                         Ethernet.convertMacAddrCanonicalToDisplay( info.srcMac ),
                         IntfCli.Intf.getShortname( info.intf ),
                         info.vlanId, timestampToStr( info.lastCcmReceivedTime ) )
                  print table.output()
                  printed = True
      if not printed:
         table = printHeaderAndCreateTable( domain, None )
         print table.output()

BasicCli.addShowCommandClass( CfmContinuityCheckDatabaseCmd )

#--------------------------------------------------------------------------------
# clear cfm continuity-check database
#--------------------------------------------------------------------------------
class ClearCcmDbCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cfm continuity-check database'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfm,
      'continuity-check' : 'Continuity Check information',
      'database' : 'Database entries',
   }

   @staticmethod
   def handler( mode, args ):
      if not waitForAgent():
         return
      command = 'clear continuity-check database'
      AgentCommandRequest.runSocketCommand( mode.entityManager, CfmAgent.name,
                                            "CfmCliCallback",
                                            command, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearCcmDbCmd )

#--------------------------------------------------------------------------------
# clear cfm continuity-check intermediate-point database
#--------------------------------------------------------------------------------
class ClearMipCcmDbCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cfm continuity-check intermediate-point database'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfm,
      'continuity-check' : 'Continuity Check information',
      'intermediate-point' : CliCommand.guardedKeyword( 'intermediate-point',
         helpdesc='Maintenance Intermediate Point information',
         guard=defaultMipSupported ),
      'database' : 'Database entries',
   }

   @staticmethod
   def handler( mode, args ):
      if not waitForAgent():
         return
      command = 'clear intermediate-point database'
      AgentCommandRequest.runSocketCommand( mode.entityManager, CfmAgent.name,
                                            "CfmCliCallback",
                                            command, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearMipCcmDbCmd )

#--------------------------------------------------------------------------------
# show cfm intermediate-point
#--------------------------------------------------------------------------------
class CfmIntermediatePointCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm intermediate-point'
   data = {
      'cfm' : nodeCfm,
      'intermediate-point' : nodeIntermediatePoint,
   }
   cliModel = CfmMaintenanceDomain

   @staticmethod
   def handler( mode, args ):
      cfmMaintenanceDomain = CfmMaintenanceDomain()
      for ( k, v ) in cfmStatus.mdStatus.items():
         cfmMaintenanceDomain.intermediatePoint[ k ] = v.defaultMip
      return cfmMaintenanceDomain

BasicCli.addShowCommandClass( CfmIntermediatePointCmd )

#-----------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-----------------------------------------------------------
def Plugin( entityManager ):
   global cfmConfig
   global cfmStatus
   global aleCfmConfig
   global aleCfmStatus
   global cfmHwSupportStatus
   global trapConfig

   cfmConfig = ConfigMount.mount( entityManager, "cfm/config",
                                  "CfmAgent::Config", "w" )
   trapConfig = ConfigMount.mount( entityManager, "hardware/trap/config/trapConfig",
                                   "Arnet::TrapConfig", 'w' )
   cfmStatus = LazyMount.mount( entityManager, "cfm/status",
                                   "CfmAgent::Status", "r" )
   cfmHwSupportStatus = LazyMount.mount( entityManager, "cfm/hwSupportStatus",
                                         "Cfm::HwSupportStatus", "r" )
   aleCfmConfig = LazyMount.mount( entityManager, "cfm/aleConfig",
                                   "Cfm::AleConfig", "r" )
   aleCfmStatus = LazyMount.mount( entityManager, "cfm/aleStatus",
                                   "Cfm::AleStatus", "r" )
