# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import BasicCliUtil
import Cell
import CliCommand
import CliMatcher
import CliPlugin.Ssl as Ssl
import CliPlugin.Pki as Pki
import CliPlugin.Security as Security
import CliPlugin.CertRotationModel as CertRotationModel
import CliPlugin.RotationMgr as RotationMgr
import CliToken.Clear
import CommonGuards
import LazyMount
import ShowCommand
import SslCertKey
import Tac

Constants = Tac.Type( 'Mgmt::Security::Ssl::Constants' )

rotationMgr = None
sslConfig = None
redundancyStatus = None
rotationNotReadyError = 'certificate rotation not ready'

rotationNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'rotation', helpdesc='Rotation command' ),
      guard=CommonGuards.standbyGuard )
rotationIdMatcher = CliMatcher.PatternMatcher( '[a-f0-9]+',
      helpname='UUID',
      helpdesc='Unique rotation ID' )

#------------------------------------------------------------------------------------
# security pki certificate generate signing-request
#      rotation ssl profile <name> key generate rsa <2048|3072|4096>
#      [ import-timeout <minutes> ]
#      [ digest <sha256|sha384|sha512> ]
#      parameters common-name <common-name>
#                             [ country <country-code> ]
#                             [ state <state-name> ]
#                             [ locality <locality-name> ]
#                             [ organization <org-name> ]
#                             [ organization-unit <org-unit-name> ]
#                             [ email <email> ]
#                             [ subject-alternative-name [ ip <ip1 ip2 ..> ]
#                                                        [ dns <name1 name2 ..> ]
#                                                        [ email <em1 em2 ..> ] ] ]
#------------------------------------------------------------------------------------
class PkiCertGenerateSigningRequest( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'security pki certificate generate signing-request '
              'rotation ssl profile PROFILE_NAME key generate rsa RSA_BIT_LENGTH '
              '[ import-timeout IMPORT_TIMEOUT ] [ digest DIGEST ] '
              '[ SIGN_REQ_PARAMS ]' )
   data = { 'security': Security.securityKwMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'generate': Pki.generateNode,
            'signing-request': Pki.signRequestKwMatcher,
            'rotation': rotationNode,
            'ssl': Ssl.sslMatcher,
            'profile': Ssl.profileMatcher,
            'PROFILE_NAME': Ssl.profileNameMatcher,
            'key': Pki.keyKwMatcher,
            'rsa': Pki.rsaMatcher,
            'RSA_BIT_LENGTH': Pki.rsaKeySizeMatcher,
            'import-timeout': ( 'Certificate import timeout in minutes after '
                                'which rotation expires' ),
            'IMPORT_TIMEOUT': CliMatcher.IntegerMatcher(
               Constants.minImportTimeoutMins, Constants.maxImportTimeoutMins,
               helpdesc='Minutes' ),
            'digest': Pki.digestKwMatcher,
            'DIGEST': Pki.digestMatcher,
            'SIGN_REQ_PARAMS': Pki.ParamExpression
          }
   cliModel = CertRotationModel.RotationGenerateSignRequestStatus
   privileged = True
   noMore = True

   @staticmethod
   def handler( mode, args ):
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None

      profileName = args[ 'PROFILE_NAME' ]
      newKeyBits = args[ 'RSA_BIT_LENGTH' ]
      importTimeout = args.get( 'IMPORT_TIMEOUT', Constants.defImportTimeoutMins )
      digest = args.get( 'DIGEST', Constants.defaultDigest )
      signReqParams = args.get( 'SIGN_REQ_PARAMS' )
      try:
         signReqParams = Pki.getSignRequestParams( mode, signReqParams )
      except SslCertKey.SslCertKeyError as e:
         mode.addError( 'Error generating CSR (%s)' % ( str( e ) ) )
         return None

      return rotationMgr.generateCsr( profileName, importTimeout,
                                      digest, signReqParams,
                                      newKeyBits=newKeyBits )

BasicCli.addShowCommandClass( PkiCertGenerateSigningRequest )

#------------------------------------------------------------------------------------
# security pki certificate rotation import ROTATION_ID
#------------------------------------------------------------------------------------
class PkiCertificateRotationImport( ShowCommand.ShowCliCommandClass ):
   syntax = 'security pki certificate rotation import ROTATION_ID'
   data = { 'security': Security.securityKwMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'rotation': rotationNode,
            'import': 'Import certificate for rotation ID',
            'ROTATION_ID': rotationIdMatcher,
          }
   cliModel = CertRotationModel.RotationImportStatus
   privileged = True
   noMore = True

   @staticmethod
   def handler( mode, args ):
      rotationId = args[ 'ROTATION_ID' ]
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None

      cmd = 'security pki certificate rotation import UUID'
      certPem = BasicCliUtil.getMultiLineInput( mode, cmd,
                                                prompt='Enter TEXT certificate' )
      return rotationMgr.importCertificate( rotationId, certPem )

BasicCli.addShowCommandClass( PkiCertificateRotationImport )
                              
#------------------------------------------------------------------------------------
# security pki certificate rotation commit <rotation-id>
#------------------------------------------------------------------------------------
class PkiCertificateRotationCommit( ShowCommand.ShowCliCommandClass ):
   syntax = 'security pki certificate rotation commit ROTATION_ID'
   data = { 'security': Security.securityKwMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'rotation': rotationNode,
            'commit': 'Commit certificate and key for rotation',
            'ROTATION_ID': rotationIdMatcher,
          }
   cliModel = CertRotationModel.RotationCommitStatus
   privileged = True
   noMore = True

   @staticmethod
   def handler( mode, args ):
      rotationId = args[ 'ROTATION_ID' ]
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None

      return rotationMgr.commit( rotationId )

BasicCli.addShowCommandClass( PkiCertificateRotationCommit )

#------------------------------------------------------------------------------------
# security pki certificate rotation commit ssl profile PROFILE_NAME [ dont-wait ]
#          [ ignore validity ]
#------------------------------------------------------------------------------------
class PkiCertificateRotationCommitSslProfile( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'security pki certificate rotation commit ssl profile PROFILE_NAME '
                                                '[ dont-wait ] [ ignore validity ]' )
   data = { 'security': Security.securityKwMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'rotation': rotationNode,
            'commit': 'Commit certificate and key for rotation',
            'ssl': Ssl.sslMatcher,
            'profile': Ssl.profileMatcher,
            'PROFILE_NAME': Ssl.profileNameMatcher,
            'dont-wait': 'Do not wait for ssl services to load new certificate',
            'ignore': 'Ignore checks during commit',
            'validity': Pki.validityMatcher
          }
   cliModel = CertRotationModel.RotationCommitSslProfileStatus
   privileged = True
   noMore = True

   @staticmethod
   def handler( mode, args ):
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None

      profileName = args.get( 'PROFILE_NAME' )
      dontWait = 'dont-wait' in args
      validateDates = 'ignore' not in args
      cmd = 'security pki certificate rotation commit ssl profile WORD'
      key = BasicCliUtil.getMultiLineInput( mode, cmd,
                                            prompt='Enter TEXT private key' )
      cert = BasicCliUtil.getMultiLineInput( mode, cmd,
                                             prompt='Enter TEXT certificate' )
      return rotationMgr.commitSslProfile( profileName, key, cert, validateDates,
                                           dontWait )

BasicCli.addShowCommandClass( PkiCertificateRotationCommitSslProfile )
      
#------------------------------------------------------------------------------------
# clear security pki certificate rotation ROTATION_ID
#------------------------------------------------------------------------------------
class PkiCertificateRotationClear( ShowCommand.ShowCliCommandClass ):
   syntax = 'clear security pki certificate rotation ROTATION_ID'
   data = {
            'clear': CliToken.Clear.clearKwNode,
            'security': Security.securityKwMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'rotation': rotationNode,
            'ROTATION_ID': rotationIdMatcher,
          }
   cliModel = CertRotationModel.RotationClearStatus
   privileged = True
   noMore = True

   @staticmethod
   def handler( mode, args ):
      rotationId = args[ 'ROTATION_ID' ]
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None

      return rotationMgr.clear( rotationId )

BasicCli.addShowCommandClass( PkiCertificateRotationClear )

#------------------------------------------------------------------------------------
# show security pki certificate rotation [ detail ]
#------------------------------------------------------------------------------------
class ShowPkiCertRotations( ShowCommand.ShowCliCommandClass ):
   syntax = 'show security pki certificate rotation [ detail ]'
   data = { 'security': Security.securityShowMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'rotation': rotationNode,
            'detail': ' Show additional rotation information' }
   cliModel = CertRotationModel.CertificateRotations

   @staticmethod
   def handler( mode, args ):
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None
      detail = 'detail' in args
      result = CertRotationModel.CertificateRotations( _detail=detail )
      for rotationId, rr in rotationMgr.getAllRotationReqs().iteritems():
         rotationModel = rr.getCliRotationModel()
         if rotationModel is None:
            continue
         result.rotations[ rotationId ] = rotationModel
      return result

BasicCli.addShowCommandClass( ShowPkiCertRotations )

#------------------------------------------------------------------------------------
# show security pki certificate rotation ROTATION_ID
#------------------------------------------------------------------------------------
class ShowPkiCertRotation( ShowCommand.ShowCliCommandClass ):
   syntax = 'show security pki certificate rotation ROTATION_ID'
   data = { 'security': Security.securityShowMatcher,
            'pki': Pki.pkiEnKwMatcher,
            'certificate': Pki.certKwMatcher,
            'rotation': rotationNode,
            'ROTATION_ID': rotationIdMatcher }
   cliModel = CertRotationModel.CertificateRotation

   @staticmethod
   def handler( mode, args ):
      if not rotationMgr.isReady():
         mode.addError( rotationNotReadyError )
         return None

      rotationReq = rotationMgr.getRotationReq( args[ 'ROTATION_ID' ] )
      if rotationReq is None:
         mode.addError( 'Rotation %s not found' % args[ 'ROTATION_ID' ] )
         return None

      result = rotationReq.getCliRotationModel()
      if result is None:
         mode.addError( 'Rotation %s not found' % args[ 'ROTATION_ID' ] )
      return result

BasicCli.addShowCommandClass( ShowPkiCertRotation )

# Called by tests only
def getRotationMgr():
   return rotationMgr

def Plugin( entityManager ):
   global sslConfig
   global redundancyStatus
   
   sslConfig = LazyMount.mount( entityManager, 'mgmt/security/ssl/config',
                                'Mgmt::Security::Ssl::Config', 'r' )
   sslServiceStatus = LazyMount.mount( entityManager, 
                                       'mgmt/security/ssl/serviceStatus',
                                       'Tac::Dir', 'ri' )
   mg = entityManager.mountGroup()
   redundancyStatus = mg.mount( Cell.path( 'redundancy/status' ),
                                'Redundancy::RedundancyStatus', 'r' )
   
   def _mountsComplete():
      global rotationMgr      
      rotationMgr = RotationMgr.RotationMgr( Constants.rotationBaseDirPath(),
                                             redundancyStatus, sslConfig,
                                             sslServiceStatus )
   mg.close( _mountsComplete )
