#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool
from CliModel import Dict
from CliModel import Float
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliModel import Str
from CliModel import Enum
from HttpServiceModel import HttpProtocolStatus
from HttpServiceModel import UnixProtocolStatus
from HttpServiceModel import SslProfile
from HttpServiceModel import printLineItem
from HttpServiceModel import LogLevel
from HttpServiceConstants import CapiCipherSuites
import TableOutput
import Tac

class CommandApiCertificate( Model ):
   __revision__ = 2
   
   certificate = Str( help="Certificate used by SSL/TLS" )

   def degrade( self, dictRepr, revision ):
      # This model is not really degradable
      # for old version, instead of get the ssl private key, we send empty string
      if revision == 1:
         dictRepr[ 'key' ] = 'SENSITIVE INFO:<REMOVED>'
      return dictRepr

   def render( self ):
      print 'Certificate:'
      print self.certificate

class UserStatistics( Model ):
   requestCount = Int( "Number of requests received from the user" )
   bytesIn = Int( help="The number of bytes received from the user's requests" )
   bytesOut = Int(  help="The number of bytes sent in responses to the user" )
   lastHitTime = Float( help="The UTC timestamp of the user's last request" )
   # would be cool to have the IP Addr of the requester here as well.

class CommandApiStatus( Model ):
   # Top level attributes:
   enabled = Bool( help="Whether the server is globally enabled" )
   httpServer = Submodel( valueType=HttpProtocolStatus,
                          help="The status of the HTTP server" )
   localHttpServer = Submodel( valueType=HttpProtocolStatus, optional=True,
                               help="The status of the localhost HTTP server" )
   httpsServer = Submodel( valueType=HttpProtocolStatus,
                          help="The status of the HTTPS server" )
   unixSocketServer = Submodel( valueType=UnixProtocolStatus, optional=True,
                                help="The status of the Unix Domain Socket server" )
   vrf = Str( help="The VRF the API is configured to serve in" )
   vrfs = List( valueType=str, help="All VRFs enabled in CAPI" )
   hitCount = Int( help="The number of incoming HTTP requests the server has seen" )
   requestCount = Int( help="The number of valid JSON-RPC requests received" )
   commandCount = Int( help="The number of CLI commands requested to run" )

   bytesIn = Int( help="The number of bytes received from HTTP requests" )
   bytesOut = Int(  help="The number of bytes sent in HTTP responses" )
   lastHitTime = Float( optional=True, help="The UTC timestamp of the last request" )
   executionTime = Float( help="The time in seconds spent executing CLI commands" )
   
   corsOrigins = List( valueType=str, optional=True,
                       help="Allowed CORS Origins. Present when CORS is supported" )
   users = Dict( valueType=UserStatistics,
                 help="Mapping from username to statistics about that user" )
   dscpValue = Int( help="The QoS DSCP value set in outgoing IP packets", 
                    optional=True )
   logLevel = Enum( help="Logging level for the HTTP server",
                    values=LogLevel.attributes )
   iframeAncestors = List( valueType=str, help="Allowed Iframe ancestors" )

   urls = List( valueType=str, help="URLs the server is listening on" )
   sslProfile = Submodel( optional=True, valueType=SslProfile,
                          help="The configuration and status of the SSL profile" )
   fipsEnabled = Bool( optional=True, help="Whether FIPS mode is enabled for "
                                           "HTTPS algorithms" )
   tlsProtocol = List( valueType=str, help="Allowed TLS protocol versions" )

   def render( self ):

      def statusString( capiEnabled, configured, running ):
         if configured and capiEnabled:
            return 'running' if running else 'starting'
         if configured and not capiEnabled:
            return 'shutting down' if running else 'enabled'
         else:
            return 'shutting down' if running else 'shutdown'

      def formatLastTime( lastHitTime ):
         if lastHitTime:
            return '%d seconds ago' % ( Tac.utcNow() - lastHitTime )
         else:
            return 'never'


      printLineItem( "Enabled", "Yes" if self.enabled else "No" )
      printLineItem( "HTTPS server", "%s, set to use port %d" % (
            statusString( self.enabled, self.httpsServer.configured,
                          self.httpsServer.running ),
            self.httpsServer.port ) )
      printLineItem( "HTTP server", "%s, set to use port %d" % (
            statusString( self.enabled, self.httpServer.configured,
                          self.httpServer.running ),
            self.httpServer.port ) )
      printLineItem( "Local HTTP server", "%s, no authentication, "
                     "set to use port %d" % (
            statusString( self.enabled, self.localHttpServer.configured,
                          self.localHttpServer.running ),
            self.localHttpServer.port ) )
      printLineItem( "Unix Socket server", "%s, no authentication" % (
            statusString( self.enabled, self.unixSocketServer.configured,
                          self.unixSocketServer.running ) ) )
   
      enabledVrfs = ','.join( self.vrfs ) if self.vrfs else 'None'
      printLineItem( 'VRFs', enabledVrfs )
      printLineItem( 'Hits', self.hitCount )
      printLineItem( 'Last hit', formatLastTime( self.lastHitTime ) )
      printLineItem( 'Bytes in', self.bytesIn )
      printLineItem( 'Bytes out', self.bytesOut )
      printLineItem( 'Requests', self.requestCount )
      printLineItem( 'Commands', self.commandCount )
      printLineItem( 'Duration', '%.3f seconds' % self.executionTime )

      if self.sslProfile:
         state = 'not configured'
         if self.sslProfile.state:
            state = self.sslProfile.state
         printLineItem( 'SSL Profile', '%s, %s' % ( self.sslProfile.name, state ) )
      else:
         printLineItem( 'SSL Profile', '%s' % 'none' )
      printLineItem( "FIPS Mode", "Yes" if self.fipsEnabled else "No" )

      printLineItem( 'QoS DSCP', '%s' % self.dscpValue )
      printLineItem( 'Log Level', '%s' % self.logLevel )
      iframeAncestors = ( ' '.join( self.iframeAncestors ) if self.iframeAncestors 
                          else 'None' )
      printLineItem( 'CSP Frame Ancestor', '%s' % iframeAncestors )

      printLineItem( 'TLS Protocols', '%s' % ' '.join( self.tlsProtocol ) )

      if self.users:
         tableHeadings = ( "User", "Requests", "Bytes in", "Bytes out",
                           "Last hit" )
         table = TableOutput.createTable( tableHeadings )
         f = TableOutput.Format( justify="left" )
         table.formatColumns( *( [ f ] * len( tableHeadings ) ) )
         for name, userStats in self.users.iteritems():
            table.newRow( name,
                          userStats.requestCount,
                          userStats.bytesIn,
                          userStats.bytesOut,
                          formatLastTime( userStats.lastHitTime ) )
         print table.output()

      if self.urls:
         tableHeadings = ( "URLs", )
         table = TableOutput.createTable( tableHeadings )
         f = TableOutput.Format( justify="left" )
         table.formatColumns( *( [ f ] * len( tableHeadings ) ) )
         for url in self.urls:
            table.newRow( url )
         print table.output()
        
      if self.corsOrigins:
         tableHeadings = ( "CORS Enabled Origins", )
         table = TableOutput.createTable( tableHeadings )
         f = TableOutput.Format( justify="left" )
         table.formatColumns( *( [ f ] * len( tableHeadings ) ) )
         for origin in self.corsOrigins:
            table.newRow( origin )
         print table.output()

class CipherSuite( Model ):
   cipherSuite = Enum( help='CipherSuite Name',
                       values=CapiCipherSuites.cipherSuiteFilterToNameMap.values() )
      
class CommandApiCipherSuites( Model ):
   enabledCipherSuites = List( valueType=CipherSuite, help="Enabled CipherSuites" )

   def render( self ):
      for cipher in self.enabledCipherSuites:
         print cipher.cipherSuite

class CommandApiDiffieHellman( Model ):
   diffieHellmanParameters = Str(
         help="Parameters used by SSL/TLS for diffie-hellman key exchange" )

   def render( self ):
      print self.diffieHellmanParameters
