# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno

import Arnet
import AclCli
import AclCliModel
import BasicCli
import BasicCliUtil
import CapiConstants
import CapiModel
from CliMode.Capi import CapiVrfConfigModeBase
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.HttpService as HttpService
import CliPlugin.VrfCli as VrfCli
import CliToken
import CliToken.Reset
import ConfigMgmtMode
import ConfigMount
from HttpServiceConstants import CapiCipherSuites
import HttpServiceModel
import LazyMount
import ShowCommand
import Tac
import TechSupportCli

capiConfig = None
capiExecRequest = None
serverStatus = None
serviceStatusDir = None
ipStatus = None
ip6Status = None
sslStatus = None
cliConfig = None
aclCheckpoint = None

DEFAULT_VRF = VrfCli.DEFAULT_VRF
apiShowHelpDesc = 'Show management APIs'
httpShowCommandsHelpDesc = 'Show details about the HTTP Command API'
httpsShowHelpDesc = 'Show HTTPS configuration details'
httpCommandsHelpdesc = 'Configure the HTTP Commands API'

def readFile( path ):
   """ Return the contents of the path, or the empty string if it doesn't exist """
   try:
      with open( path ) as f:
         return f.read()
   except IOError as e:
      if e.errno != errno.ENOENT:
         raise
   return ""

#
# Management config mode for Capi.
#
class CapiConfigMode( ConfigMgmtMode.ConfigMgmtMode ):

   name = "Capi configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent,
                                              session, "api-http-cmds" )
      self.config_ = capiConfig

   def enterCmd( self ):
      return "management api http-commands"

#-------------------------------------------------------------------------------
# The "[no|default] management api http-commands" mode command.
#-------------------------------------------------------------------------------
class EnterCapiMode( CliCommand.CliCommandClass ):
   syntax = """management api http-commands"""
   noOrDefaultSyntax = syntax

   data = { "management": ConfigMgmtMode.managementKwMatcher,
            "api": ConfigMgmtMode.apiKwMatcher,
            "http-commands": httpCommandsHelpdesc }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CapiConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      shutdownCapi( mode )
      capiNoVrf( mode, {} )
      capiClearHttpsCert( mode )
      capiClearHttpsKeyExchange( mode )
      capiClearHttpsCipher( mode )
      capiClearHttpsMac( mode )
      if not capiConfig.useHttpServiceCli:
         HttpService.handleDefaultProtocol( "http" )
         HttpService.handleDefaultProtocol( "https" )
         HttpService.handleDefaultProtocol( "http localhost")
         HttpService.handleDefaultProtocol( "unix-socket" )
         capiConfig.corsAllowedOrigins.clear()
         capiConfig.httpsConfig.sslProfile = ''
         capiConfig.qosDscp = 0
         capiConfig.contentFrameAncestor = ""
         capiConfig.defaultServicesEnabled = True
         capiConfig.syslogLevel = capiConfig.syslogLevelDefault
      if not HttpService.serverConfigurationExists():
         capiConfig.useHttpServiceCli = False

BasicCli.GlobalConfigMode.addCommandClass( EnterCapiMode )

#-------------------------------------------------------------------------------
# The "[no | default] shutdown" command,
# in "vrf <vrf name>" mode
# under "management api http-commands" mode.
#-------------------------------------------------------------------------------
class CapiVrfConfigMode ( CapiVrfConfigModeBase, BasicCli.ConfigModeBase ):
   name = "Capi VRF configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      CapiVrfConfigModeBase.__init__( self, ( vrfName, capiConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def shutdown( self ):
      HttpService.removeVrfFromService( self.config_, 
                                        'http-commands', self.vrfName_ )
   
   def noShutdown( self ):
      self.config().vrfService[ 'http-commands' ] = True

class VrfModeShutdown( CliCommand.CliCommandClass ):
   syntax = '''shutdown'''
   noOrDefaultSyntax = '''shutdown ...'''
   data = { 'shutdown': HttpService.vrfShutdownHelpdesc }

   @staticmethod
   def handler( mode, args ):
      mode.shutdown()

   @staticmethod
   def noHandler( mode, args ):
      mode.noShutdown()

   defaultHandler = handler

CapiVrfConfigMode.addCommandClass( VrfModeShutdown )

# {no|default} vrf VRF in manamgent api http-commands mode
def enterCapiVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( CapiVrfConfigMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def capiNoVrf( mode, args ):
   serviceAclTypeVrfMap = capiConfig.serviceAclTypeVrfMap
   vrfName = args.get( 'VRF' )
   if vrfName:
      for aclType in [ 'ip', 'ipv6' ]:
         key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrfName )
         if key in serviceAclTypeVrfMap.aclName:
            del serviceAclTypeVrfMap.aclName[ key ]
   else:
      for key in serviceAclTypeVrfMap.aclName:
         del serviceAclTypeVrfMap.aclName[ key ]

   HttpService.removeVrfFromService( capiConfig, 'http-commands', vrfName )

class EnterVrfMode( CliCommand.CliCommandClass ):
   syntax = '''vrf VRF'''
   noOrDefaultSyntax = '''vrf [ VRF ]'''
   data = { 'vrf': HttpService.vrfHelpdesc,
            'VRF': VrfCli.VrfNameExprFactory( inclAllVrf=True ) }
   handler = enterCapiVrfConfigMode
   noOrDefaultHandler = capiNoVrf

CapiConfigMode.addCommandClass( EnterVrfMode )

#-------------------------------------------------------------------------------
# The "[no|default] shutdown" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
def shutdownCapi( mode ):
   capiConfig.service[ 'http-commands' ].enabled = False

class ConfigureCapi( CliCommand.CliCommandClass ):
   syntax = """shutdown"""
   noOrDefaultSyntax = syntax

   data = { "shutdown": "Disable API access" }

   @staticmethod
   def handler( mode, args ):
      shutdownCapi( mode )

   @staticmethod
   def noHandler( mode, args ):
      if ( not capiConfig.httpsConfig.enabled and
           not capiConfig.httpConfig.enabled and
           not capiConfig.localHttpConfig.enabled and
           not capiConfig.unixConfig.enabled ):
         # No protocol is enabled
         mode.addWarning( "No protocols are enabled" )
      capiConfig.service[ 'http-commands' ].enabled = True

   defaultHandler = handler # by default we are off

CapiConfigMode.addCommandClass( ConfigureCapi )

#-------------------------------------------------------------------------------
# The "[no|default] protocol ((http [localhost])|https) [port <number>]" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.HttpProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol unix-socket" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.UnixProtocolCmd )

#-------------------------------------------------------------------------------
# The "[no | default] protocol https certificate" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
def capiClearHttpsCert( mode ):
   _updateHttpsCert( mode, '', '' )

def _updateHttpsCert( mode, certificate, key ):
   """ Updates the certificate and private key. An error is added to the mode
   if there is an error """
   import HttpServiceSsl
   if certificate and key:
      error = HttpServiceSsl.validateCertificateAndKey( certificate, key )
      if error:
         mode.addError( error )
         return

      capiConfig.httpsConfig.sslKey = key
      capiConfig.httpsConfig.sslCertificate = certificate
   elif not certificate and not key:
      capiConfig.httpsConfig.sslKey = ''
      capiConfig.httpsConfig.sslCertificate = ''
   else:
      mode.addError( "Both a Certificate and Private Key must be configured" )

class CapiHttpsCert( CliCommand.CliCommandClass ):
   syntax = """protocol https certificate"""
   noOrDefaultSyntax = syntax

   data = { "protocol": HttpService.protocolHelpdesc,
            "https": HttpService.httpsHelpdesc,
            "certificate": 'Set the HTTPS key and certificate to use' }

   @staticmethod
   def handler( mode, args ):
      if not HttpService.checkServerConfigNotSplit( mode ):
         return
      cmd = CapiHttpsCert.syntax
      certificate = BasicCliUtil.getMultiLineInput( mode, cmd, lstrip=True,
                                                prompt="Enter TEXT certificate" )
      key = BasicCliUtil.getMultiLineInput( mode, cmd, lstrip=True,
                                        prompt="\nEnter TEXT private key" )
      _updateHttpsCert( mode, certificate, key )
      HttpService.setUseHttpServiceCli( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      capiClearHttpsCert( mode )
      HttpService.setUseHttpServiceCli( mode )

CapiConfigMode.addCommandClass( CapiHttpsCert )

#-------------------------------------------------------------------------------
# The "[no | default] protocol https ssl profile" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.HttpsProfile )

#-------------------------------------------------------------------------------
# The "[no|default] protocol https cipher" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
# The union of acceptable cipher names (with their descriptions)
cipherData = { k: v[ 'helpdesc' ] for k, v
               in CapiCipherSuites.CIPHERS.iteritems() }

def capiClearHttpsCipher( mode ):
   capiConfig.httpsConfig.ciphers.clear()
   capiConfig.httpsConfig.hasCipherConfig = False

class ProtocolHttpsCipher( CliCommand.CliCommandClass ):
   syntax = '''protocol https cipher <cipher>'''
   noOrDefaultSyntax = '''protocol https cipher ...'''
   data = { 'protocol': HttpService.protocolHelpdesc,
            'https': HttpService.httpsHelpdesc,
            'cipher': 'Exclusive list of cryptographic ciphers',
            '<cipher>': CliCommand.setCliExpression( cipherData ) }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if not HttpService.checkServerConfigNotSplit( mode ):
         return
      capiConfig.httpsConfig.ciphers.clear()
      isDefault = True
      ciphers = [ key for key in CapiCipherSuites.CIPHERS if key in args ]
      if ciphers:
         capiConfig.httpsConfig.hasCipherConfig = True
      for cipher in CapiCipherSuites.CIPHERS:
         isDefault &= cipher in ciphers

      if not isDefault:
         for c in ciphers:
            capiConfig.httpsConfig.ciphers.enq( c )
      HttpService.setUseHttpServiceCli( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      capiClearHttpsCipher( mode )
      HttpService.setUseHttpServiceCli( mode )

CapiConfigMode.addCommandClass( ProtocolHttpsCipher )

#-------------------------------------------------------------------------------
# The "[no|default] protocol https key-exchange" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
# The union of acceptable key exchange algorithms (with their descriptions)
keyExchangeData = { k: v[ 'helpdesc' ] for k, v
                    in CapiCipherSuites.KEYEXCHANGE.iteritems() }

def capiClearHttpsKeyExchange( mode ):
   capiConfig.httpsConfig.keyExchange.clear()
   capiConfig.httpsConfig.hasKeyExchangeConfig = False

class ProtocolHttpsKeyExchange( CliCommand.CliCommandClass ):
   syntax = '''protocol https key-exchange <keyExchange>'''
   noOrDefaultSyntax = '''protocol https key-exchange ...'''
   data = { 'protocol': HttpService.protocolHelpdesc,
            'https': HttpService.httpsHelpdesc,
            'key-exchange': 'Exclusive list of key-exchange algorithms',
            '<keyExchange>': CliCommand.setCliExpression( keyExchangeData ) }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if not HttpService.checkServerConfigNotSplit( mode ):
         return
      capiConfig.httpsConfig.keyExchange.clear()
      isDefault = True
      keyExchange = [ key for key in CapiCipherSuites.KEYEXCHANGE if key in args ]
      if keyExchange:
         capiConfig.httpsConfig.hasKeyExchangeConfig = True
      for key in CapiCipherSuites.KEYEXCHANGE:
         isDefault &= key in keyExchange

      if not isDefault:
         for k in keyExchange:
            capiConfig.httpsConfig.keyExchange.enq( k )
      HttpService.setUseHttpServiceCli( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      capiClearHttpsKeyExchange( mode )
      HttpService.setUseHttpServiceCli( mode )

CapiConfigMode.addCommandClass( ProtocolHttpsKeyExchange )

#-------------------------------------------------------------------------------
# The "[no|default] protocol https mac" command,
# in "management api http-commands" mode.
#-------------------------------------------------------------------------------
macData = { k: v[ 'helpdesc' ] for k, v in CapiCipherSuites.MAC.iteritems() }

def capiClearHttpsMac( mode ):
   capiConfig.httpsConfig.mac.clear()
   capiConfig.httpsConfig.hasMacConfig = False

class ProtocolHttpsMac( CliCommand.CliCommandClass ):
   syntax = '''protocol https mac <mac>'''
   noOrDefaultSyntax = '''protocol https mac ...'''
   data = { 'protocol': HttpService.protocolHelpdesc,
            'https': HttpService.httpsHelpdesc,
            'mac': 'Exclusive list of Mac algorithms',
            '<mac>': CliCommand.setCliExpression( macData ) }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if not HttpService.checkServerConfigNotSplit( mode ):
         return
      capiConfig.httpsConfig.mac.clear()
      isDefault = True

      mac = [ key for key in CapiCipherSuites.MAC if key in args ]
      if mac:
         capiConfig.httpsConfig.hasMacConfig = True
      for m in CapiCipherSuites.MAC:
         isDefault &= m in mac

      if not isDefault:
         for m in mac:
            capiConfig.httpsConfig.mac.enq( m )
      HttpService.setUseHttpServiceCli( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      capiClearHttpsMac( mode )
      HttpService.setUseHttpServiceCli( mode )

CapiConfigMode.addCommandClass( ProtocolHttpsMac )

#-------------------------------------------------------------------------------
# (config-mgmt-api-http-cmds)# [no] validate-output.
# 
# This enables time-consuming output checks, which is ok in test environment.
# Output validation is useful for commands that print json directly (without
# going through a python model instance, see CliPrint). It will unmarshall the
# json back into a python model so that any error becomes visible (this only
# happens via http, "show blah | json" will not validate anything.
# "Art sanitize" will enable validation (it is off by default).
#-------------------------------------------------------------------------------
validateOutputMatcher = CliMatcher.KeywordMatcher( 'validate-output',
                           helpdesc='validate that json output conforms to model' )
class ValidateJsonOutput( CliCommand.CliCommandClass ):
   syntax = """validate-output"""
   noOrDefaultSyntax = syntax

   data = { "validate-output": CliCommand.Node( matcher=validateOutputMatcher,
                                                hidden=True ) }

   @staticmethod
   def handler( mode, args ):
      cliConfig.validateOutput = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.validateOutput = False

CapiConfigMode.addCommandClass( ValidateJsonOutput )

#-------------------------------------------------------------------------------
# The "reset https diffie-hellman parameters" command in "config" mode
#-------------------------------------------------------------------------------
class ResetDhParams( CliCommand.CliCommandClass ):
   syntax = '''reset https diffie-hellman parameters'''
   data = { 'reset': CliToken.Reset.resetKwApi,
            'https': HttpService.httpsHelpdesc,
            'diffie-hellman': 'diffie-hellman parameters',
            'parameters': 'diffie-hellman parameters' }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.addError( "Please use 'reset ssl diffie-hellman parameters' instead" )

BasicCli.EnableMode.addCommandClass( ResetDhParams )

#-------------------------------------------------------------------------------
# The "clear management api http-commands" command,
# in "enable" mode.
#-------------------------------------------------------------------------------
class ClearCounters( CliCommand.CliCommandClass ):
   syntax = '''clear management api http-commands counters'''
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'api': ConfigMgmtMode.apiKwMatcher,
            'http-commands': httpCommandsHelpdesc,
            'counters': 'Clear counters tracking API usage' }

   @staticmethod
   def handler( mode, args ):
      capiExecRequest.lastStatsResetTime = Tac.now()

BasicCli.EnableMode.addCommandClass( ClearCounters )

#-------------------------------------------------------------------------------
# The "[ no | default ] qos dscp <dscpValue>" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.QosDscp )

#-------------------------------------------------------------------------------
# The "[ no | default ] log-level <severity>" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.NginxSyslog )

#-------------------------------------------------------------------------------
# The "[ no | default ] default-services" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.DefaultServices )

#-------------------------------------------------------------------------------
# The "[ no | default ] content frame ancestors <uri>" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.FrameAncestors )

#-------------------------------------------------------------------------------
# The "cors allowed domian [ALL | URL]" command
#-------------------------------------------------------------------------------
CapiConfigMode.addCommandClass( HttpService.CorsOrigins )

#-------------------------------------------------------------------------------
# The "show management api http-commands" command
#-------------------------------------------------------------------------------
def checkUdsEnabledOnly( vrf, vrfStatus ):
   return ( vrf == DEFAULT_VRF and
            # default VRF is not configured specifically
            vrf not in capiConfig.vrfConfig and
            # there is no other protocol enabled other than uds
            not vrfStatus.httpsEnabled and
            not vrfStatus.httpEnabled and
            not vrfStatus.httpLocalEnabled and
            vrfStatus.unixEnabled )

def showCapi( mode, args ):
   vrfList = []
   httpsEnabled = False
   httpEnabled = False
   httpLocalEnabled = False
   unixEnabled = False
   # Capi is enabled if it's enabled in any VRF
   for ( vrf, status ) in serverStatus.vrfStatus.items():
      if 'http-commands' in status.vrfService:
         for service, error in status.vrfError.items():
            if error and service in [ 'nginx', 'vrf' ]:
               # Capi only cares about error message about nginx status
               # and vrf status, not other services
               mode.addWarning( error )
         if ( status.enabled and 
              status.vrfService[ 'http-commands' ].enabled and
              # if Capi is running in default VRF with only uds
              # then hide it in show command
              not checkUdsEnabledOnly( vrf, status ) ):
            vrfList.append( vrf )
            
         httpsEnabled |= status.httpsEnabled
         httpEnabled |= status.httpEnabled
         httpLocalEnabled |= status.httpLocalEnabled
         unixEnabled |= status.unixEnabled

   ret = CapiModel.CommandApiStatus()

   ret.enabled = capiConfig.service[ 'http-commands' ].enabled 
   ret.httpsServer = HttpServiceModel.HttpProtocolStatus(
      configured=capiConfig.httpsConfig.enabled,
      running=httpsEnabled, port=capiConfig.httpsConfig.port )
   ret.httpServer = HttpServiceModel.HttpProtocolStatus(
      configured=capiConfig.httpConfig.enabled,
      running=httpEnabled, port=capiConfig.httpConfig.port )
   ret.localHttpServer = HttpServiceModel.HttpProtocolStatus(
      configured=capiConfig.localHttpConfig.enabled,
      running=httpLocalEnabled, port=capiConfig.localHttpConfig.port )
   ret.unixSocketServer = HttpServiceModel.UnixProtocolStatus(
      configured=capiConfig.unixConfig.enabled, running=unixEnabled )
   # show enabled VRFs only 
   vrfList = sorted( vrfList )
   ret.vrfs = vrfList
   ret.vrf = vrfList[ 0 ] if vrfList else ""
   ret.corsOrigins = capiConfig.corsAllowedOrigins.keys()

   capiStatus = serviceStatusDir.get( 'http-commands' )
   # if we have an outstanding clear request don't send back current values
   if ( capiStatus and
        capiStatus.lastStatsResetTime >= capiExecRequest.lastStatsResetTime ):
      ret.hitCount = capiStatus.hitCount
      ret.requestCount = capiStatus.requestCount
      ret.commandCount = capiStatus.commandCount
      ret.bytesIn = capiStatus.bytesInCount
      ret.bytesOut = capiStatus.bytesOutCount
      if capiStatus.lastHitTime > 1:
         ret.lastHitTime = capiStatus.lastHitTime + Tac.utcNow() - Tac.now()
      else:
         ret.lastHitTime = None
      ret.executionTime = capiStatus.requestDuration

      for name, userStatus in capiStatus.user.iteritems():
         if userStatus.lastHitTime > 1:
            lastHitTime = userStatus.lastHitTime + Tac.utcNow() - Tac.now()
         else:
            ret.lastHitTime = None
         ret.users[ name ] = CapiModel.UserStatistics(
            requestCount=userStatus.requestCount,
            bytesIn=userStatus.bytesInCount,
            bytesOut=userStatus.bytesOutCount,
            lastHitTime=lastHitTime )
   else:
      ret.hitCount = 0
      ret.requestCount = 0
      ret.commandCount = 0
      ret.bytesIn = 0
      ret.bytesOut = 0
      ret.lastHitTime = None
      ret.executionTime = 0.0

   ret.urls = getUrlList( vrfList, unixEnabled, httpLocalEnabled ) 
   HttpService.getCommonServerInfo( ret, capiConfig )
   return ret

def getUrlList( vrfList, unixEnabled, httpLocalEnabled ):
   def formatUrl( intfName, protocol, address, port ):
      return "{:<12s}: {:s}://{:s}:{:d}".format( intfName, protocol, address, port )

   urls = []
   internalIntfId = Tac.Value( 'Arnet::InternalIntfId' )

   for intfName in Arnet.sortIntf( ipStatus.ipIntfStatus ):
      ipIntfStatus = ipStatus.ipIntfStatus[ intfName ]
      # We don't want to list an interface with no ip address
      if ipIntfStatus.activeAddrWithMask.address == '0.0.0.0':
         continue

      # We don't want to display internal ports either
      if internalIntfId.isInternalIntfId( ipIntfStatus.intfId ):
         continue

      # don't display ip address if it's not in our VRF
      ipIntfStatusVrf = ipIntfStatus.vrf if ipIntfStatus.vrf else DEFAULT_VRF
      if ipIntfStatusVrf not in vrfList:   
         continue

      if capiConfig.httpsConfig.enabled:
         urls.append( formatUrl( intfName, 'https',
                                 ipIntfStatus.activeAddrWithMask.address,
                                 capiConfig.httpsConfig.port ) )
      if capiConfig.httpConfig.enabled:
         urls.append( formatUrl( intfName, 'http',
                                 ipIntfStatus.activeAddrWithMask.address,
                                 capiConfig.httpConfig.port ) )

   for intfName in Arnet.sortIntf( ip6Status.intf ):
      ip6IntfStatus = ip6Status.intf[ intfName ]
      # We don't want to display internal ports either
      if internalIntfId.isInternalIntfId( ip6IntfStatus.intfId ):
         continue

      # don't display ip address if it's not in our VRF
      ip6IntfStatusVrf = ip6IntfStatus.vrf if ip6IntfStatus.vrf else DEFAULT_VRF
      if ip6IntfStatusVrf not in vrfList:
         continue

      for key in ip6IntfStatus.addr:
         addr = ip6IntfStatus.addr[ key ].address
         if addr == ip6IntfStatus.linkLocalAddrWithMask().address:
            continue

         if capiConfig.httpsConfig.enabled:
            urls.append( formatUrl( intfName, 'https', "[%s]" % addr,
                                    capiConfig.httpsConfig.port ) )
         if capiConfig.httpConfig.enabled:
            urls.append( formatUrl( intfName, 'http', "[%s]" % addr,
                                    capiConfig.httpConfig.port ) )
   
   if capiConfig.unixConfig.enabled and unixEnabled:
      urls.append( "{:<12s}: unix:{:s}".format( "Unix Socket", 
                                                CapiConstants.SOCKET_ENDPOINT ) )
   if capiConfig.localHttpConfig.enabled and httpLocalEnabled:
      urls.append( "%s/%s" % ( formatUrl( "Local", "http", "localhost",
                                          capiConfig.localHttpConfig.port ),
                               CapiConstants.COMMAND_ENDPOINT ) )
   
   return urls

class ShowApiHttpCommands( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc
          }
   cliModel = CapiModel.CommandApiStatus
   handler = showCapi

BasicCli.addShowCommandClass( ShowApiHttpCommands )
#-------------------------------------------------------------------------------
# The "show management api http-commands https certificate" command and
#-------------------------------------------------------------------------------
def showCapiCert( mode, args ):
   import HttpServiceSsl
   profile = capiConfig.httpsConfig.sslProfile
   status = sslStatus.profileStatus.get( profile )
   certificate = ""

   # we don't have a valid capi cert
   if not capiConfig.httpsConfig.sslCertificate:
      # if we have an ssl profile configured
      if profile:
         # we have valid ssl profile
         if status and status.state == "valid" and status.certKeyPath:
            certKey = readFile( status.certKeyPath )
            ( certificate, _ ) = HttpServiceSsl.parseCertKey( certKey )
         else: # we have an invalid ssl profile
            certificate = ""
      else: # we don't have an ssl profile configured
         certificate = readFile( HttpServiceSsl.getCertFilepath() )

   else: # we have a valid capi cert, always use it!
      certificate = readFile( HttpServiceSsl.getCertFilepath() )

   return CapiModel.CommandApiCertificate( certificate=certificate )

class ShowApiHttpCommandsCertificate( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands https certificate'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'https': httpsShowHelpDesc,
            'certificate': 'Show the HTTPS certificate'
          }
   cliModel = CapiModel.CommandApiCertificate
   handler = showCapiCert
   privileged = True

BasicCli.addShowCommandClass( ShowApiHttpCommandsCertificate )
# -----------------------------------------------------------------------------
# The 'ip access-group <acl> [in]' command
# -----------------------------------------------------------------------------
def setServiceAcl( config, aclType, aclName, vrfName ):
   key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrfName )
   config.serviceAclTypeVrfMap.aclName[ key ] = aclName

def deleteServiceAcl( config, aclType, vrfName ):
   key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrfName )
   del config.serviceAclTypeVrfMap.aclName[ key ]

def clearServiceAcl( config, vrfName ):
   if vrfName is None:
      config.serviceAclTypeVrfMap.aclName.clear()
   else:
      for aclType in [ 'ip', 'ipv6' ]:
         deleteServiceAcl( config, aclType, vrfName )

class CapiIpAcl( CliCommand.CliCommandClass ):
   syntax = """( ip | ipv6 ) access-group <acl> [ in ]"""
   noOrDefaultSyntax = """( ip | ipv6 ) access-group ..."""
   data = {
            'ip': AclCli.ipKwForServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
            'access-group': AclCli.accessGroupKwMatcher,
            '<acl>': AclCli.standardIpAclNameMatcher,
            'in': AclCli.inKwMatcher
          }

   @staticmethod
   def handler( mode, args ):
      aclName = args[ '<acl>' ]
      aclType = 'ipv6' if 'ipv6' in args else 'ip'
      setServiceAcl( capiConfig, aclType, aclName, mode.vrfName_ )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      aclType = 'ipv6' if 'ipv6' in args else 'ip'
      deleteServiceAcl( capiConfig, aclType, mode.vrfName_ )

CapiVrfConfigMode.addCommandClass( CapiIpAcl )

#-------------------------------------------------------------------------------
# The "show management api http-commands https cipher" command
#-------------------------------------------------------------------------------
def showCapiCiphers( mode, args ):
   cipherList = CapiCipherSuites.DEFAULT_CIPHERS

   if serverStatus.cipherFilter:
      cipherList = serverStatus.cipherFilter.split( ":" )

   ciphers = []
   for cipher in cipherList:
      rfcName = CapiCipherSuites.cipherSuiteFilterToNameMap.get( cipher )
      if rfcName:
         ciphers.append( CapiModel.CipherSuite( cipherSuite=rfcName ) )
      else:
         mode.addWarning( "No conversion from OpenSSL cipher %s to RFC name" %
                          cipher )

   return CapiModel.CommandApiCipherSuites( enabledCipherSuites=ciphers )

class ShowApiHttpCommandsHttpsCipher( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands https cipher'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'https': httpsShowHelpDesc,
            'cipher': 'Show the cipher suites in use'
          }
   cliModel = CapiModel.CommandApiCipherSuites
   handler = showCapiCiphers
   privileged = True

BasicCli.addShowCommandClass( ShowApiHttpCommandsHttpsCipher )

#-------------------------------------------------------------------------------
# The "show management api http-commands https diffie-hellman" command
#-------------------------------------------------------------------------------
def showCapiDhparams( mode, args ):
   constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
   dhparams = readFile( constants.dhParamPath() )
   return CapiModel.CommandApiDiffieHellman( diffieHellmanParameters=dhparams )

class ShowApiHttpCommandsHttpsDiffieHellman( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management api http-commands https diffie-hellman'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'https': httpsShowHelpDesc,
            'diffie-hellman': 'Show the diffie-hellman parameters in use'
          }
   cliModel = CapiModel.CommandApiDiffieHellman
   privileged = True
   handler = showCapiDhparams

BasicCli.addShowCommandClass( ShowApiHttpCommandsHttpsDiffieHellman )

#-----------------------------------------------------------------------------
# "show management api http-commands ( ip | ipv6 ) access-list [<acl-name>]"
#-----------------------------------------------------------------------------
def showCapiAcl( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   name = args[ '<aclNameExpr>' ]
   capiStatus = serviceStatusDir.get( 'http-commands' )
   aclStatus = capiStatus.aclStatus if capiStatus else None
   return AclCli.showServiceAcl( mode,
                                 capiConfig.serviceAclTypeVrfMap,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 name )

class ShowApiHttpCommandsAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ('show management api http-commands '
             '('
             ' ( ip access-list [ <ipAclName> ] ) | '
             ' ( ipv6 access-list [ <ipv6AclName> ] ) '
             ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'api': apiShowHelpDesc,
            'http-commands': httpShowCommandsHelpDesc,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            '<ipAclName>': AclCli.ipAclNameExpression,
            '<ipv6AclName>': AclCli.ip6AclNameExpression
          }
   cliModel = AclCliModel.AllAclList
   privileged = True
   handler = showCapiAcl

BasicCli.addShowCommandClass( ShowApiHttpCommandsAcl )

#----------------------------------------------------------------
# "clear management api http-commands ip access-list counters"
# "clear management api http-commands ipv6 access-list counters"
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management api http-commands ( ip | ipv6 ) access-list counters'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'api': ConfigMgmtMode.apiKwMatcher,
            'http-commands': httpCommandsHelpdesc,
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'counters': AclCli.countersKwMatcher }

   @staticmethod
   def handler( mode, args ):
      if 'ip' in args:
         aclType = 'ip'
      elif 'ipv6' in args:
         aclType = 'ipv6'
      else:
         assert False

      capiStatus = serviceStatusDir.get( 'http-commands' )
      aclStatus = capiStatus.aclStatus if capiStatus else None
      AclCli.clearServiceAclCounters( mode,
                                      aclStatus,
                                      aclCheckpoint,
                                      aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#-------------------------------------------------------------------------------
# Command-API commands in "show tech-support"
#-------------------------------------------------------------------------------
def _showTechCmds():
   return [ "show management api http-commands" ]

TechSupportCli.registerShowTechSupportCmdCallback( '2014-03-05 15:08:19',
                                                   _showTechCmds )

# Plug-in definition:
def Plugin( entityManager ):
   global capiConfig
   global capiExecRequest
   global serverStatus
   global serviceStatusDir
   global ipStatus
   global ip6Status
   global sslStatus
   global cliConfig
   global aclCheckpoint
   capiConfig = ConfigMount.mount( entityManager,
                                   "mgmt/capi/config",
                                   "HttpService::Config",
                                   "w" )
   capiExecRequest = LazyMount.mount( entityManager, 
                                      "mgmt/httpserver/execRequest",
                                      "HttpService::ExecRequest",
                                      "w" )
   serverStatus = LazyMount.mount( entityManager,
                                   "mgmt/httpserver/status",
                                   "HttpService::Status",
                                   "r" )
   serviceStatusDir = LazyMount.mount( entityManager,
                                       "mgmt/httpserver/service",
                                       "Tac::Dir",
                                       "ri" )
   ipStatus = LazyMount.mount( entityManager,
                               "ip/status",
                               "Ip::Status",
                               "r" )
   ip6Status = LazyMount.mount( entityManager,
                                "ip6/status",
                                "Ip6::Status",
                                "r" )
   sslStatus = LazyMount.mount( entityManager,
                                "mgmt/security/ssl/status",
                                "Mgmt::Security::Ssl::Status",
                                "r" )
   cliConfig = ConfigMount.mount( entityManager, 
                                  "cli/config", 
                                  "Cli::Config", 
                                  "w" )
   aclCheckpoint = LazyMount.mount( entityManager,
                                    'mgmt/capi/aclCheckpoint',
                                    'Acl::CheckpointStatus',
                                    'w' )
