#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
import cPickle

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin.CliRelayModels import Command, Result
from CliPlugin.ControllerCli import addCvxShutdownCallback, addNoCvxCallback
from CliPlugin.ControllerCli import serviceAfterShowKwMatcher
from CliPlugin.ControllerdbLib import registerNotifiee, controllerGuard

from Ethernet import convertMacAddrCanonicalToDisplay, convertMacAddrToCanonical
import HostnameCli
import LazyMount
import MacAddr
import Plugins
import ShowCommand

from .BugAlertModel import Bug
from .BugAlertModel import BugAlertDetail
from .BugAlertModel import BugAlertReport
from .BugAlertModel import BugAlertSwitch
from .BugAlertModel import Conjunction
from .BugAlertModel import Summary
from .BugAlertModel import Switch
from .BugAlertModel import Tag

# Ununsed import just to build the dependency on Topology package
# as topology controllerdb mount, 'topology/version3/global/status' is used
# in this plugin.
import CliPlugin.NetworkTopology # pylint: disable-msg=W0611

bugalertStatus = None
alertBase = None
serviceCfgDir = None
toSwitch = None
fromSwitch = None
topologyStatus = None

def getImpactingBugList():
   '''
   get list of bugIds of known impacting bugs
   '''

   bugList = []
   switchStatus = bugalertStatus.switchStatus
   for switchName in switchStatus:
      bugList += switchStatus[ switchName ].bugAssertion.keys()
   return list( set( bugList ) )

#-------------------------------------------------------------------------------
# show service bug-alert summary
#-------------------------------------------------------------------------------

def showSummary( mode ):
   summary = Summary()
   if serviceCfgDir.service[ "BugAlert" ].enabled:
      summary.serviceStatus = 'Enabled'
   else:
      summary.serviceStatus = 'Disabled'
   summary.numSwitchesMonitored = len( bugalertStatus.switchStatus )

   summary.alertBase = Summary.AlertBase()
   ab = summary.alertBase
   ab.genId = alertBase.genId
   ab.releasedDate = alertBase.releaseDate
   ab.numTotalBugs = len( alertBase.bug )

   buggedSwitches = 0
   switchStatus = bugalertStatus.switchStatus
   for switchName in switchStatus:
      bugs = switchStatus[ switchName ].bugAssertion.keys()
      if len( bugs ) > 0:
         buggedSwitches += 1
   summary.numSwitchesWithBugs = buggedSwitches

   bugList = getImpactingBugList()
   summary.numBugsImpacting = len( bugList )

   return summary

bugAlertKw = CliMatcher.KeywordMatcher( 'bug-alert',
                                        helpdesc='BugAlert CVX Service' )

class ShowBugAlertSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service bug-alert summary'
   data = {
      'service' : serviceAfterShowKwMatcher,
      'bug-alert' : bugAlertKw,
      'summary' : 'Summary of BugAlert agent status'
      }
   cliModel = Summary

   @staticmethod
   def handler( mode, args ):
      return showSummary( mode )

BasicCli.addShowCommandClass( ShowBugAlertSummary )

#---------------------------------------------------------------------------------
# show service bug-alert report [ switch { mac <H.H.H> | hostname <hostname> } |
#                                 bug <bug-id> ]
#---------------------------------------------------------------------------------

def showReport( mode, args ):
   bugalertReport = BugAlertReport()

   # List of switches(mac <h-h-h-h>) to create BugAlertSwitch model for
   switches = []
   # List of bugs(int bugid) to create Bug model for
   bugs = []

   # dict of <key=bugid> : <val=list impacted switches>
   bugSwitchMap = defaultdict( list )

   singleBugReport = False
   singleSwitchReport = False

   # asked for bug/switch specific report
   if 'MAC' in args:
      val = args[ 'MAC' ]
      switches.append( convertMacAddrToCanonical( val ).replace( ':', '-' ) )
      bugalertReport.reportOption = 'switchOnly-report'
      singleSwitchReport = True
   elif 'HOSTNAME' in args:
      val = args[ 'HOSTNAME' ]
      if topologyStatus and val in topologyStatus.hostsByHostname:
         # Assuming only one mac addr per hostname
         macAddr = topologyStatus.hostsByHostname[ val ].host.keys()[ 0 ]
         switches.append( macAddr.replace( ':', '-' ) )
      bugalertReport.reportOption = 'switchOnly-report'
      singleSwitchReport = True
   elif 'BUG' in args:
      val = args[ 'BUG' ]
      bugs.append( val )
      bugalertReport.reportOption = 'bugOnly-report'
      singleBugReport = True
   # asked for the whole report
   else:
      switches += bugalertStatus.switchStatus.keys()
      # bugs = bugSwitchMap.keys(), once the map is populated by iterating through
      # switches
      bugalertReport.reportOption = 'report'

   switchStatus = bugalertStatus.switchStatus

   # populate switch models
   for switchName in switches:
      if switchName not in switchStatus:
         continue
      bugalertSwitch = BugAlertSwitch()

      canonicalAddr = switchName.replace( '-', ':' )
      bugalertSwitch.switchMac = convertMacAddrCanonicalToDisplay( canonicalAddr )

      hostname = ""
      if canonicalAddr in topologyStatus.host:
         hostname = topologyStatus.host[ canonicalAddr ].hostname
      bugalertSwitch.hostname = hostname

      bugalertSwitch.eosVersion = switchStatus[ switchName ].eosVersion

      bugAsserted = sorted( switchStatus[ switchName ].bugAssertion.keys() )
      bugalertSwitch.bugExposureList = bugAsserted

      for bugId in bugAsserted:
         switchModel = Switch()
         switchModel.switchMac = bugalertSwitch.switchMac
         switchModel.hostname = bugalertSwitch.hostname

         bugSwitchMap[ bugId ].append( switchModel )

      # Add this bugalertSwitch model to BugAlerReport.switches
      # switchName is mac addr in H-H-H-H form
      bugalertReport.switches[ switchName ] = bugalertSwitch

   # create entry in bugSwitchMap for single requested bug
   if singleBugReport:
      switchList = []
      for switchName in switchStatus:
         if bugs[ 0 ] in switchStatus[ switchName ].bugAssertion.keys():
            canonicalAddr = switchName.replace( '-', ':' )
            macAddr = convertMacAddrCanonicalToDisplay( canonicalAddr )
            switchModel = Switch()
            switchModel.switchMac = macAddr
            switchModel.hostname = topologyStatus.host[ canonicalAddr ].hostname
            switchList.append( switchModel )
      bugSwitchMap[ bugs[ 0 ] ] = switchList

   # now bugSwitchMap has entries for bugs, whose model has to be populated
   bugs = bugSwitchMap.keys()

   # populate bugs model
   for bugId in bugs:
      if not alertBase.bug.get( bugId ):
         continue
      bugReport = Bug()
      bugReport.bugId = bugId
      bugReport.bugSummary = alertBase.bug[ bugId ].alertSummary
      if not singleSwitchReport:
         bugReport.switchesImpacted = sorted( bugSwitchMap[ bugId ],
                                              key=lambda sw: sw.hostname )
      verFixKeys = [ ver.asString for ver in
                     alertBase.bug[ bugId ].versionFixed.keys() ]
      bugReport.versionsFixed = sorted( verFixKeys )
      bugReport.alertNote = alertBase.bug[ bugId ].alertNote

      # Add this bugReport model to BugAlertReport.bugs
      bugalertReport.bugs[ str( bugId ) ] = bugReport

   return bugalertReport

# possible <bug-id> completions are bugids of known impacting bugs
def getMatchBugIds():
   return [ str( i ) for i in getImpactingBugList() ]

class SwitchOrBugExpr( CliCommand.CliExpression ):
   expression = "( switch ( mac MAC ) | ( hostname HOSTNAME ) ) | ( bug BUG )"
   data = {
      'switch' : 'Show information per switch',
      'mac' : 'Specify switch MAC address',
      'MAC' : MacAddr.macAddrMatcher,
      'hostname' : CliCommand.guardedKeyword( 'hostname',
                                              'Specify switch hostname',
                                              controllerGuard ),
      'HOSTNAME' : HostnameCli.HostnameMatcher( helpname="WORD",
                                                helpdesc="Switch hostname" ),
      'bug' : 'Show information per bug',
      'BUG' : CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF, helpdesc='Bug ID' )
      }

class ShowBugAlertReport( ShowCommand.ShowCliCommandClass ):
   syntax = "show service bug-alert report [ SWITCH_BUG_ID ]"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'bug-alert' : bugAlertKw,
      'report' : 'Report of bugs and impacted switches',
      'SWITCH_BUG_ID' : SwitchOrBugExpr
      }
   cliModel = BugAlertReport
   handler = showReport

BasicCli.addShowCommandClass( ShowBugAlertReport )

#---------------------------------------------------------------------------------
# show service bug-alert detail { switch { <H.H.H> | <hostname> } | bug <bug-id> }
# show service bug-alert detail [ switches | bugs ]
#---------------------------------------------------------------------------------

def showSingleDetail( mode, args ):
   # List of switches(mac <h-h-h-h>) to create BugAlertSwitch model for
   switches = []
   # List of bugs(int bugids) to create Bug model for
   bugs = []

   if 'MAC' in args:
      val = args[ 'MAC' ]
      switches.append( convertMacAddrToCanonical( val ).replace( ':', '-' ) )
   elif 'HOSTNAME' in args:
      val = args[ 'HOSTNAME' ]
      if topologyStatus and val in topologyStatus.hostsByHostname:
         #Assuming only one mac addr per hostname
         macAddr = topologyStatus.hostsByHostname[ val ].host.keys()[ 0 ]
         switches.append( macAddr.replace( ':', '-' ) )
   elif 'BUG' in args:
      val = args[ 'BUG' ]
      bugs.append( val )
   else:
      assert False

   bugalertDetail = showDetail( switches, bugs )
   bugalertDetail.combinedDetail = False

   return bugalertDetail

def showAllDetail( mode, args ):
   if 'switches' in args:
      bugalertDetail = showDetail( bugalertStatus.switchStatus, [] )
      bugalertDetail.combinedDetail = False
   elif 'bugs' in args:
      bugalertDetail = showDetail( [], alertBase.bug )
      bugalertDetail.combinedDetail = False
   else:
      bugalertDetail = showDetail( bugalertStatus.switchStatus,
                                   alertBase.bug )
      bugalertDetail.combinedDetail = True

   return bugalertDetail

def showDetail( switches, bugs ):
   '''
   populate BugAlertDetail model with requested switches and bugs
   '''

   bugalertDetail = BugAlertDetail()

   # populate model for bugs
   for bugId in bugs:
      bug = alertBase.bug.get( bugId )
      if not bug:
         continue
      bugDetail = Bug()
      bugDetail.bugId = bugId
      bugDetail.bugSummary = bug.alertSummary

      verIntroKeys = [ ver.asString for ver in bug.versionIntroduced.keys() ]
      bugDetail.versionsIntroduced = sorted( verIntroKeys )

      verFixKeys = [ ver.asString for ver in bug.versionFixed ]
      bugDetail.versionsFixed = sorted( verFixKeys )

      bugDetail.alertNote = bug.alertNote
      bugDetail.bugRuleConjunctions = []

      for conjId in sorted( bug.bugRuleConjunction.keys() ):
         conjunction = Conjunction()
         conjunction.tags = []
         bugRuleConj = bug.bugRuleConjunction[ conjId ]
         for tagName in sorted( bugRuleConj.tag.keys() ):
            tag = Tag()
            tag.tagName = tagName
            tag.negation = bugRuleConj.tag[ tagName ]
            conjunction.tags.append( tag )
         bugDetail.bugRuleConjunctions.append( conjunction )

      # Add this bugDetail model to BugAlertDetail.bugs
      bugalertDetail.bugs[ str( bugId ) ] = bugDetail

   switchStatus = bugalertStatus.switchStatus
   # populate models for switches
   for switchName in switches:
      if switchName not in switchStatus:
         continue
      bugalertSwitch = BugAlertSwitch()

      canonicalAddr = switchName.replace( '-', ':' )
      bugalertSwitch.switchMac = convertMacAddrCanonicalToDisplay( canonicalAddr )

      hostname = ""
      if canonicalAddr in topologyStatus.host:
         hostname = topologyStatus.host[ canonicalAddr ].hostname
      bugalertSwitch.hostname = hostname

      bugalertSwitch.eosVersion = switchStatus[ switchName ].eosVersion
      bugalertSwitch.hardwarePlatform = 'DCS-7050X-64s'

      assertedTags = []
      tagAssertion = switchStatus[ switchName ].tagAssertion
      for tagName in sorted( tagAssertion.keys() ):
         if tagAssertion[ tagName ].state == 'deviceTagTrue':
            assertedTags.append( tagName )
      bugalertSwitch.assertedTags = assertedTags

      appDir = toSwitch[ switchName ][ 'BugAlert' ]
      for cmdId in sorted( appDir.showCommand.keys() ):
         showCommand = Command()
         showCommand.appname = 'BugAlert' # not required
         showCommand.cmdId = cmdId # not required

         cmdRequest = appDir.showCommand[ cmdId ]
         showCommand.command = cmdRequest.showCommand
         showCommand.revision = cmdRequest.revision # not required
         showCommand.pythonExpr = cmdRequest.pythonExpression

         # showCommand.result is instantiated only when there is an entry for cmd
         # under fromSwitch dir
         fromSwitchDir = fromSwitch[ switchName ]
         if ( 'BugAlert' in fromSwitchDir ) and \
                ( cmdId in fromSwitchDir[ 'BugAlert' ] ):
            cmdResult = fromSwitchDir[ 'BugAlert' ].showCommandResult[ cmdId ]
            result = Result()
            pyExpResult = cmdResult.pythonExpressionResult

            # check if result is not empty. It is left empty only in case of
            # exception. A False pyExpression result still needs to be captured
            if pyExpResult != '':
               result.pythonExpressionResult = pyExpResult
            if cmdResult.stuffForDebuggingOnly:
               unpickledStuff  =  cPickle.loads( cmdResult.stuffForDebuggingOnly )
               exceptionStr = ""
               if len( unpickledStuff ) > 1:
                  if type( unpickledStuff[ 1 ] ) == str:
                     exceptionStr = cPickle.loads( unpickledStuff[ 1 ] ).__str__()
                  elif type( unpickledStuff[ 1 ] ) == dict:
                     exceptionStr = unpickledStuff[ 1 ].get( "message" )
                  else:
                     exceptionStr = unpickledStuff[ 1 ].__str__()
               result.exception = exceptionStr

            showCommand.result = result

         showCommand.forSwitch = True # not required, specific to CliRelayModels only
         bugalertSwitch.commands[ str( cmdId ) ] = showCommand

      # Add this bugalertSwitch model to BugAlerDetail.switches
      # switchName is mac addr in H-H-H-H form
      bugalertDetail.switches[ switchName ] = bugalertSwitch

   return bugalertDetail

# possible <bug-id> completions are bugids of bugs present in AlertBase
def alertBaseBugIds():
   return [ str( i ) for i in alertBase.bug ]

class ShowBugAlertDetailSingle( ShowCommand.ShowCliCommandClass ):
   syntax = "show service bug-alert detail SWITCH_BUG_ID"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'bug-alert' : bugAlertKw,
      'detail' : 'Detail of bugs and switches',
      'SWITCH_BUG_ID' : SwitchOrBugExpr
      }
   cliModel = BugAlertDetail
   handler = showSingleDetail

BasicCli.addShowCommandClass( ShowBugAlertDetailSingle )

class ShowBugAlertDetailAll( ShowCommand.ShowCliCommandClass ):
   syntax = "show service bug-alert detail [ switches | bugs ]"
   data = {
      'service' : serviceAfterShowKwMatcher,
      'bug-alert' : bugAlertKw,
      'detail' : 'Detail of bugs and switches',
      'switches' : 'Info for all switches',
      'bugs' : 'Info for all bugs'
   }
   cliModel = BugAlertDetail
   handler = showAllDetail

BasicCli.addShowCommandClass( ShowBugAlertDetailAll )

def handleCvxShutdown( mode, no=False ):
   enabled = bool( no )
   global toSwitch
   global fromSwitch
   global topologyStatus
   if not enabled:
      toSwitch = None
      fromSwitch = None
      topologyStatus = None

addCvxShutdownCallback( handleCvxShutdown )
addNoCvxCallback( handleCvxShutdown )

#-------------------------------------------------------------------------------
# Mounts
#-------------------------------------------------------------------------------

def doControllerdbMounts( controllerdbEntMan ):
   global toSwitch, fromSwitch, topologyStatus

   toSwitch = LazyMount.mount( controllerdbEntMan,
                               'cliRelay/version1/toSwitch', 'Tac::Dir', 'ri' )

   fromSwitch = LazyMount.mount( controllerdbEntMan,
                                 'cliRelay/version1/fromSwitch', 'Tac::Dir', 'ri' )

   topologyStatus = LazyMount.mount( controllerdbEntMan,
                                     'topology/version3/global/status',
                                     'NetworkTopologyAggregatorV3::Status', 'ri' )

@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global serviceCfgDir
   global bugalertStatus
   global alertBase

   bugalertStatus = LazyMount.mount( entityManager,
                                     "bugalert/status",
                                     "BugAlert::BugAlertStatus", "r" )
   alertBase = LazyMount.mount( entityManager,
                                "bugalert/alertbase",
                                "BugAlert::AlertBase", "r" )
   serviceCfgDir = LazyMount.mount( entityManager,
                                    "controller/service/config",
                                    "Controller::ServiceConfigDir", "r" )
   registerNotifiee( doControllerdbMounts )
