#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import textwrap

import TableOutput
from CliModel import Bool, Dict, Enum, Int, List, Model, Str, Submodel
from CliPlugin.CliRelayModels import Command


#-----------------------------------------------
# Cli model for "show service bug-alert summary"
#-----------------------------------------------
class Summary( Model ):
   serviceStatus = Enum( values=( 'Enabled', 'Disabled' ), 
                         help="Status of BugAlert service" )
   numSwitchesMonitored = Int( help="No. of switches monitored" )

   # Model for AlertBase summary
   class AlertBase( Model ):
      genId = Str( help="GenID of AlertBase" )
      releasedDate = Str( help="Last release date of AlertBase" )
      numTotalBugs = Int( help="Total no. of bugs in AlertBase" )

   alertBase = Submodel( valueType=AlertBase, help="Summary of AlertBase" )
   numSwitchesWithBugs = Int( help="No. of switches with known bug impact" )
   numBugsImpacting  = Int( help="No. of known bugs which are impacting" )

   def render( self ):
      print "Bug Alert service: %s" % self.serviceStatus
      relStr = "%s, released %s" % ( self.alertBase.genId, 
                                     self.alertBase.releasedDate )
      print " Alert Base release: " + relStr
      print " Switches monitored: %d" % self.numSwitchesMonitored
      print " Total bugs in database: %d" % self.alertBase.numTotalBugs
      print " Total known bugs impact: %d" % self.numBugsImpacting
      print " Switches with known bugs impact: %d" % self.numSwitchesWithBugs
      print ''

#---------------------------------------------------
# General model for the switch
#---------------------------------------------------
class Switch( Model ):
   switchMac = Str( help='Mac address of the switch')
   hostname = Str( help='Hostname of the switch', optional=True )
   eosVersion = Str( help='Current version of EOS on the switch', optional=True )
   hardwarePlatform = Str( help='Platform info of the switch', optional=True )

#------------------------------------------------------
# Switch model specific to BugAlert ( report + detail )
#------------------------------------------------------
class BugAlertSwitch( Switch ):
   bugExposureList = List( valueType=int,
                           help='List of bugIds of bugs exposed to the switch',
                           optional=True )
   assertedTags = List( valueType=str, help='Tags asserted by inference engine', 
                        optional=True )
   commands = Dict( valueType=Command, 
                    help='Commands issued by BugAlert to switch, keyed by cmd id', 
                    optional=True )

#--------------------------
# Models for Bug rule
#--------------------------
class Tag( Model ):
   tagName = Str( help='Tag name' )
   negation = Enum( values=( 'null', 'positive', 'negative' ),
                    help='Whether tag is negated or not in bug rule' )

class Conjunction( Model ):
   tags = List( valueType=Tag, help='Tags present in the conjunction' )

#-----------------------------------
# Model for Bug ( report + detail )
#-----------------------------------
class Bug( Model ):
   __revision__ = 2

   bugId = Int( help='bugId of the bug' )
   bugSummary = Str( help='summary of the bug' )
   switchesImpacted = List( valueType=Switch,
                              help='Switches impacted by the bug',
                              optional=True )
   versionsIntroduced = List( valueType=str,
                             help='EOS versions in which the bug is introduced',
                             optional=True )
   versionsFixed = List( valueType=str,
                            help='EOS versions in which the bug is fixed' )
   alertNote = Str( help='Alert Note for the bug' )
   bugRuleConjunctions = List( valueType=Conjunction, help='Bug Rule Conjunctions',
                               optional=True )
   def degrade( self, dictRepr, revision ):
      if revision < 2:
         if not "switchesImpacted" in dictRepr:
            return dictRepr
         dictRepr[ "switchesImpacted" ] = [ sw[ "hostname" ] for sw in dictRepr[
            "switchesImpacted" ] ]
         return dictRepr

#---------------------------------------------------   
# Model for "show service bug-alert report ..." cmds 
#---------------------------------------------------
class BugAlertReport( Model ):
   __revision__ = 2

   switches = Dict( valueType=BugAlertSwitch, help='Switch info, keyed by switchMac',
                    optional=True )
   bugs = Dict( valueType=Bug, help='Bug info, keyed by bugId', optional=True )
   reportOption = Enum( values=( 'report', 'bugOnly-report', 'switchOnly-report' ),
                      help='Choice of report requested' )

   def _headingStr( self, heading ):
      print heading
      print '=' * len( heading )

   def _renderBug( self, bug, termWidth ):
      if self.reportOption != 'bugOnly-report':
         print "Bug ID: %d" % bug.bugId
      print textwrap.fill( " Summary: %s" % bug.bugSummary, termWidth )
      if self.reportOption != 'switchOnly-report':
         switchesImpacted = ', '.join( "%s (%s)" % \
                                       ( switch.hostname, switch.switchMac ) \
                                       for switch in bug.switchesImpacted )
         print textwrap.fill( " Switches impacted: %s" % switchesImpacted,
                              termWidth )
      print " Fixed in: %s" % ', '.join( bug.versionsFixed )
      print textwrap.fill( " Alert Note: %s" % bug.alertNote, termWidth )

   def _renderBugAlertSwitch( self, bugalertSwitch ):
      if self.reportOption != 'switchOnly-report':
         print "Switch: %s (%s)" % ( bugalertSwitch.hostname, 
                                     bugalertSwitch.switchMac )
      print " EOS version: %s" % bugalertSwitch.eosVersion
      bugList = bugalertSwitch.bugExposureList
      if bugList:
         print " Bug exposure: %s" % ', '.join( [ str( i ) for i in bugList ] )
      else:
         print " Bug exposure: None"

   def render( self ):
      termWidth = TableOutput.terminalWidth()
      if self.reportOption == 'report':
         self._headingStr( "Bugs report" )
         for bugId in sorted( [ int( i ) for i in self.bugs ] ):
            self._renderBug( self.bugs[ str( bugId ) ], termWidth )
            print ''
         self._headingStr( "Switch report" )
         dispSwitches = sorted( self.switches.values(), 
                                key=lambda switchModel : switchModel.hostname )
         for switch in dispSwitches:
            self._renderBugAlertSwitch( switch )
            print ''

      if self.reportOption == 'switchOnly-report':
         if not self.switches:
            print ''
            return
         switchMac = self.switches.keys()[ 0 ]
         switch = self.switches[ switchMac ]
         switchId = "'%s' (%s)" % ( switch.hostname, switch.switchMac )
         heading = "Switch %s report" % switchId
         self._headingStr( heading )
         self._renderBugAlertSwitch( switch )
         print ''
         heading = "Bugs report (for switch %s)" % switchId
         self._headingStr( heading )
         if not self.bugs:
            print ''
            return
         for bugId in sorted( [ int( i ) for i in self.bugs.keys() ] ):
            self._renderBug( self.bugs[ str( bugId ) ], termWidth )
            print ''

      if self.reportOption == 'bugOnly-report':
         if not self.bugs:
            print ''
            return
         bugId = self.bugs.keys()[ 0 ]
         bug = self.bugs[ bugId ]
         self._renderBug( bug, termWidth )
         print ''

#---------------------------------------------------
# Model for "show service bug-alert detail ..." cmds
#---------------------------------------------------
class BugAlertDetail( Model ):
   __revision__ = 2
   switches = Dict( valueType=BugAlertSwitch, help='Switch info, keyed by switchMac',
                    optional=True )
   bugs = Dict( valueType=Bug, help='Bug info, keyed by bugId', optional=True )
   combinedDetail = Bool( help='bugs and switches combined detail' )

   # render 'tag1', 'NOT tag1' etc
   def _renderTag( self, tag ):
      tagStr = tag.tagName
      if tag.negation == 'positive':
         return tagStr
      elif tag.negation == 'negative':
         return 'NOT ' + tagStr
      elif tag.negation == 'null':
         return ''

   # render ( 'tag1' AND NOT 'tag2' ) for eg
   def _renderConjunction( self, conj ):
      conjStr = '( '
      tagsList = []
      for tag in conj.tags:
         tagsList.append( self._renderTag( tag ) )
      conjStr += ' AND '.join( tagsList ) + ' )'
      return conjStr

   def _renderBug( self, bug, termWidth ):
      print "Bug ID: %d" % bug.bugId
      print textwrap.fill( " Summary: %s" % bug.bugSummary, termWidth )
      print " Versions Introduced: %s" % ', '.join( bug.versionsIntroduced )
      print " Versions Fixed: %s" % ', '.join( bug.versionsFixed )
      print textwrap.fill( " Alert Note: %s" % bug.alertNote, termWidth )

      bugRuleConjs = []
      for conj in bug.bugRuleConjunctions:
         bugRuleConjs.append( self._renderConjunction( conj ) )
      if len( bugRuleConjs ) == 0:
         bugRuleStr = ''
      else:
         bugRuleStr = ' OR '.join( bugRuleConjs )
      print " Bug Rule: %s" % bugRuleStr

   def _renderShowCommand( self, command ):
      print '  "%s":' % command.command
      print '    Expression: "%s"' % command.pythonExpr

      if command.result:
         # result.pythonExpressionResult is instantiated only if pyexp result is 
         # found without exception
         if command.result.pythonExpressionResult:
            resultStr = '"' + command.result.pythonExpressionResult + '"'
         else:
            resultStr = 'None'
         print "    Result: %s" % resultStr

         # result.exception is instantiated only if exception has occured
         if command.result.exception:
            exceptStr = command.result.exception
         else:
            exceptStr = 'None'
         print "    Exception: %s" % exceptStr
      
      # in case there is no entry for the cmd under fromSwitch
      else:
         print "    Result:"
         print "    Exception:"

   def _renderBugAlertSwitch( self, bugalertSwitch, termWidth ):
      print "Switch Name: %s (%s)" % ( bugalertSwitch.hostname, 
                                     bugalertSwitch.switchMac )
      print " Software version: %s" % bugalertSwitch.eosVersion
      print textwrap.fill( " tags: %s" % ', '.join( bugalertSwitch.assertedTags ),
                                                    termWidth )
      print " command results:"
      for cmdId in sorted( bugalertSwitch.commands.keys() ):
         command = bugalertSwitch.commands[ cmdId ]
         self._renderShowCommand( command )

   def _headingStr( self, heading ):
      print heading
      print '=' * len( heading )

   def render( self ):
      termWidth = TableOutput.terminalWidth()
      if self.combinedDetail:
         self._headingStr( "Bugs List" )

      for bugId in sorted( [ int( i ) for i in self.bugs.keys() ] ):
         self._renderBug( self.bugs[ str( bugId ) ], termWidth )
         print ''

      if self.combinedDetail:
         self._headingStr( "Switches covered" )

      dispSwitches = sorted( self.switches.values(), 
                             key=lambda switchModel : switchModel.hostname )
      for switch in dispSwitches:
         self._renderBugAlertSwitch( switch, termWidth )
         print ''
