# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import Arnet.Bpf
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Platform
import ConfigMount
import LazyMount
import ShowCommand
import Tac

from EtbaModel import HandlerCounter, HandlerCounters
from Toggles import EtbaDutToggleLib

etbaCliConfig = None
ebraCounterConfig = None
ebraCounterStatus = None
fwdIntfCounterConfig = None
fwdIntfCounterStatus = None

# Token for etba configuration commands 
def etbaGuard( mode, token ):
   if 'NSPATH' in os.environ:
      return None
   return CliParser.guardNotThisPlatform 

clearMatcher = CliToken.Clear.clearKwNode
countersMatcher = CliMatcher.KeywordMatcher( 
   'counters', helpdesc='Routing handler counters' )
etbaMatcher = CliMatcher.KeywordMatcher(
   'etba', helpdesc='Ebra Test Bridge configuration commands' )
etbaNode = CliCommand.Node( matcher=etbaMatcher, guard=etbaGuard )
platformMatcherForClear = CliToken.Platform.platformMatcherForClear
platformMatcherForShow = CliToken.Platform.platformMatcherForShow

platformMatcherForConfig = CliToken.Platform.platformMatcherForConfig
tfaMatcher = CliMatcher.KeywordMatcher(
   'tfa', helpdesc='Test Forwarding Agent' )

#-------------------------------------------------------------
# show platform etba counters [detail]
#-------------------------------------------------------------
detailMatcher = CliMatcher.KeywordMatcher(
   'detail', helpdesc="Show counters for each routing handler" )

def updateCounters( mode, config, status ):
   config.counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( status.counterUpdateTime >=
               config.counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description='Counters update',
                   maxDelay=1, sleep=True, timeout=5 )
   except Tac.Timeout:
      if config == ebraCounterConfig:
         mode.addWarning( 'Displaying stale counters for '
                          'EbraTestBridge:routingHandlers' )
      else:
         mode.addWarning( 'Displaying stale counters for '
                          'FwdIntfDevice:routingHandlers' )

def showEtbaCounter( mode, args ):
   updateCounters( mode, ebraCounterConfig, ebraCounterStatus )
   updateCounters( mode, fwdIntfCounterConfig, fwdIntfCounterStatus )

   ebraCounters = None
   fwdIntfCounters = None
   if args.get( 'detail' ):
      ebraCounters = {}
      fwdIntfCounters = {}
      for handler, counter in ebraCounterStatus.routingHandlerCounter.items():
         if handler != "overall":
            ebraCounters[ handler ] = (
               HandlerCounter( ignored=counter.ignored, routed=counter.routed ) )

      for handler, counter in fwdIntfCounterStatus.routingHandlerCounter.items():
         fwdIntfCounters[ handler ] = (
            HandlerCounter( ignored=counter.ignored, routed=counter.routed ) )

   # Can't do this with ebraIgnored/ebraRouted because even when a packet
   # is successully processed by an EbraTestBridge routing handler,
   # every other EbraTestBridge routing handler will still try to process the packet.
   # When a FwdIntfDevice routing handler successfully processes a packet,
   # other FwdIntfDevice routing handlers won't be called.
   fwdIntfIgnored = min( fwdIntfCounterStatus.routingHandlerCounter.values(),
                         key=( lambda counter: counter.ignored ) )
   fwdIntfRouted = sum( counter.routed for counter in
                        fwdIntfCounterStatus.routingHandlerCounter.values() )

   return HandlerCounters(
      ebraCounters=ebraCounters,
      ebraIgnored=ebraCounterStatus.routingHandlerCounter[ "overall" ].ignored,
      ebraRouted=ebraCounterStatus.routingHandlerCounter[ "overall" ].routed,
      fwdIntfCounters=fwdIntfCounters,
      fwdIntfIgnored=fwdIntfIgnored.ignored,
      fwdIntfRouted=fwdIntfRouted )

class ShowPlatformEtbaCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform etba counters [ detail ]'
   data = {
            'platform': platformMatcherForShow,
            'etba': etbaNode,
            'counters': countersMatcher,
            'detail': detailMatcher,
          }
   handler = showEtbaCounter
   cliModel = HandlerCounters

BasicCli.addShowCommandClass( ShowPlatformEtbaCountersCmd )

#-------------------------------------------------------------
# clear platform etba counters
#-------------------------------------------------------------
def clearCounters( mode, args ):
   ebraCounterConfig.resetCountersTrigger = (
      ( ebraCounterConfig.resetCountersTrigger + 1 ) % 256 )
   fwdIntfCounterConfig.resetCountersTrigger = (
      ( fwdIntfCounterConfig.resetCountersTrigger + 1 ) % 256 )

class clearCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear platform etba counters'
   data = { 
            'clear': clearMatcher,
            'platform': platformMatcherForClear,
            'etba': etbaNode,
            'counters': countersMatcher,
          }
   handler = clearCounters

BasicCli.EnableMode.addCommandClass( clearCountersCmd )

#-----------------------------------------------------------------------------------
# "platform tfa qtrace packet filter FILTER_NAME EXPRESSION" command, in config mode
#-----------------------------------------------------------------------------------
def setQtraceFilter( mode, args ):
   filterName = args[ 'FILTER_NAME' ]
   filterExpression = args[ 'EXPRESSION' ]

   if Arnet.Bpf.isValidPcapFilterExpression( filterExpression ):
      etbaCliConfig.qtraceFilter[ filterName ] = filterExpression
   else:
      mode.addError( "Syntax error in expression '%s'" % filterExpression )

def noQtraceFilter( mode, args ):
   filterName = args[ 'FILTER_NAME' ]

   del etbaCliConfig.qtraceFilter[ filterName ]

class etbaQtraceFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'platform tfa qtrace packet filter FILTER_NAME EXPRESSION'
   noOrDefaultSyntax = 'platform tfa qtrace packet filter FILTER_NAME ...'

   data = {
            'platform': platformMatcherForConfig,
            'tfa': tfaMatcher,
            'qtrace': 'Quicktrace logging',
            'packet': 'Packet information',
            'filter': 'PCAP filter',
            'FILTER_NAME': CliMatcher.PatternMatcher(
                              pattern=r'[a-zA-Z0-9_-]+',
                              helpname='WORD', helpdesc='Filter name' ),
            'EXPRESSION': CliMatcher.StringMatcher(
                              helpdesc='PCAP filter expression', helpname='WORD' ),
   }

   handler = setQtraceFilter
   noOrDefaultHandler = noQtraceFilter

if EtbaDutToggleLib.toggleFastEtbaEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( etbaQtraceFilterCmd )

def Plugin( entityManager ):
   global etbaCliConfig
   global ebraCounterConfig
   global ebraCounterStatus  
   global fwdIntfCounterConfig
   global fwdIntfCounterStatus

   etbaCliConfig = ConfigMount.mount(
      entityManager, "bridging/etba/cli/config",
      "Bridging::Etba::CliConfig", "w" )
   ebraCounterConfig = ConfigMount.mount(
      entityManager, "bridging/etba/counter/ebra/config",
      "Bridging::Etba::RoutingHandlerCounterConfig", "w" )
   ebraCounterStatus = LazyMount.mount(
      entityManager, "bridging/etba/counter/ebra/status",
      "Bridging::Etba::RoutingHandlerCounterStatus", "r" )
   fwdIntfCounterConfig = ConfigMount.mount(
      entityManager, "bridging/etba/counter/fwdIntf/config",
      "Bridging::Etba::RoutingHandlerCounterConfig", "w" )
   fwdIntfCounterStatus = LazyMount.mount(
      entityManager, "bridging/etba/counter/fwdIntf/status",
      "Bridging::Etba::RoutingHandlerCounterStatus", "r" )
