# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import MacAddress
from CliModel import Bool
from CliModel import Int
from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Enum
from CliModel import Float
from CliModel import Submodel
from CliModel import Bool # pylint: disable-msg=W0404
from CliModel import Str
from CliModel import DeferredModel
from IntfModels import Interface
import datetime
import Ethernet
import Tac
import IntfCli
import Arnet

switchportModes = {
      'access' : 'static access',
      'trunk' : 'trunk',
      'tool' : 'tool',
      'tap' : 'tap',
      'tapTool' : 'tap-tool',
      'dot1qTunnel' : 'tunnel'
}
sourceportFilterEnum = ( 'disabled', 'enabled' )
allowedVlansOnlyStr = 'allowedVlansOnly'
allConfiguredVlansStr = 'allConfiguredVlans'
vlanForwardingEnum = ( allowedVlansOnlyStr, allConfiguredVlansStr )

class _BaseTableEntry ( Model ):
   vlanId = Int( help="VLAN Id of MAC address" )
   macAddress = MacAddress( help="MAC Address" )

   def macToString( self ):
      # pylint: disable-msg=E1101
      return Ethernet.convertMacAddrCanonicalToDisplay( self.macAddress.stringValue )

class UnicastMacAddressTable( Model ):
   class UnicastTableEntry( _BaseTableEntry ):
      moves = Int( help="Number of moves", optional=True )
      lastMove = Float( help="Timestamp of last move", optional=True )
      entryType = Enum( values=( "static", "dynamic" ),
                        help="MAC Address creation type" )
      interface = Interface( help="Interface of the MAC Address" )

   tableEntries = List( valueType=UnicastTableEntry,
                        help="List of all of the Unicast entries" )

   def render( self ):
      fmt = '%4d    %-14s    %-8s    %-8s   %-7s %s'
      print '          Mac Address Table'
      print '------------------------------------------------------------------'
      print
      print 'Vlan    Mac Address       Type        Ports      Moves   Last Move'
      print '----    -----------       ----        -----      -----   ---------'
      for unicastTableEntry in self.tableEntries:
         movesStr = ''
         lastMoveStr = ''
         if unicastTableEntry.moves is not None:
            movesStr = str( unicastTableEntry.moves )
         if unicastTableEntry.lastMove is not None:
            lastMoveDelta = int( Tac.now() - unicastTableEntry.lastMove )
            lastMoveStr = str( datetime.timedelta(
                                    seconds=lastMoveDelta) ) + ' ago'

         row = fmt % ( unicastTableEntry.vlanId,
                       unicastTableEntry.macToString(),
                       unicastTableEntry.entryType.upper(),
                       IntfCli.Intf.getShortname( unicastTableEntry.interface ),
                       movesStr,
                       lastMoveStr )
         print row.rstrip()
      print 'Total Mac Addresses for this criterion:', \
             len( self.tableEntries )

class MulticastMacAddressTable( Model ):
   class MulticastTableEntry( _BaseTableEntry ):
      entryType = Enum( values=( "static", ), help="Type" )
      interfaces = List( valueType=Interface, help="Interfaces of the MAC Address" )

   tableEntries = List( valueType=MulticastTableEntry,
                        help="List of all of the Multicast entries" )

   def render( self ):
      fmt = '%4d    %-14s    %-8s    %-8s'
      formatInterface = '                                      %8s'
      print
      print '          Multicast Mac Address Table'
      print '------------------------------------------------------------------'
      print
      print 'Vlan    Mac Address       Type        Ports'
      print '----    -----------       ----        -----'
      for multicastTableEntry in self.tableEntries:
         interfaceSet = self._createInterfacesString(
                                                   multicastTableEntry.interfaces )
         row = fmt % ( multicastTableEntry.vlanId,
                       multicastTableEntry.macToString(),
                       multicastTableEntry.entryType.upper(),
                       interfaceSet[ 0 ] )
         print row.rstrip()

         for i in range( 1, len( interfaceSet ) ):
            row = formatInterface % ( interfaceSet[ i ] )
            print row.rstrip()

      print 'Total Mac Addresses for this criterion:', \
             len( self.tableEntries )

   def _createInterfacesString( self, interfaces ):
      interfaceSet = [ ]
      currInterfaceString = ''
      for intf in interfaces:
         shortInftName = IntfCli.Intf.getShortname( intf )
         if( len( currInterfaceString + shortInftName ) > 40 ):
            interfaceSet.append(currInterfaceString)
            currInterfaceString = ''
         currInterfaceString += shortInftName + ' '

      interfaceSet.append(currInterfaceString)
      return interfaceSet

class MacAddressTable( DeferredModel ):
   __revision__ = 2
   unicastTable = Submodel( valueType=UnicastMacAddressTable,
                            help="Unicast MAC address table" )
   multicastTable = Submodel( valueType=MulticastMacAddressTable, optional=True,
                              help="Multicast MAC address table")
   disabledMacLearningVlans = List( valueType=int,
                              help="VLANs with disabled MAC learning",
                              sinceRevision=2 )

class MacSourceCheckSystemMac( Model ):
   sourceCheckCPUMac = Bool( help="Source check CPU Mac" )

   def render( self ):
      if self.sourceCheckCPUMac:
         print 'Source check for system mac enabled'
      else:
         print 'Source check for system mac disabled'

class MacSourceCheckInvalid( Model ):
   sourceCheckInvalid = Bool( help="Source check invalid Mac" )

   def render( self ):
      if self.sourceCheckInvalid:
         print 'Source check for invalid mac enabled'
      else:
         print 'Source check for invalid mac disabled'

class MacAddressTableAgingTime( Model ):
   hostAgingTime =  Float( help="Host Aging Time" )

   def render( self ):
      print 'Global Aging Time:  %d' % self.hostAgingTime
      
class MacAddressTableCount( Model ):
   class VlanCount( Model ):
      dynamic = Int( help="Number of dynamic address MAC entries" )
      unicast = Int( help="Number of unicast address MAC entries" )
      multicast = Int( help="Number of multicast address MAC entries" )
      
   vlanCounts = Dict( keyType=int, valueType=VlanCount,
                  help="Mapping between a VLAN Id and it's counters" )

   def render( self ):
      for vlanId in sorted( self.vlanCounts ):
         vlanCount = self.vlanCounts[ vlanId ]
         print
         print 'Mac Entries for Vlan %d:' % vlanId
         print '---------------------------'
         print 'Dynamic Address Count            : %d' % vlanCount.dynamic
         print 'Unicast Static  Address Count    : %d' % vlanCount.unicast
         print 'Multicast Static  Address Count  : %d' % vlanCount.multicast
         print 'Total Mac Addresses              : %d' % \
                      ( vlanCount.dynamic + vlanCount.unicast + vlanCount.multicast )

      print
      
      # BUG191 This part of the command has been removed until we have a meaningful
      # value to display.
      #print 'Total Mac Address Space Available: %d' % 0
      #print
      
class Dot1QTunnel( Model ):
   interfaces = List( valueType=Interface, 
                      help="List of interfaces in dot1q-tunnel mode" )
   
   def render( self ):
      print "dot1q-tunnel mode LAN Port (s)"
      print "------------------------------"
      
      if not self.interfaces:
         print "No ports have been configured as dot1q-tunnel"
      else:
         for interface in self.interfaces:
            print IntfCli.Intf.getShortname( interface )

SWITCH_FORWARDING_MODES = Tac.Type( "Bridging::SwitchForwardingMode" )

class SwitchForwardingMode( Model ):
   class SupportedMode( Model ):
      supportedMode = Enum( help="Supported switching modes", 
                            values=SWITCH_FORWARDING_MODES.attributes )
      
   currentMode = Enum( help="Current switching mode", 
                       values=SWITCH_FORWARDING_MODES.attributes )
   supportedModes = List( help="List of supported modes", valueType=SupportedMode )

   currentL3McastMode = Enum( help="Current multicast forwarding mode",
                            values=SWITCH_FORWARDING_MODES.attributes )
   supportedL3McastModes = List( help="List of supported multicast modes", 
                               valueType=SupportedMode )

   def render( self ):
      print 'Current switching mode:   ',
      if self.currentMode == SWITCH_FORWARDING_MODES.storeAndForward:
         print 'store and forward'
      elif self.currentMode == SWITCH_FORWARDING_MODES.cutThrough:
         print 'cut through'
      else:
         print 'default'
      
      if self.supportedL3McastModes and \
            self.currentL3McastMode != SWITCH_FORWARDING_MODES.defaultMode:
         print 'Multicast override mode:  ',
         if self.currentL3McastMode == SWITCH_FORWARDING_MODES.storeAndForward:
            print 'store and forward'
         elif self.currentL3McastMode == SWITCH_FORWARDING_MODES.cutThrough:
            print 'cut through'
         else:
            assert False, "Multicast forwarding mode is not a valid value."

      if self.supportedModes:
         print 'Available switching modes:',
         optionList = []
         for swMode in self.supportedModes:
            if swMode.supportedMode == SWITCH_FORWARDING_MODES.storeAndForward:
               optionList.append( 'store and forward' )
            elif swMode.supportedMode == SWITCH_FORWARDING_MODES.cutThrough:
               optionList.append( 'cut through' )
            else:
               optionList.append( 'default' )
         print ', '.join( optionList )

      if self.supportedL3McastModes and \
            self.currentL3McastMode != SWITCH_FORWARDING_MODES.defaultMode:
         print 'Available multicast override modes:',
         optionList = []
         for swMode in self.supportedL3McastModes:
            if swMode.supportedMode == SWITCH_FORWARDING_MODES.storeAndForward:
               optionList.append( 'store and forward' )
            elif swMode.supportedMode == SWITCH_FORWARDING_MODES.cutThrough:
               optionList.append( 'cut through' )
            else:
               assert False, "Multicast override mode is not a valid value"
         print ', '.join( optionList )

class PauseFramePassThrough( Model ):
   pauseFramePassThrough =  Bool( help="Pause frame visibility" )

   def render( self ):
      if self.pauseFramePassThrough:
         print 'Pass through: Enabled'
      else:
         print 'Pass through: Disabled'
 
class SwitchPortInfo( Model ):
   mode = Enum( values=switchportModes.keys(), help="Administrative or "
                "operational mode of the switchport" )
   phoneTrunk = Bool( help="Phone Trunk mode is enabled" )
   macLearning = Bool( help="Whether MAC address learning is enabled" )
   tpid = Str( help="Dot1q ethertype/Tag protocol ID in hex", optional=True )
   tpidStatus = Bool( help="Whether tpid is active or not", optional=True )
   dot1qVlanTagRequired = Bool( help="Allow only vlan tagged frames",
                                default=False )
   dot1qVlanTagRequiredStatus = Bool( help="Allow only vlan tagged frames is"
                                      " active on the port", default=False )
   dot1qVlanTagDisallowed = Bool( help="Drop tagged frames is"
                                  " active on the port", default=False )
   accessVlanId = Int( help="Access VLAN ID" )
   accessVlanName = Str( help="Access VLAN name" )
   trunkingNativeVlanId = Int( help="Trunking native mode VLAN ID" )
   trunkingNativeVlanName = Str( help="Trunking native mode VLAN name", 
                                 optional=True )
   privateVlanMapping = Str( help="Administrative private VLANs mapped to the"
                             " switchport. Represented in canonical form",
                             optional=True )
   trunkAllowedVlans = Str( help="Allowed trunk VLANs on the switchport."
                            " Represented in canonical form."
                            " For eg: ( 2-40,70,100-200 ), ALL(1-4094), NONE etc" )
   staticAllowedVlans = Str( help="Static VLANs out of the allowed trunk VLANs"
                             " on the switchport. Represented in canonical form",
                             optional=True )
   dynamicAllowedVlans = Dict( keyType=str, valueType=str, 
                               help="A mapping between a process name and dynamic"
                               " VLANs out of the allowed trunk VLANs on the"
                               " switchport. Represented in canonical form" )
   staticTrunkGroups = List( valueType=str, help="List of static trunk group names" )
   dynamicTrunkGroups = List( valueType=str,
                              help="List of dynamic trunk group names" )
   sourceportFilterMode = Enum( values = sourceportFilterEnum,
                                help = "Whether source interface "
                                "filtering is disabled or not" )
   vlanForwardingMode = Enum( values = vlanForwardingEnum,
                                help = "Whether the VLAN forwarding mode "
                                "is set to allowed VLANs on the port or "
                                "all configured VLANs" )
   phoneVlan = Int( help="Phone VLAN ID" )
   phoneTrunkUntagged = Bool( help="Phone trunk untagged is enabled" )

   def render( self ):
      mode = switchportModes.get( self.mode )
      if mode == "trunk" and self.phoneTrunk:
         mode += " phone"
      print 'Administrative Mode: %s' % mode
      print 'Operational Mode: %s' % mode
      macLearningStr = ( self.macLearning and 'enabled' ) or 'disabled'
      print 'MAC Address Learning: %s' % macLearningStr
      if self.tpid:
         status = ''
         if self.tpidStatus is not None:
            status = ' (active)' if self.tpidStatus else ' (inactive)'
         print 'Dot1q ethertype/TPID: %s%s' % ( self.tpid, status )
      print 'Dot1q VLAN Tag:',
      if self.dot1qVlanTagRequiredStatus:
         print 'Required'
      elif self.dot1qVlanTagDisallowed:
         print 'Disallowed'
      else:
         print 'Allowed'
      print 'Access Mode VLAN: %d (%s)' % ( self.accessVlanId,
                                               self.accessVlanName )
      if self.trunkingNativeVlanId == 0:
         print 'Trunking Native Mode VLAN: none'
         print 'Administrative Native VLAN tagging: enabled'
      elif self.trunkingNativeVlanName:
         print 'Trunking Native Mode VLAN: %d (%s)' % \
               ( self.trunkingNativeVlanId, 
                 self.trunkingNativeVlanName )
         print 'Administrative Native VLAN tagging: disabled' 
      if self.privateVlanMapping:
         print 'Administrative private VLAN mapping: %s' % \
               self.privateVlanMapping
      print 'Trunking VLANs Enabled: %s' % self.trunkAllowedVlans
      if self.dynamicAllowedVlans:
         print 'Static VLANs: %s' % self.staticAllowedVlans
         print 'Dynamic VLANs:'
         for key in self.dynamicAllowedVlans:
            print '  %s: %s' % ( key, self.dynamicAllowedVlans[ key ] )
      # print trunkGroups if switchIntfConfig exists
      if self.mode is not None:
         trunkGroupNames = ', '.join( self.staticTrunkGroups )
         print "Static Trunk Groups: %s" % trunkGroupNames 
         trunkGroupNames = ', '.join( self.dynamicTrunkGroups )
         print "Dynamic Trunk Groups: %s" % trunkGroupNames 
      print 'Source interface filtering: %s' % self.sourceportFilterMode
      print 'VLAN forwarding mode: %s' % self.vlanForwardingMode
      if self.phoneVlan:
         print 'Phone VLAN: %d' % self.phoneVlan
      if self.phoneTrunk:
         print 'Phone trunk tagging: %s' % \
            ( "untagged" if self.phoneTrunkUntagged else "tagged" )

class SwitchPort( Model ):
   enabled = Bool( help="Whether switchport is enabled" )
   switchportInfo = Submodel( valueType=SwitchPortInfo, 
                              help="Switchport information", optional=True )

class SwitchPorts( Model ):
   switchports = Dict( keyType=Interface, valueType=SwitchPort,
                       help="A collection of switchports indexed by interface name" )

   def render( self ):
      for intfName in Arnet.sortIntf( self.switchports ):
         switchport = self.switchports[ intfName ]
         print 'Name: %s' % IntfCli.Intf.getShortname( intfName ) 
         print 'Switchport: %s' % ( switchport.enabled and 'Enabled' or 'Disabled' )
         switchportInfo = switchport.switchportInfo
         if switchportInfo:
            switchportInfo.render()
         print 

class ConfigSource( Model ):
   source = Str( help="Name of the config source" )
   config = List( valueType=str,
                  help="List of config commands published by this source" )

   def render( self ):
      print "\tSource: %s" % self.source
      for cmd in self.config:
         print "\t\t%s" % cmd
      print

class SwitchportsConfigSources( Model ):
   interfaces = Dict( keyType=Interface, valueType=ConfigSource,
                      help="Switchport config source indexed by interface "
                      "name" )
   def render( self ):
      for intfName in Arnet.sortIntf( self.interfaces ):
         configSource = self.interfaces[ intfName ]
         print "Interface: %s" % intfName
         configSource.render()
