# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements bridging configuration.  In particular, it provides:
# -  the "[no] mac address-table aging-time" command
# -  the "show bridge mac-address-table aging time" command
# -  the "[no] mac address-table static" command
# -  the "show mac address-table" command
# -  the "show mac address-table count" command
# -  the "show interfaces [<intf>] switchport" command.
# -  the "show dot1q-tunnel [<intf>]" command, in "enable" mode.
# -  the "[no|default] switch forwarding-mode 
#                              cut-through | store-and-forward" command.
# -  the "[no|default] mac address-table reserved forward" command
# -  the "[no|default] mac pause-frame pass-through" command
# -  the "show mac pause-frame" command
#
# This module does not contain VLAN-specific commands.
#-------------------------------------------------------------------------------
import Tac, Arnet, CliParser, BasicCli, MacAddr, Ethernet, Logging, EbraLogMsgs
import VlanCli, IntfCli, EbraEthIntfCli, LazyMount, UtmpDump, EbraCliLib
import ConfigMount, SmashLazyMount
import Intf, CliExtensions, EbraLib
import CliToken.Clear
import CliToken.Mac
import CliToken.Switch
import os, sys
import Cell
import CliCommand, CliMatcher
import ShowCommand, BasicCliModes
from Intf.IntfRange import IntfRangeMatcher

from Vlan import computeVlanRangeSet, vlanSetToCanonicalString
from VlanCli import switchportMatcher
from CliPlugin import BridgingCliModel
from BridgingCliModel import Dot1QTunnel
from BridgingCliModel import MacAddressTable
from BridgingCliModel import MulticastMacAddressTable
from BridgingCliModel import MacAddressTableAgingTime
from BridgingCliModel import MacAddressTableCount
from EbraLib import defaultTpid
from BridgingCliModel import SwitchPort, SwitchPorts # pylint: disable=E0611
from BridgingCliModel import SwitchPortInfo # pylint: disable=E0611
from TypeFuture import TacLazyType
from BridgingCliModel import allowedVlansOnlyStr, allConfiguredVlansStr
from VlanCli import vlanIdMatcher

VlanTagFormat = TacLazyType( 'Bridging::VlanTagFormat' )
PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )
# pylint: disable-msg=W0621

# Module globals set up by the Plugin function below
cliConfig = None
flushConfig = None
flushReply = None
redundancyStatus = None
bridgingConfig = None
bridgingStatus = None
bridgingHwCapabilities = None
bridgingSwitchIntfConfig = None
intfTpidStatus = None
bridgingFdbConfig = None
dynAllowedVlanDir = None
vlanTagFormatDropStatusDir = None
flushReplySm = None

#-------------------------------------------------------------------------------
# Some basic tokens for MAC address tables.
#-------------------------------------------------------------------------------
matcherAddressTable = CliMatcher.KeywordMatcher( 'address-table',
      helpdesc='MAC forwarding table' )
matcherAgingTimeConfig = CliMatcher.KeywordMatcher( 'aging-time',
      helpdesc='Set MAC address table entry maximum age')
matcherAgingTimeShow = CliMatcher.KeywordMatcher( 'aging-time',
      helpdesc='aging-time keyword')
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface keyword' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN Keyword' )
nodeMacAddressTable = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'mac-address-table', helpdesc='_' ),
      hidden=True )

class MacAddrTableExprForConfig( CliCommand.CliExpression ):
   expression = '( mac address-table ) | mac-address-table'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'address-table': matcherAddressTable,
      'mac-address-table': nodeMacAddressTable,
   }

class MacAddrTableExprForClear( CliCommand.CliExpression ):
   expression = '( mac address-table ) | mac-address-table'
   data = {
      'mac': CliToken.Mac.macMatcherForClear,
      'address-table': matcherAddressTable,
      'mac-address-table': nodeMacAddressTable,
   }

class MacAddrTableExprForShow( CliCommand.CliExpression ):
   expression = '( mac address-table ) | mac-address-table'
   data = {
      'mac': CliToken.Mac.macMatcherForShow,
      'address-table': matcherAddressTable,
      'mac-address-table': nodeMacAddressTable,
   }

#--------------------------------------------------------------------------------
# [ no | default ] ( ( mac address-table ) | mac-address-table ) 
# aging-time ( DISABLE_AGING | AGING_TIME )
#--------------------------------------------------------------------------------
defaultAgingTime = 300

def setAgingTime( mode, args ):
   cliConfig.hostAgingTime = args.get( 'AGING_TIME', 0 )

def noAgingTime( mode, args):
   cliConfig.hostAgingTime = defaultAgingTime

class SetAgingTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'MAC_ADDR_TABLE aging-time ( DISABLE_AGING | AGING_TIME )'
   noOrDefaultSyntax = 'MAC_ADDR_TABLE aging-time ...'
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'aging-time': matcherAgingTimeConfig,
      'DISABLE_AGING': CliMatcher.IntegerMatcher( 0, 0,
         helpdesc='Enter 0 to disable aging' ),
      'AGING_TIME': CliMatcher.IntegerMatcher( 10, 1000000,
         helpdesc='Aging time in seconds' ),
   }
   handler = setAgingTime
   noOrDefaultHandler = noAgingTime

BasicCliModes.GlobalConfigMode.addCommandClass( SetAgingTimeCmd )

#-------------------------------------------------------------------------------
# The "show bridge mac-address-table aging timeout" command, in "enable" mode.
#
# The full syntax of this command is:
#   show bridge mac-address-table aging timeout
# legacy:
#   show mac address-table aging-time
#-------------------------------------------------------------------------------

def doShowAgingTime( mode, args ):
   return MacAddressTableAgingTime( hostAgingTime=cliConfig.hostAgingTime )

matcherAgingTimeLegacy = CliCommand.Node( matcher=matcherAgingTimeShow, 
                           deprecatedByCmd=
                           'show bridge mac address-table aging timeout' )

class BridgeAgingTimeoutCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bridge MAC_ADDR_TABLE aging timeout'
   data = {
      'bridge': 'Bridge information',
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'aging': 'Aging keyword',
      'timeout': 'Timeout keyword',
   }
   handler = doShowAgingTime
   cliModel = MacAddressTableAgingTime

BasicCli.addShowCommandClass( BridgeAgingTimeoutCmd )

#--------------------------------------------------------------------------------
# show mac address-table aging-time
#--------------------------------------------------------------------------------
class AgingTimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE aging-time'
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'aging-time': matcherAgingTimeLegacy,
   }
   handler = doShowAgingTime
   cliModel = MacAddressTableAgingTime

BasicCli.addShowCommandClass( AgingTimeCmd )
# Providing Cli Hooks to individual features for static (configured) 
# mac entries. 
# 
# 1. bridgingCheckStaticMacHook: Check if configuredHost can be added to 
#    fdbConfig. Registered extension functions (hooks) return True/False.
# 2. bridgingAddStaticMacHook: Any extra work to be done when a static
#    mac address is added. Hooks return True/False.
# 3. bridgingDelStaticMacHook: Any extra work to be done when a static mac 
#    address is deleted. Hooks return True/False.
# 4. bridgingExtraStaticMacHook: To get any additional static mac entries, 
#    e.g., when the 'show mac address-table configured' is issued. Hooks 
#    return a list of Bridging::ConfiguredHost. 
# 
# These hooks can be utilized by external individual features that manipulate 
# static mac entries. For instance, Vxlan, which keeps vxlan mac table for 
# remote hosts adds the hook extensions to these Cli Hooks. 

bridgingCheckStaticMacHook = CliExtensions.CliHook()
bridgingAddStaticMacHook = CliExtensions.CliHook()
bridgingDelStaticMacHook = CliExtensions.CliHook()
bridgingExtraStaticMacHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# The "[no] mac address-table static" command, in "config" mode.
#
# The full syntax of this command is:
#
#   mac address-table static <mac_addr> vlan <vlan_id>
#                                                   {interface <eth_intf>+|drop}
#   {no|default} mac address-table static <mac_addr> vlan <vlan_id>
#                                                   [interface <eth_intf>+|drop]
#-------------------------------------------------------------------------------
staticKwMatcher = CliMatcher.KeywordMatcher( 'static', helpdesc='static keyword' )

def getIntfNames( intfs ):
   intfNames = set()
   if ( hasattr(intfs, "__iter__") ):
      for intf in intfs:
         if isinstance( intf, Intf.IntfRange.IntfList ):
            intfNames.update( list( intf.intfNames() ) )
         elif intf.name not in intfNames:
            intfNames.update( [ intf.name ] )
   else:
      intfNames.update( list( intfs.intfNames() ) )
   return intfNames

def setStatic( mode, args ):
   intfNamesOrDrop = args.get( 'INTFS' )
   if intfNamesOrDrop is not None:
      intfNamesOrDrop = getIntfNames( intfNamesOrDrop )
   else:
      intfNamesOrDrop = args.get( 'drop' )
   macAddr = args[ 'MACADDR' ]
   vlanId = args[ 'VLAN_ID' ]
   no = CliCommand.isNoOrDefaultCmd( args )
   setStaticHelper( mode, macAddr, vlanId, intfNamesOrDrop, no=no )

labelType = Tac.Type( 'Arnet::MplsLabel' )

def setStaticHelper( mode, macAddr, vlanId, intfNamesOrDrop,
                     label=labelType.null, no=False ):
   intfNames = set()
   if intfNamesOrDrop is not None and intfNamesOrDrop != 'drop':
      intfNames = intfNamesOrDrop

   macAddr = Ethernet.convertMacAddrToCanonical( macAddr )

   if Ethernet.isBroadcast( macAddr ) \
      or Ethernet.isIPMulticast( macAddr, allowIanaReserved=True ) \
      or macAddr == '00:00:00:00:00:00':
      mode.addError( 'Only unicast or non-IP multicast '
                     'addresses can be configured statically.' )
      return
   elif Ethernet.isUnicast( macAddr ) and \
        intfNamesOrDrop is not None and \
        ( intfNamesOrDrop != 'drop' and len( intfNames ) > 1 ):
      mode.addError( 'Only one interface is allowed for unicast addresses.' )
      return
   elif Ethernet.isMulticast( macAddr ) and intfNamesOrDrop == 'drop':
      mode.addError( 'Cannot drop on a multicast address.' )
      return

   fdbConfig = cliConfig.fdbConfig.newMember( vlanId.id )

   def staticAddrNotFound( vlanId, macAddr ):
      # Ebra may not be aware of this mac address. So check  
      # with the registered cli hooks, if anyone is aware.  
      delStatus = False
      for hook in bridgingDelStaticMacHook.extensions():
         status = hook( macAddr, vlanId )
         delStatus |= status
      if not delStatus:
         # None of the hooks were successful
         mode.addWarning( 'MAC address could not be removed: Address not found' )

   def delStaticAddrAndNotify( vlanId, macAddr ):
      del table[ macAddr ]
      for hook in bridgingDelStaticMacHook.extensions():
         hook( macAddr, vlanId )
      
   # Unicast case
   if Ethernet.isUnicast( macAddr ):
      table = fdbConfig.configuredHost
      if no:
         a = table.get( macAddr )
         if not a:
            staticAddrNotFound( vlanId, macAddr )
            return
         else:
            if intfNamesOrDrop is None:
               delStaticAddrAndNotify( vlanId, macAddr )
            elif intfNamesOrDrop != 'drop' and a.intf != '':
               delStaticAddrAndNotify( vlanId, macAddr )
            elif intfNamesOrDrop == 'drop' and a.intf == '':
               delStaticAddrAndNotify( vlanId, macAddr )
            else:
               staticAddrNotFound( vlanId, macAddr )
            return

      # Address needs to be added to the collection of configuredHost
      if intfNamesOrDrop == 'drop':
         intf = ''
      else:
         for intfIn in intfNames:
            intf = intfIn 
      
      # Confirm from the hooks, if it is OK to add static mac entry. Don't 
      # add if any of the hooks returns False.
      accept = True
      for hook in bridgingCheckStaticMacHook.extensions():
         if not hook( mode, macAddr, vlanId, intfNamesOrDrop ):
            accept = False

      if accept:
         table.addMember( Tac.Value(
            "Bridging::ConfiguredHost", address=macAddr, intf=intf,
            entryType='configuredStaticMac', label=label ) )
         # Notify add through the add CliHook
         for hook in bridgingAddStaticMacHook.extensions():
            hook( macAddr, vlanId, intfNamesOrDrop )

   # Multicast case, exceptions addressed above
   else:
      if mode.session_.guardsEnabled() \
         and not bridgingHwCapabilities.staticMcastSupported:
         mode.addError( 'Adding a static multicast MAC address is not '
                        'supported on this platform' )
         return
      table = fdbConfig.ethGroup
      if no:
         if not macAddr in table:
            staticAddrNotFound( vlanId, macAddr )
            return
         else:
            a = table.get( macAddr )
            if intfNamesOrDrop is None:
               delStaticAddrAndNotify( vlanId, macAddr )
            else:
               for intf in intfNames:
                  intfList = a.intf
                  if intf in intfList:
                     del a.intf[ intf ]
                  else:
                     mode.addError( 'Interface ' + intf + ' could not be removed '
                                    'from entry ' + str( macAddr ) )
                     mode.addError( 'The interface is not configured '
                                    'for the address' )
               # If there are no more interfaces remaining, delete the multicast
               # table entry
               if len( a.intf ) == 0:
                  delStaticAddrAndNotify( vlanId, macAddr )
            return

      # Again confirm from the Cli hooks, if it is OK to add this static mac.
      proceed = True
      for hook in bridgingCheckStaticMacHook.extensions():
         if not hook( mode, macAddr, vlanId, intfNamesOrDrop ):
            proceed = False
      if not proceed:
         return
      
      # Address needs to be added to the collection ethGroup
      # 'drop' is not a valid option (enforced above)
      entry = table.get( macAddr )
      if not entry:
         entry = fdbConfig.newEthGroup( macAddr ) 
      for intf in intfNames:
         entry.intf[ intf ] = True
                   
class MacAddressStaticCmd( CliCommand.CliCommandClass ):
   syntax = ( 'MAC_ADDR_TABLE static MACADDR vlan '
              'VLAN_ID ( drop | ( interface { INTFS } ) ) ' )
   noOrDefaultSyntax = ( 'MAC_ADDR_TABLE static '
                         'MACADDR vlan VLAN_ID '
                         '[ drop | ( interface { INTFS } ) ] ' )
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'static': staticKwMatcher,
      'MACADDR': MacAddr.MacAddrMatcher(),
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
      'drop': 'Drop frames',
      'interface': 'Interface',
      'INTFS': IntfRangeMatcher( 
                   explicitIntfTypes=EbraEthIntfCli.switchPortIntfTypes ), 
   }

   handler = setStatic
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( MacAddressStaticCmd )

#-------------------------------------------------------------------------------
# The "show mac address-table" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show mac address-table [static|dynamic|unicast|configured] [address <mac_addr>]
#                                            [interface <intf>] [vlan <vlan_id>]
#-------------------------------------------------------------------------------
def filterUnicastMacAddr( macEntry, intfSet, addrTypeFilter='' ):
   # Filter by intf
   if len( intfSet ) > 0 and \
          len( set( [ macEntry.intf ] ) & intfSet ) == 0:
      return False
   # Filter by type
   if addrTypeFilter is not None and addrTypeFilter != 'unicast':
      if addrTypeFilter == 'static':
         return macEntry.entryType in [ 'configuredStaticMac', 'peerStaticMac',
                                        'configuredRemoteMac',
                                        'peerConfiguredRemoteMac',
                                        'evpnConfiguredRemoteMac' ]
      elif addrTypeFilter == 'configured':
         return macEntry.entryType in [ 'configuredStaticMac',
                                        'configuredRemoteMac' ]
      else: 
         return macEntry.entryType in [ 'learnedDynamicMac', 'peerDynamicMac' ,
                                        'learnedRemoteMac', 'receivedRemoteMac',
                                        'peerLearnedRemoteMac', 
                                        'peerReceivedRemoteMac',
                                        'evpnDyamicRemoteMac',
                                        'peerEvpnRemoteMac',
                                        'configuredDynamicMac' ]

   # Don't filter by type
   else:
      return True

def filterMulticastMacAddr( macEntry, intfSet, addrTypeFilter='' ):
   # Do not return multicast entries if dynamic
   assert addrTypeFilter != 'unicast'
   if addrTypeFilter is not None and addrTypeFilter == 'dynamic':
      return False

   # Filter by intf
   if isinstance( intfSet, Intf.IntfRange.IntfList ):
      intfSet = set( list( intfSet.intfNames() ) )

   return not ( intfSet is not None and
                len( intfSet ) > 0 and
                len( set( macEntry.intf ) & intfSet ) == 0 )


def getStaticMulticastTable( mode, vlans, macAddr, intfSet, vlanConfigs, 
                             addrTypeFilter='' ):

   if mode.session_.guardsEnabled() \
          and not bridgingHwCapabilities.staticMcastSupported:
      return None
   
   multicastTable = MulticastMacAddressTable()

   for v in vlans:
      if isinstance( v, VlanCli.Vlan ):
         vlanId = v.id
      else:
         vlanId = int( v )
      vlanConfig = vlanConfigs.get( vlanId )
      if not vlanConfig and addrTypeFilter != "configured":
         continue
      fdbConfig = bridgingFdbConfig.fdbConfig.get( vlanId )

      if fdbConfig:
         groupTable = fdbConfig.ethGroup
      else:
         continue

      addresses = []

      if macAddr is not None:
         macEntry = groupTable.get( macAddr )
         if macEntry and filterMulticastMacAddr( macEntry, intfSet, 
               addrTypeFilter=addrTypeFilter ):
            addresses = [ macEntry ]
      else:
         addresses = ( addr for addr in groupTable.itervalues()
                       if filterMulticastMacAddr( addr, intfSet,
                          addrTypeFilter=addrTypeFilter ) )

      for addr in sorted( addresses, key=lambda x: x.addr ):
         printIntfs = Arnet.sortIntf( i for i in addr.intf )

         mac = addr.getRawAttribute( "addr" )  # Avoid string conversion.
         multicastTable.tableEntries.append(
            MulticastMacAddressTable.MulticastTableEntry( vlanId=vlanId,
                                                          macAddress=mac,
                                                          interfaces=printIntfs,
                                                          entryType="static" ) )
   return multicastTable

def doShowTable( mode, args ):
   showMacTable = Tac.newInstance( "ShowMacTableNs::ShowMacTable",
                                   cliConfig.force(),
                                   bridgingStatus,
                                   bridgingFdbConfig.force(),
                                   bridgingConfig.force() )

   # Pass the filters to C++
   vlan = args.get( 'VLAN_ID' )
   vlanId = vlan.id if vlan else 0
   showMacTable.vlanIdFilter = Tac.Value(
            'Bridging::VlanIdOrNone', vlanId )
   if args.get( 'MAC_ADDRESS' ):
      newMacFilter = Tac.Value( 'Arnet::EthAddr' )
      newMacFilter.stringValue = args[ 'MAC_ADDRESS' ]
      showMacTable.macFilter = newMacFilter
      showMacTable.macFilterPresent = True
   showMacTable.addrTypeFilter = args.get( 'ADDRTYPE_FILTER', '' )
   if args.get( 'INTFS' ):
      intfNames = set( list( args[ 'INTFS' ].intfNames() ) )
      for intfName in intfNames:
         intfId = Tac.Value( 'Arnet::IntfId' , intfName )
         showMacTable.intfFilter[ intfId ] = True

   # Should the multicast table be printed ?
   if not mode.session_.guardsEnabled() \
         or bridgingHwCapabilities.staticMcastSupported:
      showMacTable.populateMulticast = True

   # Get Mac addresses from other modules
   # for example, Vxlan has a list of MACs to be shown.
   if args.get( 'ADDRTYPE_FILTER' ) == 'configured':
      if args.get( 'VLAN_ID' ):
         vlans = []
         if bridgingFdbConfig.fdbConfig.has_key( args[ 'VLAN_ID' ].id ):
            vlans = [ args[ 'VLAN_ID' ] ]
      else:
         vlans = [ VlanCli.Vlan( vlan ) for vlan in bridgingFdbConfig.fdbConfig ]

      for v in vlans:
         notifiedAddresses = []
         for hook in bridgingExtraStaticMacHook.extensions():
            notifiedAddresses.extend( hook( None, v ) )
         if notifiedAddresses is not None:
            vlanObj = Tac.Value( 'Bridging::VlanId', v.id )
            notifiedMac = showMacTable.notifiedMac.newMember( vlanObj )
            count = 0
            for cHost in notifiedAddresses:
               notifiedMac.host[count] = cHost
               count = count + 1

   # Populate the entries
   # implemented in C++ for efficiency
   showMacTable.populateMacTblDisplayBuffer()

   # Print the entries
   # Implemented in C++ for efficiency
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   revision = mode.session_.requestedModelRevision()
   showMacTable.render( fd, fmt, revision )
   sys.stdout.flush()

   # Deferred Model.
   # Return the empty model without data
   return BridgingCliModel.MacAddressTable

class MacAddressTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show MAC_ADDR_TABLE ' 
              '[ ADDRTYPE_FILTER ] [ address MAC_ADDRESS ] [ interface INTFS ] ' 
              ' [ vlan VLAN_ID ]' )
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'ADDRTYPE_FILTER': CliMatcher.EnumMatcher( {
         'static': 'Static entry type',
         'dynamic': 'Dynamic entry type',
         'unicast': 'Unicast entry type',
         'configured': 'Configured MAC entries',
      } ),
      'address': 'Address keyword',
      'MAC_ADDRESS': MacAddr.MacAddrMatcher(),
      'interface': 'Interface',
      'INTFS': IntfRangeMatcher(),
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
   }

   cliModel = MacAddressTable
   handler = doShowTable

BasicCli.addShowCommandClass( MacAddressTableCmd )

#-------------------------------------------------------------------------------
# The "clear mac address-table" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   clear ( ( mac address-table ) | mac-address-table ) dynamic
#         [ ( [ vlan VLANID ] [ interface ( ETHINTF | ETHSUBINTF | VXLANINTF ) ] ) |
#           ( vlan VLANID address MACADDR ) ]
#
# EbraEnableMode adds ETHINTF (switchPortMatcher comprising Eth, Lag, and Recirc)
# and ETHSUBINTF, while VxlanEnableMode adds VXLANINTF
#-------------------------------------------------------------------------------

def doMacAddressTableClear( mode, args ):
   key = "%d.%f" % ( os.getpid(), Tac.now() )
   value = Tac.Value( "Bridging::HostTableFlushRequest" )
   vlanLog = "all vlans"
   vlanId = args.get( 'VLANID' )
   if vlanId is not None:
      vlanLog = "vlan %d" % vlanId.id
      value.vlanId = vlanId.id
   intfLog = "all interfaces"
   ethIntf = args.get( 'ETHINTF' )
   ethSubIntf = args.get( 'ETHSUBINTF' )
   vxlanIntf = args.get( 'VXLANINTF' )
   intf = ethIntf or ethSubIntf or vxlanIntf
   if intf is not None:
      intfLog = "interface " + intf.name
      value.intf = intf.name
   addrLog = "all addresses"
   macAddr = args.get( 'MACADDR' )
   if macAddr is not None:
      addrLog = "address " + macAddr
      value.addr = macAddr
   flushConfig.hostTableFlushRequest[ key ] = value

   info = UtmpDump.getUserInfo()
   Logging.log( EbraLogMsgs.ETH_MAC_ADDR_TABLE_CLEAR,
                intfLog, vlanLog, addrLog,
                info[ 'user' ], info[ 'tty' ], info[ 'ipAddr'] )

#-------------------------------------------------------------------------------
# The "show mac address-table count" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show mac address-table count [vlan <vlan_id>]
#-------------------------------------------------------------------------------
def getCountersViaSmash( vlans ):
   counters = {}
   vlanConfigs = bridgingConfig.vlanConfig

   # initialize the counters dictionary with the valid vlans
   for v in vlans:
      vlanConfig = vlanConfigs.get( v.id )
      if not vlanConfig:
         continue

      counters[ v.id ] = { 'dynamic': 0, 'static': 0, 'multicast': 0 }
      fdbConfig = bridgingFdbConfig.fdbConfig.get( v.id )
      if fdbConfig:
         counters[ v.id ][ 'multicast' ] = len( fdbConfig.ethGroup )

   # iterate through the entire Smash table and populate the counters dictionary
   for entry in bridgingStatus.smashFdbStatus.itervalues():
      if counters.has_key( entry.fid ):
         if entry.entryType in [ 'configuredStaticMac', 'peerStaticMac', 
                                 'configuredRemoteMac', 'peerConfiguredRemoteMac',
                                 'evpnConfiguredRemoteMac', 'evpnIntfStaticMac' ]:
            counters[ entry.fid ][ 'static' ] += 1
         else:
            counters[ entry.fid ][ 'dynamic' ] += 1

   return counters

def doShowCount( mode, args):
   ret = MacAddressTableCount()
   if args.get( 'VLAN_ID' ):
      if args[ 'VLAN_ID' ].lookup( mode ):
         vlans = [ args[ 'VLAN_ID' ] ]
      else:
         # Just fail silently if the VLAN doesn't exist.  This is compatible
         # with the industry-standard.
         return ret
   else:
      vlans = VlanCli.Vlan.getAll( mode, secondaryVlans=False )

   # get counters through smash
   smashCounters = getCountersViaSmash( vlans )

   for v in vlans:
      counters = smashCounters.get( v.id )
      if counters is None:
         continue

      vlanCount = MacAddressTableCount.VlanCount()
      vlanCount.dynamic = counters[ 'dynamic' ]
      vlanCount.unicast = counters[ 'static' ]
      vlanCount.multicast = counters[ 'multicast' ]
      ret.vlanCounts[ v.id ] = vlanCount

   return ret

class MacAddressTableCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE count [ vlan VLAN_ID ]'
   data = { 
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'count': 'Total number of MAC entries',
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
   }

   cliModel = MacAddressTableCount
   handler = doShowCount

BasicCli.addShowCommandClass( MacAddressTableCountCmd )     

#-------------------------------------------------------------------------------
# The "show interfaces [<intf>] switchport" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show interfaces [<intf>] switchport
#   show interfaces switchport module <mod>
#   show interfaces module <mod> switchport
#-------------------------------------------------------------------------------
def doShowSwitchport( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   def convertToPrintForm( allowedVlans ):
      if allowedVlans == '':
         return 'NONE'
      elif allowedVlans == '1-4094':
         return 'ALL'
      else:
         return allowedVlans
      
   switchPorts = SwitchPorts()
   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   if intfs is None:
      return
   
   # Note that if no interface name is specified, we only show the interfaces that
   # are configured as switchports.  However, if an interface name is specified, we
   # show that interface regardless of whether it is configured as a switchport.
   if intf is None:
      print 'Default switchport mode: %s' % cliConfig.defaultSwitchportMode
      if cliConfig.defaultPhoneVlan:
         print 'Default switchport phone VLAN: %d' % cliConfig.defaultPhoneVlan
      print 'Default switchport phone COS: %d' % \
             cliConfig.defaultPhoneCos
      print 'Default switchport phone trunk tagging: %s' % ( 'untagged' if 
            cliConfig.defaultPhoneTrunkUntagged else 'tagged' )
      print
      intfs = [ i for i in intfs if switchIntfConfigIfEnabled( mode, i ) ]

   vlanTagFormatDrop = {}
   vlanTagRequiredDropFormats = \
      Tac.newInstance( "Bridging::VlanTagRequired" ).vlanTagFormatDrop
   vlanTagDisallowedDropFormats = \
      Tac.newInstance( "Bridging::VlanTagDisallowed" ).vlanTagFormatDrop
   for vlanTagFormatDropStatus in vlanTagFormatDropStatusDir.entityPtr.values():
      for intf, status in \
            vlanTagFormatDropStatus.intfVlanTagFormatDropStatus.items():
         vlanTagFormatDrop[ intf ] = dict( status.vlanTagFormatDrop.items() )

   for intf in intfs:
      if PortChannelIntfId.isPortChannelIntfId( intf.name ):
         # get active lagMembers for port channel
         # we cannot do Tac.Type( "Lag::LagStatusFilter" ).filterOnActive here 
         # as we cannot depend on Lag package
         lagMembers = \
               EbraCliLib.getLagMembersCallBack( # pylint: disable-msg=not-callable
               mode, intf.name, lacpOnly=False, status='filterOnActive' ).keys()
         vlanTagFormatDrop[ intf.name ] = {}
         if lagMembers:
            for tagFormat in vlanTagRequiredDropFormats:
               if all( vlanTagFormatDrop.get( m, {} ).get( tagFormat, False )
                       for m in lagMembers ):
                  vlanTagFormatDrop[ intf.name ][ tagFormat ] = True
            for tagFormat in vlanTagDisallowedDropFormats:
               if all( vlanTagFormatDrop.get( m, {} ).get( tagFormat, False )
                       for m in lagMembers ):
                  vlanTagFormatDrop[ intf.name ][ tagFormat ] = True
            for p in lagMembers:
               vlanTagFormatDrop.pop( p, None )

   for i in intfs:
      switchport = SwitchPort()
      switchIntfConfig = switchIntfConfigIfEnabled( mode, i )
      switchIntfBridgeConfig = switchIntfBridgeConfigIfEnabled( mode, i )
      switchport.enabled = True if switchIntfConfig else False

      if switchIntfConfig:
         switchportInfo = SwitchPortInfo()
         accessVlan = VlanCli.Vlan( switchIntfConfig.accessVlan )
         if accessVlan.lookup( mode ):
            accessVlanName = accessVlan.name( mode )
         else:
            accessVlanName = 'inactive'

         trunkNativeVlan = VlanCli.Vlan( switchIntfConfig.trunkNativeVlan )
         if trunkNativeVlan.lookup( mode ):
            trunkNativeVlanName = trunkNativeVlan.name( mode )
         else:
            trunkNativeVlanName = 'inactive'

         trunkAllowedVlans = convertToPrintForm( switchIntfConfig.trunkAllowedVlans )

         switchportInfo.mode = switchIntfConfig.switchportMode
         switchportInfo.phoneTrunk = switchIntfConfig.phoneTrunk
         switchportInfo.macLearning = switchIntfConfig.macLearningEnabled

         if hasattr( i, 'intfId' ):
            intfName = i.intfId
         else:
            intfName = i.name

         if bridgingHwCapabilities.vlanTpidSupported:
            tpid = switchIntfConfig.vlanTpid
            if not intfName.startswith( 'Et' ):
               status = None
            else:
               e = intfTpidStatus.entry.get( intfName )
               if e is None:
                  if tpid == defaultTpid:
                     status = True
                  else:
                     status = None
               else:
                  status = ( e.tpid == tpid and e.active )
            switchportInfo.tpid = hex( tpid )
            if status is not None:
               switchportInfo.tpidStatus = status
         else:
            # Set the default TPID ( 0x8100 )
            switchportInfo.tpid = hex( defaultTpid )
            switchportInfo.tpidStatus = True

         # Set dot1qVlanTagRequired and dot1qVlanTagRequiredStatus
         switchportInfo.dot1qVlanTagRequired = all(
            switchIntfConfig.vlanTagFormatDrop.get( tagFormat, False ) \
               for tagFormat in vlanTagRequiredDropFormats )
         dot1qVlanTagRequiredStatus = False
         dot1qVlanTagDisallowed = False
         vlanTagFormatDropStatus = vlanTagFormatDrop.get( intfName, None )
         if vlanTagFormatDropStatus:
            # Checks if vlanTagFormatDropStatus has all vlanTagRequiredDropFormats
            # vlanTagRequiredDropFormats collection has priority and untagged
            # set to true
            # If they are present then dot1qVlanTagRequiredStatus is set to true
            dot1qVlanTagRequiredStatus = all(
               vlanTagFormatDropStatus.get( tagFormat,
                  False ) for tagFormat in vlanTagRequiredDropFormats ) 
            # Checks if vlanTagFormatDropStatus has all vlanTagDisallowedDropFormats
            # vlanTagDisallowedDropFormats collection has singleTagged set to true
            # If they are present then dot1qVlanTagDisallowed is set to true
            dot1qVlanTagDisallowed = all(
               vlanTagFormatDropStatus.get( tagFormat,
                  False ) for tagFormat in vlanTagDisallowedDropFormats )
         switchportInfo.dot1qVlanTagRequiredStatus = dot1qVlanTagRequiredStatus
         switchportInfo.dot1qVlanTagDisallowed = dot1qVlanTagDisallowed

         # BUG210 The operational mode should be 'down' if the port is down.
         switchportInfo.accessVlanId = accessVlan.id
         switchportInfo.accessVlanName = accessVlanName
         switchportInfo.trunkingNativeVlanId = trunkNativeVlan.id
         if trunkNativeVlan.id != 0:
            switchportInfo.trunkingNativeVlanName = trunkNativeVlanName

         if bridgingHwCapabilities.privateVlansSupported:
            if not switchIntfConfig.defaultPrivateVlanMapping:
               mappedPrivateVlans = vlanSetToCanonicalString(
                  switchIntfConfig.privateVlanMapping )
               switchportInfo.privateVlanMapping = \
                     convertToPrintForm( mappedPrivateVlans )

         dynamicTrunkGroupNames = []

         # Make a list of dynamic allowed vlans for this interface from different
         # agents. Also concatanate these vlans strings into a single allowed vlan 
         # string.
         dynAllowedVlans = { }
         allVlans = switchIntfConfig.trunkAllowedVlans
         intfId = i.name
         isRecirc = Tac.Type( "Arnet::PortChannelIntfId" ).isRecirc( intfId )
         isUnconnectedEth = Tac.Type( "Arnet::EthIntfId" ).isUnconnected( intfId ) 
         noDynamicInfo = i.isSubIntf() or isRecirc or isUnconnectedEth
         if not noDynamicInfo:
            for key in dynAllowedVlanDir.entityPtr:
               entity = dynAllowedVlanDir.entityPtr.get( key, None )
               if entity:
                  for groupName, tg in entity.trunkGroup.iteritems():
                     if i.name in tg.intfList:
                        dynamicTrunkGroupNames.append( groupName )
                  allowedVlans = entity.vlans.get( i.name, None )
                  if allowedVlans is not None and allowedVlans != "":
                     if allVlans != "":
                        allVlans += ","
                     allVlans += allowedVlans
                     canonicalStr = vlanSetToCanonicalString( 
                        computeVlanRangeSet( allowedVlans ) )
                     dynAllowedVlans[ key ] = convertToPrintForm( canonicalStr )

         # If there were any dynamic allowed vlans
         # then print union of dynamic and static allowed vlans
         # as the VLANs enabled on this interface and 
         # also print each set separately.
         if len( dynAllowedVlans ):
            switchportInfo.staticAllowedVlans = trunkAllowedVlans
            trunkAllowedVlans = vlanSetToCanonicalString( 
               computeVlanRangeSet( allVlans ) )
            trunkAllowedVlans = convertToPrintForm( trunkAllowedVlans )
            dynVlans = {}
            for key in dynAllowedVlans:
               dynVlans[ key ] = dynAllowedVlans[ key ]
            switchportInfo.dynamicAllowedVlans = dynVlans

         switchportInfo.trunkAllowedVlans = trunkAllowedVlans

         # BUG3226 Need to re-enable this output when the 'switchport block' commands
         # are re-enabled.
         #print 'Unknown unicast blocked: %s' % blockUnicastStr
         #print 'Unknown multicast blocked: %s' % blockMulticastStr

         staticTrunkGroupNames = switchIntfConfig.trunkGroup.keys()
         staticTrunkGroupNames.sort()
         switchportInfo.staticTrunkGroups = staticTrunkGroupNames
         dynamicTrunkGroupNames.sort()
         switchportInfo.dynamicTrunkGroups = dynamicTrunkGroupNames
         switchportInfo.phoneVlan = switchIntfBridgeConfig.phoneVlan
         switchportInfo.phoneTrunkUntagged = (
              switchIntfBridgeConfig.phoneTrunkTagMode == 'phoneTrunkUntagged' )

         if not bridgingHwCapabilities.sourceportFilterSupported:
            switchportInfo.sourceportFilterMode = 'enabled'
         else:
            if switchIntfConfig.sourceportFilterMode == 'sourceportFilterEnabled':
               switchportInfo.sourceportFilterMode = 'enabled'
            else:
               switchportInfo.sourceportFilterMode = 'disabled'
         if switchIntfConfig.vlanForwardingMode == \
               'vlanForwardingModeAllowedVlansOnPort':
            switchportInfo.vlanForwardingMode = allowedVlansOnlyStr
         else:
            switchportInfo.vlanForwardingMode = allConfiguredVlansStr
         switchport.switchportInfo = switchportInfo
      switchPorts.switchports[ i.name ] = switchport
   return switchPorts

def switchIntfBridgeConfigIfEnabled( mode, intf ):
   switchIntfConfigs = bridgingConfig.switchIntfConfig

   if hasattr( intf, 'intfid' ):
      switchIntfConfig = switchIntfConfigs.switchIntfConfig.get( intf.intfid )
   else:
      switchIntfConfig = switchIntfConfigs.get( intf.name )
   return switchIntfConfig

# If the SwitchInterfaceConfig for the given interface is present and not
# disabled, return it.  Otherwise return None.
def switchIntfConfigIfEnabled( mode, intf ):
   switchIntfConfigs = bridgingSwitchIntfConfig
   # this function is called with different types of intfs, some have the 
   # intfId attribute, use this attr it if has it instead of name
   #BUG27653
   if hasattr( intf, 'intfId' ):
      switchIntfConfig = switchIntfConfigs.switchIntfConfig.get( intf.intfId )
   else:
      switchIntfConfig = switchIntfConfigs.switchIntfConfig.get( intf.name )
   enabled = switchIntfConfig and switchIntfConfig.enabled
   if enabled:
      return switchIntfConfig
   else:
      return None

class ShowIntfSwitchport( IntfCli.ShowIntfCommand ):
   syntax = "show interfaces switchport"
   data = dict( switchport='Details on switchports' )
   cliModel = SwitchPorts
   handler = doShowSwitchport
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfSwitchport )

#-------------------------------------------------------------------------------
# The "show dot1q-tunnel [interface <intf>]" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show dot1q-tunnel [interface <intf>]
#-------------------------------------------------------------------------------


def doShowDot1QTunnel( mode, args ):
   ret = Dot1QTunnel()
   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTERFACE' ) )
   if intfs is None:
      return ret
   
   # Note that if no interface name is specified, we only show the interfaces that
   # are configured as switchports.  However, if an interface name is specified, we
   # show that interface regardless of whether it is configured as a switchport.
   def intfIsDot1QTunnel( i ):
      switchIntfConfig = switchIntfConfigIfEnabled( mode, i )
      return switchIntfConfig and switchIntfConfig.switchportMode == 'dot1qTunnel'
   intfs = filter( intfIsDot1QTunnel, intfs )

   for intfName in Arnet.sortIntf( i.name for i in intfs ):
      ret.interfaces.append( intfName )

   return ret

class Dot1QTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1q-tunnel [ interface INTERFACE ]'
   data = {
         'dot1q-tunnel': 'Details for 802.1q tunnels',
         'interface': 'Interface keyword',
         'INTERFACE': IntfRangeMatcher(),
         }
   handler = doShowDot1QTunnel
   cliModel = Dot1QTunnel

BasicCli.addShowCommandClass( Dot1QTunnelCmd )

#-------------------------------------------------------------------------------
# The "switch forwarding-mode cut-through | store-and-forward" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   switch forwarding-mode cut-through
#   switch forwarding-mode store-and-forward
#
#-------------------------------------------------------------------------------
# we only enable the command if the platform supports it. 

SwitchForwardingMode = Tac.Type( "Bridging::SwitchForwardingMode" )

def supportedForwardingModeGuard( mode, token ):
   if len( bridgingHwCapabilities.supportedForwardingMode ) >= 1:
      return None
   return CliParser.guardNotThisPlatform

def storeAndForwardGuard( mode, token ):
   if len( bridgingHwCapabilities.supportedForwardingMode ) and \
          ( SwitchForwardingMode.storeAndForward in 
            bridgingHwCapabilities.supportedForwardingMode.values() ):
      return None
   return CliParser.guardNotThisPlatform

def cutThroughGuard( mode, token ):
   if len( bridgingHwCapabilities.supportedForwardingMode ) and \
          ( SwitchForwardingMode.cutThrough in 
            bridgingHwCapabilities.supportedForwardingMode.values() ):
      return None
   return CliParser.guardNotThisPlatform

def setCutThrough( mode, args ):
   if bridgingHwCapabilities.supportedForwardingMode and \
          bridgingHwCapabilities.supportedForwardingMode[ 0 ] == \
          SwitchForwardingMode.cutThrough:
      cliConfig.forwardingMode = SwitchForwardingMode.defaultMode
   else:
      cliConfig.forwardingMode = SwitchForwardingMode.cutThrough

def setStoreAndForward( mode, args ):
   if bridgingHwCapabilities.supportedForwardingMode and \
          bridgingHwCapabilities.supportedForwardingMode[ 0 ] == \
          SwitchForwardingMode.storeAndForward:
      cliConfig.forwardingMode = SwitchForwardingMode.defaultMode
   else:
      cliConfig.forwardingMode = SwitchForwardingMode.storeAndForward


matcherForwardingMode = CliCommand.guardedKeyword( 'forwarding-mode',
                            helpdesc='Set the switch forwarding mode',
                            guard=supportedForwardingModeGuard )

class CutThroughCmd( CliCommand.CliCommandClass ):
   syntax = 'switch forwarding-mode cut-through'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'forwarding-mode' : matcherForwardingMode,
      'cut-through' : CliCommand.guardedKeyword( 'cut-through',
                         helpdesc='Set the switch to cut-through packets',
                         guard=cutThroughGuard ),
      }

   handler = setCutThrough

BasicCli.GlobalConfigMode.addCommandClass( CutThroughCmd )

class StoreAndForwardCmd( CliCommand.CliCommandClass ):
   syntax = 'switch forwarding-mode store-and-forward'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'forwarding-mode' : matcherForwardingMode,
      'store-and-forward' : CliCommand.guardedKeyword( 'store-and-forward',
                               helpdesc='Set the switch to store-and-forward '
                               'packets', guard=storeAndForwardGuard ),
      }

   handler = setStoreAndForward

BasicCli.GlobalConfigMode.addCommandClass( StoreAndForwardCmd )

#-------------------------------------------------------------------------------
# The "switch forwarding-mode store-and-forward multicast" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   switch forwarding-mode store-and-forward multicast
#
#-------------------------------------------------------------------------------
# we only enable the command if the platform supports it. 

def mcastForwardingModeGuard( mode, token ):
   if len( bridgingHwCapabilities.supportedL3McastForwardingMode ):
      return None
   return CliParser.guardNotThisPlatform

def mcastStoreAndForwardGuard( mode, token ):
   if len( bridgingHwCapabilities.supportedL3McastForwardingMode ) and \
          ( SwitchForwardingMode.storeAndForward in
            bridgingHwCapabilities.supportedL3McastForwardingMode.values() ):
      return None
   return CliParser.guardNotThisPlatform


def setMcastStoreAndForward( mode, args ):
   cliConfig.l3McastForwardingMode = SwitchForwardingMode.storeAndForward

class McastStoreAndForwardCmd( CliCommand.CliCommandClass ):
   syntax = 'switch forwarding-mode store-and-forward multicast'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'forwarding-mode' : matcherForwardingMode,
      'store-and-forward' : CliCommand.guardedKeyword( 'store-and-forward',
                               helpdesc='Set the switch to store-and-forward '
                               'packets', guard=mcastStoreAndForwardGuard,
                                hidden=True ),
      'multicast' : CliCommand.guardedKeyword( 'multicast',
                       helpdesc='Set the forwarding mode for multicast packets only',
                       guard=mcastForwardingModeGuard, hidden=True ),
      }
   handler = setMcastStoreAndForward

BasicCli.GlobalConfigMode.addCommandClass( McastStoreAndForwardCmd )

#-------------------------------------------------------------------------------
# The "[no|default] switch forwarding-mode [store-and-forward] [multicast]" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   [no|default] switch forwarding-mode
#   [no|default] switch forwarding-mode store-and-forward multicast
#
#-------------------------------------------------------------------------------
# we only enable the multicast option if the platform supports it

def noForwardingMode( mode, args ):
   if 'store-and-forward' in args and 'multicast' in args:
      cliConfig.l3McastForwardingMode = SwitchForwardingMode.defaultMode
   else:
      cliConfig.forwardingMode = SwitchForwardingMode.defaultMode

class NoForwardingModeCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'switch forwarding-mode [ store-and-forward ] [ multicast ]'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'forwarding-mode' : 'Set switch forwarding to the default mode',
      'store-and-forward' : CliCommand.guardedKeyword( 'store-and-forward',
                               helpdesc='Set the switch to store-and-forward '
                               'packets', guard=mcastStoreAndForwardGuard ),
      'multicast' : CliCommand.guardedKeyword( 'multicast',
                       helpdesc='Set the forwarding mode for multicast packets only',
                       guard=mcastForwardingModeGuard ),
      }

   noOrDefaultHandler = noForwardingMode

BasicCli.GlobalConfigMode.addCommandClass( NoForwardingModeCmd )

#-------------------------------------------------------------------------------
# The "show switch forwarding-mode" command
#
# The full syntax of this command is:
#
#   show switch forwarding-mode
#-------------------------------------------------------------------------------

def doShowForwardingMode( mode, args ):
   ret = BridgingCliModel.SwitchForwardingMode()
   ret.currentMode = bridgingConfig.forwardingMode
   for supportedMode in bridgingHwCapabilities.supportedForwardingMode.values():
      ret.supportedModes.append( ret.SupportedMode( supportedMode=supportedMode ) )
   ret.currentL3McastMode = bridgingConfig.l3McastForwardingMode
   for supportedMode in \
         bridgingHwCapabilities.supportedL3McastForwardingMode.values():
      SupportedL3McastMode = ret.SupportedMode( supportedMode=supportedMode )
      ret.supportedL3McastModes.append( SupportedL3McastMode )
   return ret

class ShowForwardingModeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show switch forwarding-mode'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForShow,
      'forwarding-mode' : 'Show switching mode configuration',
      }

   handler = doShowForwardingMode
   cliModel = BridgingCliModel.SwitchForwardingMode

BasicCli.addShowCommandClass( ShowForwardingModeCmd )

#-------------------------------------------------------------------------------
# The "switchport default mode access | routed" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   switchport default mode access
#   switchport default mode routed
#-------------------------------------------------------------------------------
# we only enable the command if the platform supports it. 

SwitchportMode = Tac.Type( "Bridging::SwitchportMode" )

def setSwitchportDefaultMode( mode, args ):
   if args[ 'MODE_TYPE' ] == 'access':
      cliConfig.defaultSwitchportMode = SwitchportMode.access
   else:
      cliConfig.defaultSwitchportMode = SwitchportMode.routed

class SwitchportDefaultModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default mode MODE_TYPE'
   data = {
      'switchport': switchportMatcher,
      'default': 'Configure switchport default behavior',
      'mode': 'Set the switchport default mode',
      'MODE_TYPE': CliMatcher.EnumMatcher( {
         'access': 'Set the switchport default mode to access',
         'routed': 'Set the switchport default mode to routed',
      } ),
   }

   handler = setSwitchportDefaultMode

BasicCliModes.GlobalConfigMode.addCommandClass( SwitchportDefaultModeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] mac address-table reserved forward" command, in "config mode".
#
# The full syntax of this command is:
#
#    mac address-table reserved forward <mac_addr>
#    no mac address-table reserved forward <mac_addr>
#    default mac address-table reserved forward <mac_addr>
#-------------------------------------------------------------------------------
def ieeeReservedMacForwardSupportedGuard( mode, token ):
   if ( bridgingHwCapabilities.ieeeReservedMacForwardAddrSupported or
        bridgingHwCapabilities.ieeeReservedMacForwardGroup ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def ieeeReservedMacForwardAddrSupportedGuard( mode, token ):
   if bridgingHwCapabilities.ieeeReservedMacForwardAddrSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def ieeeReservedMacForwardGroupSupportedGuard( mode, token ):
   if bridgingHwCapabilities.ieeeReservedMacForwardGroup:
      return None
   else:
      return CliParser.guardNotThisPlatform

def shouldSetIeeeReservedMac( token ):
   # Check to make sure macAddrAll and individual entries are mutually exclusive
   ethAddrAll = Arnet.EthAddr( EbraLib.ieeeReservedConfigAll )
   if token == EbraLib.ieeeReservedTokenAll:
      if cliConfig.ieeeReservedForwarding:
         return False
   else:
      if ethAddrAll in cliConfig.ieeeReservedForwarding:
         return False
   return True

def shouldNoIeeeReservedMac( token ):
   # Check to make sure macAddrAll and individual entries are mutually exclusive
   ethAddrAll = Arnet.EthAddr( EbraLib.ieeeReservedConfigAll )
   if token == EbraLib.ieeeReservedTokenAll:
      if ethAddrAll not in cliConfig.ieeeReservedForwarding:
         return False
   else:
      if ethAddrAll in cliConfig.ieeeReservedForwarding:
         return False
   return True

def _ieeeReservedAddrForConfToken( mode, token ):
   if token == EbraLib.ieeeReservedTokenAll:
      macAddr = EbraLib.ieeeReservedConfigAll
   elif EbraLib.isIeeeReservedAddressToken( token ):
      macAddr = EbraLib.ieeeReservedAddressTokenToConfig( token )
   elif EbraLib.isIeeeReservedGroupToken( token ):
      macAddr = EbraLib.ieeeReservedGroupTokenToConfig( token )
   else:
      assert False, "Invalid reserved address token: " + repr( token )
   return Arnet.EthAddr( macAddr )

def setIeeeReservedMac( mode, token ):
   # For address ranges (except the special range 01-0f), the platform will
   # accept any address in the range.
   macAddr = _ieeeReservedAddrForConfToken( mode, token )
   cliConfig.ieeeReservedForwarding[ macAddr ] = True

def noIeeeReservedMac( mode, token ):
   macAddr = _ieeeReservedAddrForConfToken( mode, token )
   if macAddr in cliConfig.ieeeReservedForwarding:
      del cliConfig.ieeeReservedForwarding[ macAddr ]

def setIeeeReservedMacCommon( mode, args ):
   mac = args.get( 'MAC_ADDR' )
   if mac is None or not shouldSetIeeeReservedMac( mac ):
      return
   return setIeeeReservedMac( mode, mac )

def noIeeeReservedMacCommon( mode, args ):
   mac = args.get( 'MAC_ADDR' )
   if mac is None or not shouldNoIeeeReservedMac( mac ):
      return
   return noIeeeReservedMac( mode, mac )

class macAddressExpression( CliCommand.CliExpression ):
   expression = 'MAC_FORWARD_NODE | MAC_FORWARD_ADDR_NODE | MAC_FORWARD_GROUP_NODE'
   data = {
           'MAC_FORWARD_NODE': CliCommand.Node(
                matcher=CliMatcher.EnumMatcher( {
                           '0180.c200.0001-0180.c200.000f': 'Forward all frames '
                                'destined to 0180.c200.0001 to 0180.c200.000f',
                           '0180.c200.0001': 'Forward all frames destined to '
                                '0180.c200.0001 except Ethernet Pause frames',
                           '0180.c200.0002': 'Forward all frames destined to '
                                '0180.c200.0002',
                } ),
                alias='MAC_ADDR',
                guard=ieeeReservedMacForwardSupportedGuard ),
           'MAC_FORWARD_ADDR_NODE': CliCommand.Node(
                matcher=CliMatcher.EnumMatcher( {
                           '0180.c200.0003': 'Forward all frames destined to '
                                '0180.c200.0003',
                           '0180.c200.0004': 'Forward all frames destined to '
                                '0180.c200.0004',
                           '0180.c200.0005': 'Forward all frames destined to '
                                '0180.c200.0005',
                           '0180.c200.0006': 'Forward all frames destined to '
                                '0180.c200.0006',
                           '0180.c200.0007': 'Forward all frames destined to '
                                '0180.c200.0007',
                           '0180.c200.0008': 'Forward all frames destined to '
                                '0180.c200.0008',
                           '0180.c200.0009': 'Forward all frames destined to '
                                '0180.c200.0009',
                           '0180.c200.000a': 'Forward all frames destined to '
                                '0180.c200.000a',
                           '0180.c200.000b': 'Forward all frames destined to '
                                '0180.c200.000b',
                           '0180.c200.000c': 'Forward all frames destined to '
                                '0180.c200.000c',
                           '0180.c200.000d': 'Forward all frames destined to '
                                '0180.c200.000d',
                           '0180.c200.000e': 'Forward all frames destined to '
                                '0180.c200.000e',
                           '0180.c200.000f': 'Forward all frames destined to '
                                '0180.c200.000f',
                } ),
                alias='MAC_ADDR',
                guard=ieeeReservedMacForwardAddrSupportedGuard ),
           'MAC_FORWARD_GROUP_NODE': CliCommand.Node(
                matcher=CliMatcher.EnumMatcher( {
                           '0180.c200.0003,0180.c200.000e': 'Forward all frames '
                                'destined to 0180.c200.0003, and 0180.c200.000e',
                           '0180.c200.0004-0180.c200.000d,0180.c200.000f':
                                'Forward all frames destined to 0180.c200.0004 '
                                'to 0180.c200.000d, and 0180.c200.000f',
                } ),
                alias='MAC_ADDR',
                guard=ieeeReservedMacForwardGroupSupportedGuard ),
         }

class ReservedForwardCmd( CliCommand.CliCommandClass ):
   syntax = 'MAC_ADDR_TABLE reserved forward MAC_ADDR'
   noOrDefaultSyntax = syntax
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'reserved': CliCommand.guardedKeyword( 'reserved',
                     helpdesc='Reserved',
                     guard=ieeeReservedMacForwardSupportedGuard ),
      'forward': 'Forward',
      'MAC_ADDR': macAddressExpression,
   }

   handler = setIeeeReservedMacCommon
   noOrDefaultHandler = noIeeeReservedMacCommon

BasicCliModes.GlobalConfigMode.addCommandClass( ReservedForwardCmd )

#-------------------------------------------------------------------------------
# The "[no|default] mac pause-frame pass-through" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   mac pause-frame pass-through
#   no mac pause-frame pass-through
#   default mac pause-frame pass-through
#-------------------------------------------------------------------------------


# we only enable the command if the platform supports it. 

matcherPauseFrame = CliMatcher.KeywordMatcher( 'pause-frame',
                    helpdesc='Pause frame visibility' )

def supportedPausePassThroughGuard( mode, token ):
   if bridgingHwCapabilities.pausePassThroughSupported:
      return None
   return CliParser.guardNotThisPlatform

def setPauseFramePassThrough( mode, args ):
   cliConfig.pausePassThrough = True

def noPauseFramePassThrough( mode, args ):
   cliConfig.pausePassThrough = False

class MacPauseFramePassThroughCmd( CliCommand.CliCommandClass ):
   syntax = 'mac pause-frame pass-through'
   noOrDefaultSyntax = syntax
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'pause-frame': CliCommand.Node( matcher=matcherPauseFrame,
                     guard=supportedPausePassThroughGuard ),
      'pass-through': ( 'Set pause frame to be visible '
                        'across ethernet MAC chip pipeline' )
   }
   handler = setPauseFramePassThrough
   noOrDefaultHandler = noPauseFramePassThrough

BasicCliModes.GlobalConfigMode.addCommandClass( MacPauseFramePassThroughCmd )

#-------------------------------------------------------------------------------
# The "show mac pause-frame" command
#
# The full syntax of this command is:
#
#   show mac pause-frame
#-------------------------------------------------------------------------------

def doShowPauseFramePassThrough( mode, args ):
   ret = BridgingCliModel.PauseFramePassThrough()
   ret.pauseFramePassThrough = bridgingConfig.pausePassThrough
   return ret

class MacPauseFrameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac pause-frame'
   data = {
      'mac': CliToken.Mac.macMatcherForShow,
      'pause-frame': matcherPauseFrame
   }
   handler = doShowPauseFramePassThrough
   cliModel = BridgingCliModel.PauseFramePassThrough

BasicCli.addShowCommandClass( MacPauseFrameCmd )

def initFlushReplySm():
   global flushReplySm
   if flushReplySm is None:
      flushReplySm = Tac.newInstance( "Ebra::HostTableFlushReplySm", 
                                      flushConfig, flushReply, redundancyStatus )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig, bridgingConfig, bridgingStatus, bridgingSwitchIntfConfig
   global bridgingHwCapabilities
   global intfTpidStatus
   global flushConfig
   global flushReply
   global redundancyStatus
   global dynAllowedVlanDir
   global vlanTagFormatDropStatusDir
   global bridgingFdbConfig

   mount = LazyMount.mount
   cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli", 
                                  "Bridging::Input::CliConfig", "w" )
   bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                                   "Bridging::HwCapabilities", "r" )
   bridgingConfig = mount( entityManager, "bridging/config",
                           "Bridging::Config", "r" )
   bridgingSwitchIntfConfig = mount( entityManager, "bridging/switchIntfConfig",
                                     "Bridging::SwitchIntfConfigDir", "r" )
   intfTpidStatus = mount( entityManager, "bridging/tpid/status",
                           "Bridging::IntfTpidStatus", "r" )
   bridgingFdbConfig = mount( entityManager, "bridging/fdbConfigDir",
                              "Bridging::FdbConfigDir", "r" )
   dynAllowedVlanDir = mount( entityManager, "bridging/input/dynvlan/allowedvlan",
                              "Tac::Dir", "ri" )
   vlanTagFormatDropStatusDir = mount( entityManager, 
                                       "bridging/vlanTagFormatDrop/status",
                                       "Tac::Dir", "ri" )
   bridgingStatus = SmashLazyMount.mount( entityManager, "bridging/status",
                                          "Smash::Bridging::Status",
                                          SmashLazyMount.mountInfo( 'reader' ) )

   mg = entityManager.mountGroup()
   flushConfig = mg.mount( "bridging/flush/request/cli", 
                           "Bridging::HostTableFlushRequestDir", "w" )
   flushReply = mg.mount( "bridging/flush/reply/all",
                          "Bridging::HostTableFlushReplyDir", "r" )
   redundancyStatus = mg.mount( Cell.path( "redundancy/status" ),
                                "Redundancy::RedundancyStatus", "r" )
   mg.close( initFlushReplySm )

