# Copyright (c) 2005-2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
'boot port numbering' commands modify the 'PORT_NUMBERING_SCHEME'
setting stored in flash:boot-config.  These commands are special because they:

* modify boot-config immediately
* do not affect Sysdb state
* are not reflected in running-config or startup-config
"""

import BasicCli
import BasicCliModes
import BasicCliUtil
import Cell
import CliParser
import CliCommand
import CliMatcher
import CliPlugin.TechSupportCli
import EosVersion
import FileCli
import LazyMount
import ReloadCli
import SimpleConfigFile
import ShowCommand
from SysMgrModels import CapiStringToCliString, PortNumbering
import Tracing
import FileUrl
import errno, os

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

portNumberingInfo = None
entityMib = None

def legalSchemes():
   schemes = []
   if portNumberingInfo.multiplePortNumberingsSupported:
      schemes.append( '' )
      for portNumbering in portNumberingInfo.supportedPortNumbering.values():
         schemes.append( portNumbering.portType + '_' + portNumbering.schemeName )
   return schemes

def isPortTypeSupported( mode, token ):
   supportedPortNumberings = portNumberingInfo.supportedPortNumbering.values()
   for numbering in supportedPortNumberings:
      if( numbering.portType == token ):
         return None
   return CliParser.guardNotThisPlatform

def isSchemeNameSupported( mode, token ):
   supportedPortNumberings = portNumberingInfo.supportedPortNumbering.values()
   for numbering in supportedPortNumberings:
      if( numbering.portType == str( nodeQsfp.matcher_ ) and
          token == numbering.schemeName ):
         return None
   return CliParser.guardNotThisPlatform

def areMultiplePortNumberingsSupported( mode, token ):
   if portNumberingInfo.multiplePortNumberingsSupported:
      return None
   return CliParser.guardNotThisPlatform

def defaultPortNumbering():
   if( portNumberingInfo.multiplePortNumberingsSupported and
       portNumberingInfo.defaultPortNumbering ):
      scheme = portNumberingInfo.defaultPortNumbering
      return scheme.portType + '_' + scheme.schemeName
   else:
      return 'notAvailable'

def bootConfig( mode ):
   url = FileUrl.localBootConfig( mode.entityManager, mode.session_.disableAaa_ )
   bootConfigFilename = url.realFilename_
   t0( 'BootConfigFile is %s.' % bootConfigFilename )
   return SimpleConfigFile.SimpleConfigFileDict( bootConfigFilename,
                                                 createIfMissing=True,
                                                 autoSync=True )

def portNumberingDict():
   # The dictionary this function returns is guaranteed to have
   # 'PORT_NUMBERING_SCHEME': legal scheme/Unknown key, value entry.
   config = { 'PORT_NUMBERING_SCHEME' : 'Unknown' }
   portNumberingFileName = os.path.join( EosVersion.etcDir(), 'port-numbering' )

   try:
      fileContent = file( portNumberingFileName ).read()
      if fileContent:
         config = dict( line.split( '=', 1 )
               for line in fileContent.split( '\n' ) if '=' in line )
         if( not config.has_key( 'PORT_NUMBERING_SCHEME' ) or \
               config.get( 'PORT_NUMBERING_SCHEME' ) not in legalSchemes() ):
            config = { 'PORT_NUMBERING_SCHEME' : 'Unknown' }
   except IOError, e:
      # IOError will happen if /etc/port-numbering doesn't exist.
      if e.errno != errno.ENOENT:
         raise

   return config

matcherBoot = CliMatcher.KeywordMatcher( 'boot',
      helpdesc='System boot configuration' )
matcherNumbering = CliMatcher.KeywordMatcher( 'numbering',
      helpdesc='Modify port numbering' )
matcherPort = CliMatcher.KeywordMatcher( 'port',
      helpdesc='Modify port numbering' )
matcherQsfp = CliMatcher.KeywordMatcher( 'qsfp',
      helpdesc='Modify QSFP port numbering' )
nodePort = CliCommand.Node( matcher=matcherPort,
      guard=areMultiplePortNumberingsSupported )
nodeQsfp = CliCommand.Node( matcher=matcherQsfp, guard=isPortTypeSupported )

def eraseStartupConfigAndReloadConfirmation( mode, schemeName, portType,
      noOrDefault='' ):
   warning = ''
   if( not schemeName and not portType ):
      warning = 'This command will delete the startup-config and reboot\n' \
            'the system to make %s port numbering take effect.\n' % noOrDefault
   else:
      warning = 'This command will delete the startup-config and reboot\n' \
            'the system to make %s %s port numbering take effect.\n' % ( schemeName,
                  portType )
   
   prompt = 'Do you wish to proceed with this command? [y/n]'
   
   mode.addWarning( warning )
   return BasicCliUtil.confirm( mode, prompt )

#------------------------------------------------------------------------------------
# (Fixed)   boot port numbering qsfp dense | boot port numbering qsfp sparse
# (Modular) boot port numbering qsfp expanded
#------------------------------------------------------------------------------------
def setQsfpPortNumberingSchemeName( mode, args ):
   scheme = args[ 'SCHEME_NAME' ]
   numberingScheme = 'PORT_NUMBERING_SCHEME'
   config = bootConfig( mode )
   try:
      if scheme == 'dense':
         if config.get( numberingScheme ) == 'qsfp_dense':
            return
         if not eraseStartupConfigAndReloadConfirmation( mode, 'dense', 'qsfp' ):
            return

         config[ numberingScheme ] = 'qsfp_dense'
         t0( 'Erasing startup-config for qsfp_dense scheme.' )
         FileCli.eraseStartupConfig( mode, now=True )
         ReloadCli.doReload( mode, power=True, now=True )

      elif scheme == 'expanded':
         if config.get( numberingScheme ) == 'qsfp_expanded':
            return

         config[ numberingScheme ] = 'qsfp_expanded'
         print ( "\n! System reboot needed to make "
                 "expanded QSFP port numbering take effect.\n" )

      elif scheme == 'sparse':
         if numberingScheme not in config:
            return
         if not eraseStartupConfigAndReloadConfirmation( mode, 'sparse', 'qsfp' ):
            return

         config.pop( numberingScheme )
         t0( 'Erasing startup-config for qsfp_sparse scheme.' )
         FileCli.eraseStartupConfig( mode, now=True )
         ReloadCli.doReload( mode, power=True, now=True )

   except ( IOError, SimpleConfigFile.ParseError ) as e:
      mode.addError( 'Error setting qsfp port numbering scheme to %s (%s)' %
                     ( scheme, e ) )
      t0( 'Error setting qsfp port numbering scheme to %s.', scheme )
      return

class BootPortNumberingQsfpSchemeNameCmd( CliCommand.CliCommandClass ):
   syntax = 'boot port numbering qsfp SCHEME_NAME'
   data = {
      'boot': matcherBoot,
      'port': nodePort,
      'numbering': matcherNumbering,
      'qsfp': nodeQsfp,
      'SCHEME_NAME': CliCommand.Node( matcher=CliMatcher.EnumMatcher(
                     { 'dense': 'Make QSFP ports first.',
                       'sparse': 'Make QSFP ports last.',
                       'expanded': 'Expand QSFP port numbering' } ),
                        guard=isSchemeNameSupported ),
   }

   handler = setQsfpPortNumberingSchemeName

BasicCli.GlobalConfigMode.addCommandClass( BootPortNumberingQsfpSchemeNameCmd )

#------------------------------------------------------------------------------------
# [ no | default ] boot port numbering 
#------------------------------------------------------------------------------------
def noOrDefaultPortNumbering( mode, noOrDefault=None ): 
   # 'noOrDefault' is either 'True' or 'default'
   if( noOrDefault == True ):
      noOrDefault = 'no'

   config = bootConfig( mode )
   if not config.has_key( 'PORT_NUMBERING_SCHEME' ):
      return 

   if Cell.cellType() == 'fixed':
      if not eraseStartupConfigAndReloadConfirmation( mode, '', '', noOrDefault ):
         return

   try:
      config.pop( 'PORT_NUMBERING_SCHEME' )
   except (IOError, SimpleConfigFile.ParseError) as e:
      mode.addError( 'Error setting %s port numbering scheme (%s)' %\
            ( noOrDefault, e ) )
      t0( 'Error setting %s port numbering scheme.' )
      return

   if Cell.cellType() == 'fixed':
      t0( 'Erasing startup-config for %s scheme.' % ( noOrDefault ) )
      FileCli.eraseStartupConfig( mode, now=True )
      ReloadCli.doReload( mode, power=True, now=True )
   else:
      print ( "\n! System reboot needed to make "
              "default QSFP port numbering take effect.\n" )

#--------------------------------------------------------------------------------
# [ no | default ] boot port numbering
#--------------------------------------------------------------------------------
class BootPortNumberingCmd( CliCommand.CliCommandClass ):
   syntax = 'boot port numbering'
   noOrDefaultSyntax = syntax
   data = {
      'boot': matcherBoot,
      'port': nodePort,
      'numbering': matcherNumbering,
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefaultPortNumbering( mode, 'default' )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      noOrDefaultPortNumbering( mode, True )

BasicCliModes.GlobalConfigMode.addCommandClass( BootPortNumberingCmd )

#------------------------------------------------------------------------------------
# show port numbering 
#------------------------------------------------------------------------------------
def isModular():
   return ( entityMib.root and
            entityMib.root.tacType.fullTypeName != "EntityMib::FixedSystem" )

def showPortNumbering( mode ):
   # Since the command that invokes this function is only registered for fixed 
   # systems, 'fixedSystem' will always be there.
   scheme = 'notAvailable'
   if portNumberingInfo.multiplePortNumberingsSupported:
      scheme = portNumberingDict().get( 'PORT_NUMBERING_SCHEME' )
      # If scheme is "", current port numbering is the default one.
      # Get it from Sysdb.
      if not scheme:
         scheme = defaultPortNumbering()
   return scheme

def getModularPortNumbering( mode ):
   scheme = 'notAvailable'
   if portNumberingInfo.multiplePortNumberingsSupported:
      scheme = portNumberingDict().get( 'PORT_NUMBERING_SCHEME' )
      # If scheme is "", current port numbering is the default one.
      if not scheme or scheme == 'Unknown':
         scheme = 'default'
   return scheme

#------------------------------------------------------------------------------------
# show port numbering 
#------------------------------------------------------------------------------------
class PortNumberingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show port numbering'
   data = {
            'port' : 'Show port numbering',
            'numbering' : 'Show port numbering',
          }

   cliModel = PortNumbering

   @staticmethod
   def handler( mode, args ):
      ret = PortNumbering()
      if isModular():
         numberingFunc = getModularPortNumbering 
      else:
         numberingFunc = showPortNumbering
      ret.numberingScheme = \
            CapiStringToCliString.reverse()[ numberingFunc( mode ) ]
      return ret

BasicCli.addShowCommandClass( PortNumberingCmd )

def Plugin( entityManager ):
   global portNumberingInfo, entityMib
   portNumberingInfo = LazyMount.mount(
         entityManager, 'hardware/portNumberingInfo',
         'Hardware::PortNumberingInfo', 'ri' )

   entityMib = LazyMount.mount( entityManager,
                                'hardware/entmib',
                                'EntityMib::Status', 'r' )

   CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2017-06-01 00:00:00',
      lambda: [ 'show port numbering' ] )

