#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
import Tracing
import Tac
import cStringIO
import AgentCommandRequest
import AgentDirectory
import BmpAgent
import LazyMount
import CommonGuards
from Ark import switchTimeToUtc
import CliPlugin.TechSupportCli
import CliPlugin.VrfCli as VrfCli
from CliPlugin.BmpCliModels import AllBmpActiveSummary
from CliPlugin.BmpCliModels import AllBmpActive
from CliPlugin.BmpCliModels import AllBmpStationSummary
from CliPlugin.BmpCliModels import AllBmpPassiveSummary
from CliPlugin.BmpCliModels import AllBmpPassive
from CliPlugin.BmpCliModels import AllBmpConvergence
from CliPlugin.BmpCliModels import AllBmpStation
from CliPlugin.BmpCliModels import BmpPassiveSummary
from CliPlugin.BmpCliModels import BmpPassive
from CliPlugin.BmpCliModels import BmpActiveSummary
from CliPlugin.BmpCliModels import BmpActive
from CliPlugin.BmpCliModels import BmpStationSummary
from CliPlugin.BmpCliModels import BmpStation
from CliPlugin.BmpCliModels import StationHostnameOrAddress
from CliPlugin.BmpCliModels import BgpAdjRibInExportStatus
from CliPlugin.BmpCliModels import BmpGlobalConfig
from CliPlugin.BmpCliModels import BmpStationPolicies
from CliPlugin.BmpCliModels import BmpTcpStatistics
from CliPlugin.BmpCliModels import BmpStationConvergence
from CliPlugin.BmpCliModels import EorStatistic
from CliPlugin import AgentLibShowCommands, ShowTaskSchedulerModel

traceHandle = Tracing.Handle( 'Bmp' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

bmpStatus = None
bmpConfig = None
bgpClientConfig = None

AddrType = Tac.Type( 'Routing::Bmp::IpAddrOrHostname::AddrType' )
AfType = Tac.Type( 'Arnet::AddressFamily' )

matcherActive = CliMatcher.KeywordMatcher( 'active',
      helpdesc='Active station information' )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP information' )
matcherMonitoring = CliMatcher.KeywordMatcher( 'monitoring',
      helpdesc='BGP monitoring' )
matcherPassive = CliMatcher.KeywordMatcher( 'passive',
      helpdesc='Listener information' )
matcherStation = CliMatcher.KeywordMatcher( 'station',
      helpdesc='Monitoring station information' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary', helpdesc='Summary view' )
nodeInternal = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'internal',
         helpdesc='BMP Agent internal status' ),
      maxMatches=1, hidden=True, guard=CommonGuards.standbyGuard )

stationNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: (
                         bmpConfig.bmpStation.members() ),
                        'BGP monitoring station name' )

def isBmpRunning( mode ):
   return AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'Bmp' )

def runSocketCommand( mode, *args, **kwargs ):
   # The bmp agent will not be running in the case of a cohab test
   # In this case the agentCommandRequestSm will be started by createBmpRootSm
   if isBmpRunning( mode ) or mode.entityManager.local():
      AgentCommandRequest.runSocketCommand( mode.entityManager, *args, **kwargs )

def getMonitoringStatus():
   # BGP Monitoring Status is set in
   # routing/bgp/export/clientconfig/adjribin/Bmp/bmpEnabled
   bgpExportBmpClientConfig = bgpClientConfig.entityPtr.get( 'Bmp' )
   return False if bgpExportBmpClientConfig is None \
         else bgpExportBmpClientConfig.bmpEnabled

def checkBmpRunning( func ):
   def checkBmpRunning_wrapper( mode, **kwargs ):
      if not isBmpRunning( mode ) and not mode.entityManager.local():
         mode.addError( "Bmp agent is not running" )
         return None
      else:
         if kwargs:
            return func( mode, **kwargs )
         else:
            return func( mode )
   return checkBmpRunning_wrapper

def afiSafiStr( afiSafi ):
   if afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiUnicast' ):
      afiStr = 'IPv4 Unicast'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiUnicast' ):
      afiStr = 'IPv6 Unicast'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsLabels' ):
      afiStr = 'IPv6 Labeled Unicast'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiMplsVpn' ):
      afiStr = 'VPN IPv4'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsVpn' ):
      afiStr = 'VPN IPv6'
   else:
      afiStr = 'Unknown'
   return afiStr

def afiSafiExportConfig():
   afiSafiExport = {}
   for afiSafi, enabled in bmpConfig.afiSafiExportFlag.iteritems():
      afiSafiExport[ afiSafiStr( afiSafi ) ] = enabled
   return afiSafiExport

#--------------------------------------------------------------------------------
# show bgp monitoring active [ station STATION ]
#--------------------------------------------------------------------------------
class BgpMonitoringActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring active [ station STATION ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'active': matcherActive,
      'station': matcherStation,
      'STATION': stationNameMatcher
   }
   cliModel = AllBmpActive

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      station = args.get( 'STATION' )
      allActiveStatus = {}
      if station:
         status = bmpStatus.bmpActiveStatus.get( station )
         if status:
            allActiveStatus[ station ] = status
      else:
         allActiveStatus = bmpStatus.bmpActiveStatus

      model = AllBmpActive()
      model.bgpAdjRibInExportStatus = \
            BgpAdjRibInExportStatus( monitoringEnabled=getMonitoringStatus() )
      model.stations = {}
      for name, status in allActiveStatus.iteritems():
         stationModel = BmpActive()
         stationModel.address = StationHostnameOrAddress()
         if status.ipAddr.af != AfType.ipunknown:
            stationModel.address.ip = status.ipAddr
         stationModel.vrfName = status.vrfName
         stationModel.remotePort = status.remotePort
         stationModel.retryInterval = status.retryInterval

         stationModel.connected = status.connected
         if not status.connected:
            reason = status.reason
            if reason:
               stationModel.reason = reason[ 0 ].lower() + reason[ 1 : ]
            if status.nextRetryTime and status.nextRetryTime != float( "inf" ):
               stationModel.retryTime = switchTimeToUtc( status.nextRetryTime )

         stationModel.connectionAttempts = status.connectionAttemptCount
         stationModel.connectionSuccesses = status.connectionSuccessCount
         stationModel.connectionErrors = status.connectionErrorCount

         model.stations[ name ] = stationModel
      return model

BasicCli.addShowCommandClass( BgpMonitoringActiveCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring active summary
#--------------------------------------------------------------------------------
class BgpMonitoringActiveSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring active summary'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'active': matcherActive,
      'summary': matcherSummary,
   }
   cliModel = AllBmpActiveSummary

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      model = AllBmpActiveSummary()
      model.bgpAdjRibInExportStatus = \
            BgpAdjRibInExportStatus( monitoringEnabled=getMonitoringStatus() )
      model.stations = {}

      activeStatus = bmpStatus.bmpActiveStatus
      for status in activeStatus.itervalues():
         stationModel = BmpActiveSummary()
         stationModel.connected = status.connected
         if not status.connected and status.nextRetryTime and \
            status.nextRetryTime != float( "inf" ):
            stationModel.retryTime = switchTimeToUtc( status.nextRetryTime )
         model.stations[ status.station ] = stationModel
      return model

BasicCli.addShowCommandClass( BgpMonitoringActiveSummaryCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring [ station STATION ] [ detail ]
#--------------------------------------------------------------------------------
def stationToModel( config, status ):
   model = BmpStation()
   if config.description:
      model.description = config.description
   model.connectionMode = config.connectionMode
   model.vrfName = config.vrfName

   remoteHost = config.remoteHost
   model.address = StationHostnameOrAddress()
   if remoteHost != config.remoteHostDefault:
      if remoteHost.addrType == AddrType.ip:
         model.address.ip = remoteHost.ipAddr
      elif remoteHost.addrType == AddrType.hostname:
         model.address.hostName = remoteHost.host

   if config.connectionMode == 'active':
      model.remotePort = config.remotePort

   if config.bmpExportPolicyStationConfig:
      model.bmpExportPolicyStationConfig = \
         BmpStationPolicies(
            prePolicyExport=config.bmpExportPolicyStationConfig.prePolicyExport,
            postPolicyExport=config.bmpExportPolicyStationConfig.postPolicyExport )

   model.connected = ( status.sessionSmStatus is not None and
                       status.sessionSmStatus.connected )
   model.state = status.stationSmStatus.stationState

   if model.state == 'up':
      model.upTime = switchTimeToUtc( status.stationSmStatus.lastUpTime )
   model.flapCount = status.stationSmStatus.flapCount
   return model

class BgpMonitoringCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring [ station STATION ] [ detail ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'station': matcherStation,
      'STATION': stationNameMatcher,
      'detail': 'Detailed view'
   }
   cliModel = AllBmpStation

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      station = args.get( 'STATION' )
      if station:
         allStationStatus = {}
         status = bmpStatus.bmpStation.get( station )
         if status:
            allStationStatus[ station ] = status
      else:
         allStationStatus = bmpStatus.bmpStation

      model = AllBmpStation()
      model.bgpAdjRibInExportStatus = \
            BgpAdjRibInExportStatus( monitoringEnabled=getMonitoringStatus() )
      afiSafiExport = afiSafiExportConfig()
      model.bmpGlobalConfig = \
            BmpGlobalConfig( timestampMode=bmpConfig.timestampMode,
                  prePolicyExport=bmpConfig.bmpExportPolicyConfig.prePolicyExport,
                  postPolicyExport=bmpConfig.bmpExportPolicyConfig.postPolicyExport,
                  afiSafiExport=afiSafiExport,
                  exportSixPe=bmpConfig.exportSixPe,
                  exportIpv6LuTunnel=bmpConfig.exportIpv6LuTunnel )
      model.stations = {}

      for name, status in allStationStatus.iteritems():
         config = bmpConfig.bmpStation.get( name )
         if not config:
            continue
         stationModel = stationToModel( config, status )

         if 'detail' in args:
            tcpModel = None
            buff = cStringIO.StringIO()
            command1 = 'bmpSocketStats' + status.station
            runSocketCommand( mode, BmpAgent.name, "state",
                  command1, stringBuff=buff )
            try:
               dataDict = eval( buff.getvalue() ) # pylint: disable-msg=eval-used
               if 'tcpStats' in dataDict:
                  tcpModel = BmpTcpStatistics()
                  tcpStats = dataDict[ 'tcpStats' ]
                  tcpModel.setAttrFromDict( tcpStats )
                  stationModel.tcpSocketStatistics = tcpModel
            except SyntaxError:
               t0( "Error: Unable to retrieve TCP socket information correctly" )

         model.stations[ name ] = stationModel
      return model

BasicCli.addShowCommandClass( BgpMonitoringCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring internal
#--------------------------------------------------------------------------------
class BgpMonitoringInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring internal'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
   }
   privileged = True

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      runSocketCommand( mode, BmpAgent.name, 'state', '' )

BasicCli.addShowCommandClass( BgpMonitoringInternalCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring internal convergence
#--------------------------------------------------------------------------------
def getConvergenceModel( status ):
   stationModel = BmpStationConvergence()
   sessionStatus = status.sessionSmStatus
   stationModel.converged = sessionStatus.converged
   stationModel.eorStatistics = {}
   if stationModel.converged:
      stationModel.lastConvergedTimestamp = \
         switchTimeToUtc( sessionStatus.lastConvergedTimestamp )
      stationModel.convergenceTime = sessionStatus.convergenceTime

   eorStatCol = sessionStatus.eorStatCol
   for peerKey, eorStat in eorStatCol.iteritems():
      eorStatModel = EorStatistic()
      eorStatModel.peerAddr = peerKey.addr
      eorStatModel.initialPeer = eorStat.initialPeer
      eorStatModel.pendingCount = len( eorStat.pendingEor )
      stationModel.eorStatistics[ peerKey.addr ] = eorStatModel

   return stationModel

class BgpMonitoringInternalConvergenceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring internal convergence'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
      'convergence': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'convergence', helpdesc='BMP convergence status' ),
         guard=CommonGuards.standbyGuard ),
   }
   cliModel = AllBmpConvergence
   privileged = True

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      allStationStatus = bmpStatus.bmpStation
      model = AllBmpConvergence()
      model.stations = {}

      for status in allStationStatus.itervalues():
         stationModel = getConvergenceModel( status )
         model.stations[ status.station ] = stationModel
      return model

BasicCli.addShowCommandClass( BgpMonitoringInternalConvergenceCmd )

#--------------------------------------------------------------------------------
# Deprecate 'show bgp monitoring internal scheduler reset'.  It was a
# lazy implementation and doesn't fit the new one.
# --------------------------------------------------------------------------------
class BgpMonitoringInternalSchedulerResetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring internal scheduler reset'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
      'scheduler': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'scheduler', helpdesc='BMP Agent scheduler' ),
         guard=CommonGuards.standbyGuard ),
      'reset': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'reset',
         helpdesc='Reset scheduler statistics' ),
         deprecatedByCmd='clear agent Bmp task scheduler' ),
   }

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      # Since it's deprecated, this should never get called, but
      # if it is, let's make sure that the user knows that it didn't
      # work.
      mode.addError( 'Use "clear agent Bmp task scheduler"' )

BasicCli.addShowCommandClass( BgpMonitoringInternalSchedulerResetCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring internal scheduler
#       [ { ( verbose | history ) } ]
#--------------------------------------------------------------------------------
class BgpMonitoringInternalSchedulerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = \
      'show bgp monitoring internal scheduler' \
      ' [ { ( verbose | history ) } ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
      'scheduler': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'scheduler', helpdesc='BMP Agent scheduler' ),
         guard=CommonGuards.standbyGuard ),
      'verbose': 'Include more information',
      'history': 'Display scheduler event history',
   }
   privileged = True
   cliModel = ShowTaskSchedulerModel.Overall

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      # Translate options to those used in AgentCli
      translation = { 'verbose': 'detail',
                      'history': 'history' }
      agentArgs = { 'AGENT': 'Bmp',
                    'SCHEDULER_OPTS': [ translation[ i ] for i in translation
                                        if i in args ],
      }
      return AgentLibShowCommands.showTaskSchedulerHandler( mode, agentArgs )

BasicCli.addShowCommandClass( BgpMonitoringInternalSchedulerCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring passive [ VRF ]
#--------------------------------------------------------------------------------
class BgpMonitoringPassiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring passive [ VRF ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'passive': matcherPassive,
      'VRF': VrfCli.VrfExprFactory( helpdesc='VRF name', inclDefaultVrf=True ),
   }
   cliModel = AllBmpPassive

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      vrfName = args.get( 'VRF' )
      if vrfName:
         listenStatus = {}
         for af in [ AfType.ipv4, AfType.ipv6 ]:
            passiveKey = Tac.ValueConst( 'Routing::Bmp::PassiveKey', vrfName, af )
            vrfStatus = bmpStatus.bmpListenStatus.get( passiveKey )
            if vrfStatus:
               listenStatus[ passiveKey ] = vrfStatus
      else:
         listenStatus = bmpStatus.bmpListenStatus

      model = AllBmpPassive()
      model.bgpAdjRibInExportStatus = \
            BgpAdjRibInExportStatus( monitoringEnabled=getMonitoringStatus() )
      model.listenersV4 = {}
      model.listenersV6 = {}

      for passiveKey, vrfStatus in listenStatus.iteritems():
         # Populate BmpPassive model
         passiveModel = BmpPassive()
         passiveModel.port = vrfStatus.port
         passiveModel.listening = vrfStatus.listening
         reason = vrfStatus.reason
         if not vrfStatus.listening and reason:
            passiveModel.reason = reason[ 0 ].lower() + reason[ 1 : ]
         passiveModel.connectionsAccepted = vrfStatus.connectionsAccepted
         passiveModel.connectionsRejected = vrfStatus.connectionsRejected
         passiveModel.stations = { name: StationHostnameOrAddress( ip=str( ip ) )
                                   for ip,
                                   name in vrfStatus.stationHost.iteritems() }

         # Add BmpPassive model into AllBmpPassive model
         if passiveKey.addrFamily == AfType.ipv4:
            model.listenersV4[ passiveKey.vrfName ] = passiveModel
         elif passiveKey.addrFamily == AfType.ipv6:
            model.listenersV6[ passiveKey.vrfName ] = passiveModel
      return model

BasicCli.addShowCommandClass( BgpMonitoringPassiveCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring passive summary
#--------------------------------------------------------------------------------
class BgpMonitoringPassiveSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring passive summary'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'passive': matcherPassive,
      'summary': matcherSummary,
   }
   cliModel = AllBmpPassiveSummary

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      model = AllBmpPassiveSummary()
      listenStatus = bmpStatus.bmpListenStatus
      model.bgpAdjRibInExportStatus = \
            BgpAdjRibInExportStatus( monitoringEnabled=getMonitoringStatus() )
      model.listenersV4 = {}
      model.listenersV6 = {}

      for passiveKey, vrfStatus in listenStatus.iteritems():
         # Populate the BmpPassiveSummary model
         passiveModel = BmpPassiveSummary()
         passiveModel.port = vrfStatus.port
         passiveModel.stationCount = len( vrfStatus.stationHost )
         passiveModel.listening = vrfStatus.listening

         # Add BmpPassiveSummary model into AllBmpPassiveSummary model
         if passiveKey.addrFamily == AfType.ipv4:
            model.listenersV4[ passiveKey.vrfName ] = passiveModel
         elif passiveKey.addrFamily == AfType.ipv6:
            model.listenersV6[ passiveKey.vrfName ] = passiveModel

      return model

BasicCli.addShowCommandClass( BgpMonitoringPassiveSummaryCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring summary
#--------------------------------------------------------------------------------
class BgpMonitoringSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring summary'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'summary': matcherSummary,
   }
   cliModel = AllBmpStationSummary

   @staticmethod
   @checkBmpRunning
   def handler( mode, args ):
      model = AllBmpStationSummary()
      model.bgpAdjRibInExportStatus = \
            BgpAdjRibInExportStatus( monitoringEnabled=getMonitoringStatus() )

      afiSafiExport = afiSafiExportConfig()
      model.bmpGlobalConfig = \
            BmpGlobalConfig( timestampMode=bmpConfig.timestampMode,
                  prePolicyExport=bmpConfig.bmpExportPolicyConfig.prePolicyExport,
                  postPolicyExport=bmpConfig.bmpExportPolicyConfig.postPolicyExport,
                  afiSafiExport=afiSafiExport,
                  exportSixPe=bmpConfig.exportSixPe,
                  exportIpv6LuTunnel=bmpConfig.exportIpv6LuTunnel )
      model.stations = {}

      stationStatus = bmpStatus.bmpStation
      for status in stationStatus.itervalues():
         stationModel = BmpStationSummary()
         stationModel.state = status.stationSmStatus.stationState
         if stationModel.state == 'up':
            stationModel.upTime = switchTimeToUtc(
                  status.stationSmStatus.lastUpTime )
         model.stations[ status.station ] = stationModel
      return model

BasicCli.addShowCommandClass( BgpMonitoringSummaryCmd )

#-------------------------------------------------------------------------------
# BMP commands in "show tech-support"
#-------------------------------------------------------------------------------

def _showTechSupportCmds():

   cmds = [ "show bgp monitoring",
            "show bgp monitoring summary",
            "show bgp monitoring active",
            "show bgp monitoring passive" ]

   return cmds

BMP_TECHSUPPORT_TIMESTAMP = '2017-08-25 11:30:00'

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      BMP_TECHSUPPORT_TIMESTAMP, _showTechSupportCmds,
      summaryCmdCallback=lambda: [ 'show bgp monitoring summary' ] )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bmpStatus
   global bmpConfig
   global bgpClientConfig

   bmpStatus = LazyMount.mount( entityManager, 'routing/bmp/status',
                                'Routing::Bmp::BmpStatus', 'r' )
   bmpConfig = LazyMount.mount( entityManager, 'routing/bmp/config',
                                'Routing::Bmp::BmpConfig', 'r' )
   bgpClientConfig = LazyMount.mount( entityManager,
                                 'routing/bgp/export/clientconfig/adjribin',
                                 'Tac::Dir', 'ri' )
