#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
from CliMatcher import StringMatcher
import CliParser
import CliPlugin.BmpCli as BmpCli
from CliPlugin.BmpCli import RouterBmpStationMode
from CliPlugin.BmpCli import RouterBmpStationModelet
from CliPlugin.IntfCli import Intf
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher

matcherConnection = CliMatcher.KeywordMatcher(
   'connection',
   helpdesc='Connection configuration for the BGP monitoring session' )

#--------------------------------------------------------------------------------
# [no | default ] authentication-key ( [ 0 ] AUTHUNENCRYPTEDPASSWD ) |
#                                    ( 7 AUTHENCRYPTEDPASSWD )
#--------------------------------------------------------------------------------
class AuthenticationKeyCmd( CliCommand.CliCommandClass ):
   syntax = ( 'authentication-key ( [ 0 ] AUTHUNENCRYPTEDPASSWD ) | '
              '( 7 AUTHENCRYPTEDPASSWD )' )
   noOrDefaultSyntax = 'authentication-key ...'
   data = {
      'authentication-key': CliMatcher.KeywordMatcher(
         'authentication-key',
         helpdesc='Configure authentication key for the BGP monitoring station' ),
      '0': 'Encryption type -unencrypted',
      'AUTHUNENCRYPTEDPASSWD': CliMatcher.PatternMatcher(
         pattern='[^\\s]{1,80}',
         helpdesc='password(up to 80 chars)', helpname='LINE' ),
      '7': 'Encryption type -proprietary',
      'AUTHENCRYPTEDPASSWD': CliMatcher.PatternMatcher(
         pattern='[^\\s]+',
         helpdesc='encrypted password', helpname='LINE' ),
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      if '7' in args:
         stationModelet.setBmpStationEnKey( args[ 'AUTHENCRYPTEDPASSWD' ] )
      else:
         stationModelet.setBmpStationUnenKey( args[ 'AUTHUNENCRYPTEDPASSWD' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noBmpStationKey()

RouterBmpStationMode.addCommandClass( AuthenticationKeyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connection address ( ADDR | V6ADDR | HOSTNAME ) [ vrf VRFNAME ]
#--------------------------------------------------------------------------------
class ConnectionAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'connection address ( ADDR | V6ADDR | HOSTNAME ) [ vrf VRFNAME ]'
   noOrDefaultSyntax = 'connection address ...'
   data = {
      'connection': 'Connection configuration for the BGP monitoring session',
      'address': "Specify the BGP monitoring station's remote address",
      'ADDR': IpAddrMatcher( helpdesc='BMP station address' ),
      'V6ADDR': Ip6AddrMatcher( helpdesc='BMP station address' ),
      'HOSTNAME': CliMatcher.PatternMatcher(
         pattern='[0-9a-zA-Z\\_\\.\\-:]+',
         helpdesc='BMP station hostname', helpname='WORD',
         priority=CliParser.PRIO_LOW ),
      'vrf': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'vrf',
            helpdesc='Configure BMP in a VRF' ),
         hidden=True ),
      'VRFNAME': CliMatcher.DynamicNameMatcher( BmpCli.getVrfNames, 'VRF name' ),
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      vrfName = None
      if 'VRFNAME' in args:
         vrfName = args[ 'VRFNAME' ]

      addr = None
      if 'ADDR' in args:
         addr = args[ 'ADDR' ]
      elif 'V6ADDR' in args:
         addr = args[ 'V6ADDR' ]
      elif 'HOSTNAME' in args:
         addr = args[ 'HOSTNAME' ]

      stationModelet.setRemoteAddr( addr, vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noRemoteAddr()

RouterBmpStationMode.addCommandClass( ConnectionAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connection keepalive IDLETIME PROBEINTERVAL PROBECOUNT
#--------------------------------------------------------------------------------
class ConnectionKeepaliveCmd(
      CliCommand.CliCommandClass ):
   syntax = 'connection keepalive IDLETIME PROBEINTERVAL PROBECOUNT'
   noOrDefaultSyntax = 'connection keepalive ...'
   data = {
      'connection': matcherConnection,
      'keepalive': CliMatcher.KeywordMatcher(
         'keepalive',
         helpdesc="Specify the BGP monitoring station's TCP Keepalive parameters" ),
      'IDLETIME': CliMatcher.IntegerMatcher(
         1, 86400, helpdesc='Idle time (seconds) before TCP Keepalive' ),
      'PROBEINTERVAL': CliMatcher.IntegerMatcher(
         1, 3600, helpdesc='Interval (seconds) between TCP Keepalive Probes' ),
      'PROBECOUNT': CliMatcher.IntegerMatcher(
         1, 1000,
         helpdesc='Number of Keepalive probes before closing connection' ),
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      idleTime = args[ 'IDLETIME' ]
      probeInterval = args[ 'PROBEINTERVAL' ]
      probeCount = args[ 'PROBECOUNT' ]
      stationModelet.setTcpKeepalive( idleTime, probeInterval, probeCount )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noTcpKeepalive()

RouterBmpStationMode.addCommandClass( ConnectionKeepaliveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connection mode
#    ( active port PORT [ reconnect - timer RECONNECTTIME ] ) | ( passive )
#--------------------------------------------------------------------------------
class ConnectionModeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'connection mode '
              '( active port PORT [ reconnect-timer RECONNECTTIME ] ) | '
              '( passive )' )
   noOrDefaultSyntax = 'connection mode ...'
   data = {
      'connection': matcherConnection,
      'mode': CliMatcher.KeywordMatcher(
         'mode',
         helpdesc="Specify the BGP monitoring station's connection mode" ),
      'active': 'Active mode',
      'port': 'BGP monitoring protocol port number for stations in passive mode',
      'PORT': CliMatcher.IntegerMatcher(
         1, 65535, helpdesc='Configure the BMP port to this value' ),
      'reconnect-timer': "Configure the BGP monitoring session's reconnect timer",
      'RECONNECTTIME': CliMatcher.IntegerMatcher(
         1, 3600, helpdesc='Configure the reconnect timer to this value' ),
      'passive': 'Passive mode',
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      if 'active' in args:
         reconnectTime = None
         if 'RECONNECTTIME' in args:
            reconnectTime = args[ 'RECONNECTTIME' ]
         stationModelet.setActiveMode( args[ 'PORT' ], reconnectTime )
      else:
         stationModelet.setPassiveMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noMode()

RouterBmpStationMode.addCommandClass( ConnectionModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description DESCRIPTION
#--------------------------------------------------------------------------------
class DescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description': 'Assign a description to the BGP monitoring station',
      'DESCRIPTION': StringMatcher(
         helpdesc='Text describing this BGP monitoring station' ),
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.setBmpStationDescription( args[ 'DESCRIPTION' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noBmpStationDescription()

RouterBmpStationMode.addCommandClass( DescriptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] export-policy received routes { pre-policy | post-policy }
#--------------------------------------------------------------------------------
class ExportPolicyReceivedRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'export-policy received routes { pre-policy | post-policy }'
   noOrDefaultSyntax = 'export-policy received routes ...'
   data = {
      'export-policy': 'Per-station BGP monitoring protocol export policy',
      'received': CliMatcher.KeywordMatcher(
         'received',
         helpdesc='BGP monitoring protocol received route selection' ),
      'routes': CliMatcher.KeywordMatcher(
         'routes',
         helpdesc='BGP monitoring protocol received route selection' ),
      'pre-policy': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'pre-policy',
            helpdesc='Export BGP routes before input policies are applied' ),
         maxMatches=1 ),
      'post-policy': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'post-policy',
            helpdesc='Export BGP routes after input policies are applied' ),
         maxMatches=1 ),
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.setBmpReceivedRoutes( args )

   @staticmethod
   def noHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noBmpReceivedRoutes()

   @staticmethod
   def defaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.defaultBmpReceivedRoutes()

RouterBmpStationMode.addCommandClass( ExportPolicyReceivedRoutesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown [ reason MESSAGE ]
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown [ reason MESSAGE ]'
   noOrDefaultSyntax = 'shutdown ...'
   data = {
      'shutdown': 'Administratively shut down the BGP monitoring station',
      'reason': 'BGP monitoring station shutdown reason to be saved',
      'MESSAGE': CliMatcher.StringMatcher( helpname='MESSAGE',
         helpdesc='Shut down message' )
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.setShutdown( args.get( 'MESSAGE' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noShutdown()

RouterBmpStationMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] statistics-interval STATSINTERVAL
#--------------------------------------------------------------------------------
class StatisticsIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'statistics-interval STATSINTERVAL'
   noOrDefaultSyntax = 'statistics-interval ...'
   hidden = True
   data = {
      'statistics-interval': CliMatcher.KeywordMatcher(
         'statistics-interval',
         helpdesc='Configure statistic interval forthe BGP monitoring station' ),
      'STATSINTERVAL': CliMatcher.IntegerMatcher(
         1, 2 ** 32 - 1,
         helpdesc='Interval for sending statisticsreport in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.setStatsInterval( args[ 'STATSINTERVAL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noStatsInterval()

RouterBmpStationMode.addCommandClass( StatisticsIntervalCmd )

#--------------------------------------------------------------------------------
# update-source INTF
#--------------------------------------------------------------------------------
class UpdateSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'update-source INTF'
   noOrDefaultSyntax = 'update-source ...'
   data = {
      'update-source': CliMatcher.KeywordMatcher(
         'update-source',
         helpdesc=(
            'Specify a local source interface for the BGP monitoring session' )
      ),
      'INTF': Intf.matcherWithIpSupport,
   }

   @staticmethod
   def handler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.sourceIntf( args[ 'INTF' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stationModelet = mode.modeletMap[ RouterBmpStationModelet ]
      stationModelet.noSourceIntf()

RouterBmpStationMode.addCommandClass( UpdateSourceCmd )
