#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BmpUtils
import CliCommand
import CliMatcher
import CliPlugin.BmpCli as BmpCli
from CliPlugin.RoutingBgpCli import RouterBgpBaseMode
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import Tac

import Toggles.BmpToggleLib

# Tac Entities
bmpConfig = None

# Enum Type
TimestampMode = Tac.Type( 'Routing::Bmp::TimestampMode' )

matcherMonitoring = CliMatcher.KeywordMatcher(
   'monitoring', helpdesc='BGP monitoring protocol configuration' )
matcherReceived = CliMatcher.KeywordMatcher(
   'received', helpdesc='BGP monitoring protocol received route selection' )
matcherRoutes = CliMatcher.KeywordMatcher(
   'routes', helpdesc='BGP monitoring protocol received route selection' )
matcherTimestamp = CliMatcher.KeywordMatcher(
   'timestamp',
   helpdesc='BGP monitoring protocol Per-Peer Header timestamp behavior' )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring port PORT
# XXX: BUG157364 add support for VRF
#--------------------------------------------------------------------------------
class MonitoringPortCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring port PORT'
   noOrDefaultSyntax = 'monitoring port ...'
   data = {
      'monitoring': matcherMonitoring,
      'port': CliMatcher.KeywordMatcher(
         'port',
         helpdesc=( 'BGP monitoring protocol port number '
                    'for stations in passive mode' ) ),
      'PORT': CliMatcher.IntegerMatcher(
         1024, 65535, helpdesc='BGP monitoring protocol port number' ),
   }

   @staticmethod
   def handler( mode, args ):
      vrfName = DEFAULT_VRF
      bmpConfig.localPort[ vrfName ] = args[ 'PORT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = DEFAULT_VRF
      if vrfName in bmpConfig.localPort:
         del bmpConfig.localPort[ vrfName ]

RouterBgpBaseMode.addCommandClass( MonitoringPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring received routes post-policy
#--------------------------------------------------------------------------------
class MonitoringReceivedRoutesPostPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring received routes post-policy'
   noSyntax = 'monitoring received routes post-policy ...'
   defaultSyntax = 'monitoring received routes post-policy ...'
   data = {
      'monitoring': matcherMonitoring,
      'received': matcherReceived,
      'routes': matcherRoutes,
      'post-policy': CliMatcher.KeywordMatcher(
         'post-policy',
         helpdesc='Export BGP routes after input policies are applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      bmpConfig.bmpExportPolicyConfig = \
         BmpUtils.BmpExportPolicyConfig(
            bmpConfig.bmpExportPolicyConfig.prePolicyExport, True )

   @staticmethod
   def noHandler( mode, args ):
      bmpConfig.bmpExportPolicyConfig = \
         BmpUtils.BmpExportPolicyConfig(
            bmpConfig.bmpExportPolicyConfig.prePolicyExport, False )

   @staticmethod
   def defaultHandler( mode, args ):
      MonitoringReceivedRoutesPostPolicyCmd.handler( mode, args )

RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesPostPolicyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring received routes pre-policy
#--------------------------------------------------------------------------------
class MonitoringReceivedRoutesPrePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring received routes pre-policy'
   noSyntax = 'monitoring received routes pre-policy ...'
   defaultSyntax = 'monitoring received routes pre-policy ...'
   data = {
      'monitoring': matcherMonitoring,
      'received': matcherReceived,
      'routes': matcherRoutes,
      'pre-policy': CliMatcher.KeywordMatcher(
         'pre-policy',
         helpdesc='Export BGP routes before input policies are applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      bmpConfig.bmpExportPolicyConfig = \
         BmpUtils.BmpExportPolicyConfig(
            True, bmpConfig.bmpExportPolicyConfig.postPolicyExport )

   @staticmethod
   def noHandler( mode, args ):
      bmpConfig.bmpExportPolicyConfig = \
         BmpUtils.BmpExportPolicyConfig(
            False, bmpConfig.bmpExportPolicyConfig.postPolicyExport )

   @staticmethod
   def defaultHandler( mode, args ):
      MonitoringReceivedRoutesPrePolicyCmd.handler( mode, args )

RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesPrePolicyCmd )

if Toggles.BmpToggleLib.toggleBmpArBgpVpnToggleEnabled():
   #--------------------------------------------------------------------------------
   # [ no | default ] monitoring received routes address-family \
   #                  ( ipv4 unicast [ disable ] ) | \
   #                  ( ipv6 unicast [ disable ] | \
   #                         labeled-unicast { 6pe | tunnel } ) |\
   #                  ( vpn-ipv4 ) | (vpn-ipv6) )
   #--------------------------------------------------------------------------------
   class MonitoringReceivedRoutesAfiSafiCmd( CliCommand.CliCommandClass ):
      syntax = (
         'monitoring received routes address-family'
         ' ( ipv4 unicast [ disable ] ) |'
         ' ( ipv6 ( unicast [ disable ] ) |'
         '        ( labeled-unicast { 6pe | tunnel } ) ) |'
         ' ( vpn-ipv4 ) | '
         ' ( vpn-ipv6 ) ' )
      noOrDefaultSyntax = (
         'monitoring received routes address-family'
         ' ( ipv4 unicast ) |'
         ' ( ipv6 ( unicast | labeled-unicast ) ) |'
         ' ( vpn-ipv4 ) | ( vpn-ipv6 ) ... ' )
      data = {
         'monitoring': matcherMonitoring,
         'received': matcherReceived,
         'routes': matcherRoutes,
         'address-family': 'Export specified address family',
         'ipv4': 'IPv4 related',
         'ipv6': 'IPv6 related',
         'unicast': 'Unicast sub-address family',
         'labeled-unicast': 'Labeled-unicast sub-address family',
         '6pe': CliCommand.Node(
            CliMatcher.KeywordMatcher(
               '6pe',
               helpdesc='Export 6PE paths' ),
            maxMatches=1 ),
         'tunnel': CliCommand.Node(
            CliMatcher.KeywordMatcher(
               'tunnel',
               helpdesc='Export IPv6 Labeled-unicast tunnel paths' ),
            maxMatches=1,
            hidden=True ),
         'disable': 'Disable export of this address family',
         'vpn-ipv4': 'MPLS L3 VPN IPv4 unicast address family',
         'vpn-ipv6': 'MPLS L3 VPN IPv6 unicast address family',
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         afi = 'afiIpv4' if 'ipv4' in args or 'vpn-ipv4' in args else 'afiIpv6'
         if 'labeled-unicast' in args:
            safi = 'safiMplsLabels'
         elif 'vpn-ipv4' in args or 'vpn-ipv6' in args:
            safi = 'safiMplsVpn'
         else:
            safi = 'safiUnicast'
         args[ 'AFISAFI' ] = Tac.Value( 'Routing::Bgp::AfiSafi', afi, safi )

      @staticmethod
      def handler( mode, args ):
         afiSafi = args[ 'AFISAFI' ]
         if 'disable' in args:
            bmpConfig.afiSafiExport[ afiSafi ] = False
         else:
            bmpConfig.afiSafiExport[ afiSafi ] = True
            if 'labeled-unicast' in args:
               bmpConfig.exportSixPe = '6pe' in args
               bmpConfig.exportIpv6LuTunnel = 'tunnel' in args

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         afiSafi = args[ 'AFISAFI' ]
         del bmpConfig.afiSafiExport[ afiSafi ]
         if 'labeled-unicast' in args:
            bmpConfig.exportSixPe = False
            bmpConfig.exportIpv6LuTunnel = False

   RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesAfiSafiCmd )
else:
   #--------------------------------------------------------------------------------
   # [ no | default ] monitoring received routes address-family \
   #                  ( ipv4 unicast [ disable ] ) | \
   #                  ( ipv6 unicast [ disable ] | labeled-unicast { 6pe | tunnel } )
   #--------------------------------------------------------------------------------
   class MonitoringReceivedRoutesAfiSafiCmd( CliCommand.CliCommandClass ):
      syntax = (
         'monitoring received routes address-family'
         ' ( ipv4 unicast [ disable ] ) |'
         ' ( ipv6 ( unicast [ disable ] ) |'
         '        ( labeled-unicast { 6pe | tunnel } ) )' )
      noOrDefaultSyntax = (
         'monitoring received routes address-family'
         ' ( ipv4 unicast ) |'
         ' ( ipv6 ( unicast | labeled-unicast ) ) ... ' )
      data = {
         'monitoring': matcherMonitoring,
         'received': matcherReceived,
         'routes': matcherRoutes,
         'address-family': 'Export specified address family',
         'ipv4': 'IPv4 related',
         'ipv6': 'IPv6 related',
         'unicast': 'Unicast sub-address family',
         'labeled-unicast': 'Labeled-unicast sub-address family',
         '6pe': CliCommand.Node(
            CliMatcher.KeywordMatcher(
               '6pe',
               helpdesc='Export 6PE paths' ),
            maxMatches=1 ),
         'tunnel': CliCommand.Node(
            CliMatcher.KeywordMatcher(
               'tunnel',
               helpdesc='Export IPv6 Labeled-unicast tunnel paths' ),
            maxMatches=1,
            hidden=True ),
         'disable': 'Disable export of this address family',
      }

      @staticmethod
      def adapter( mode, args, argsList ):
         afi = 'afiIpv4' if 'ipv4' in args else 'afiIpv6'
         safi = 'safiMplsLabels' if 'labeled-unicast' in args else 'safiUnicast'
         args[ 'AFISAFI' ] = Tac.Value( 'Routing::Bgp::AfiSafi', afi, safi )

      @staticmethod
      def handler( mode, args ):
         afiSafi = args[ 'AFISAFI' ]
         if 'disable' in args:
            bmpConfig.afiSafiExport[ afiSafi ] = False
         else:
            bmpConfig.afiSafiExport[ afiSafi ] = True
            if 'labeled-unicast' in args:
               bmpConfig.exportSixPe = '6pe' in args
               bmpConfig.exportIpv6LuTunnel = 'tunnel' in args

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         afiSafi = args[ 'AFISAFI' ]
         del bmpConfig.afiSafiExport[ afiSafi ]
         if 'labeled-unicast' in args:
            bmpConfig.exportSixPe = False
            bmpConfig.exportIpv6LuTunnel = False

   RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesAfiSafiCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring station BMPSTATIONNAME
#--------------------------------------------------------------------------------
class MonitoringStationBmpstationnameCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring station BMPSTATIONNAME'
   noOrDefaultSyntax = 'monitoring station BMPSTATIONNAME ...'
   data = {
      'monitoring': matcherMonitoring,
      'station': 'BGP monitoring station configuration',
      'BMPSTATIONNAME': CliMatcher.DynamicNameMatcher(
         lambda mode: ( sorted( bmpConfig.bmpStation.members() ) ),
         helpdesc='BGP monitoring station name' ),
   }

   @staticmethod
   def handler( mode, args ):
      BmpCli.gotoBmpStationMode( mode, args[ 'BMPSTATIONNAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      BmpCli.deleteBmpStationMode( mode, args[ 'BMPSTATIONNAME' ] )

RouterBgpBaseMode.addCommandClass( MonitoringStationBmpstationnameCmd )

#--------------------------------------------------------------------------------
# ( no | default ) monitoring timestamp ( none | send-time )
#--------------------------------------------------------------------------------
class MonitoringTimestampCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring timestamp ( none | send-time )'
   noOrDefaultSyntax = 'monitoring timestamp ...'
   data = {
      'monitoring': matcherMonitoring,
      'timestamp': matcherTimestamp,
      'none': 'Zero',
      'send-time': 'Time when BGP monitoring protocol message is sent',
   }

   @staticmethod
   def handler( mode, args ):
      if 'none' in args:
         bmpConfig.timestampMode = TimestampMode.none
      else:
         bmpConfig.timestampMode = TimestampMode.sendTime

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bmpConfig.timestampMode = bmpConfig.timestampModeDefault

RouterBgpBaseMode.addCommandClass( MonitoringTimestampCmd )

def Plugin( entityManager ):
   global bmpConfig
   bmpConfig = ConfigMount.mount( entityManager, 'routing/bmp/config',
                                  'Routing::Bmp::BmpConfig', 'w' )
