# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet.TcpUtils
import BasicCli
import BmpUtils
import CliCommand
import CliParser
import ConfigMount
import LazyMount
import Tac
import Tracing
from BgpLib import (
      isValidPeerAddr,
      isValidV6PeerAddr,
)
from CliMode.Bmp import RoutingBmpStationMode
from CliPlugin.RoutingBgpCli import (
      bgpMatcherForConfig,
      bgpNeighborConfig,
      RouterBgpSharedModelet,
      deleteRouterBgpVrfHook,
      delNeighborConfigIfDefault,
      PeerCliExpression,
)
from CliPlugin.RoutingBgpNeighborCli import validatePeer
import CliToken.RoutingBgp as bgpTokens
from IpLibConsts import DEFAULT_VRF
from ReversibleSecretCli import decodeKey

# Tracing
traceHandle = Tracing.Handle( 'Bmp' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

# Constants
U32_MAX_VALUE =  0xFFFFFFFF

# Tac Entities
allVrfConfig = None
bmpConfig = None

# Enum Type
ConnectionMode = Tac.Type( 'Routing::Bmp::ConnectionMode' )
TimestampMode = Tac.Type( 'Routing::Bmp::TimestampMode' )
IpAddrOrHostname = Tac.Type( 'Routing::Bmp::IpAddrOrHostname' )

def cleanupBmpConfig( vrfName ):
   # Determine which attrs to automatically reset to default
   noCleanupAttrs = { 'parent', 'bmpStation', 'localPort', 'afiSafiExport',
                      'afiSafiExportFlag' }
   cleanupAttrs = { i.name for i in bmpConfig.tacType.attributeQ
                    if i.writable and i.name not in noCleanupAttrs }
   defaultBmpConfig = \
      Tac.root.newEntity( 'Routing::Bmp::BmpConfig', 'defaultBmp' )

   # Bmp does not depend on Bgp's VRF configuration
   if vrfName == DEFAULT_VRF:
      bmpConfig.bmpStation.clear()
      if vrfName in bmpConfig.localPort:
         del bmpConfig.localPort[ vrfName ]
      # Reset remaining config attrs to default values
      for attr in cleanupAttrs:
         setattr( bmpConfig, attr, getattr( defaultBmpConfig, attr ) )
      bmpConfig.bmpActivate = False

class RouterBmpStationMode( RoutingBmpStationMode, BasicCli.ConfigModeBase ):
   name = 'Bgp monitoring station configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, bmpStationName ):
      self.bmpStationName = bmpStationName
      RoutingBmpStationMode.__init__( self, bmpStationName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# "monitoring station" config mode
#-------------------------------------------------------------------------------
class RouterBmpStationModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.bmpStationName = mode.bmpStationName

   def setBmpStationDescription( self, desc ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.description = desc

   def noBmpStationDescription( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if config:
         config.description = config.descriptionDefault

   def setShutdown( self, msg=None ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.shutdownMsg = msg if msg else config.shutdownMsgDefault
      config.shutdown = True

   def noShutdown( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.shutdownMsg = config.shutdownMsgDefault
      config.shutdown = False

   def setBmpStationUnenKey( self, authUnencryptedPasswd ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.authenticationKey = authUnencryptedPasswd

   def setBmpStationEnKey( self, authEncryptedPasswd ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      try:
         config.authenticationKey = decodeKey( authEncryptedPasswd,
                                               key=self.bmpStationName +
                                                   '_passwd',
                                               algorithm='MD5' )
      except: # pylint: disable-msg=W0702
         self.mode_.addError( 'Invalid encrypted password' )
         return

   def noBmpStationKey( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if config:
         config.authenticationKey = config.authenticationKeyDefault

   def setStatsInterval( self, statsInterval ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.statsInterval = statsInterval

   def noStatsInterval( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if config:
         config.statsInterval = config.statsIntervalDefault

   def sourceIntf( self, intf ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.sourceIntf = intf.name

   def noSourceIntf( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if config:
         config.sourceIntf = config.sourceIntfDefault

   def setRemoteAddr( self, address, vrfName ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if vrfName:
         config.vrfName = vrfName
      else:
         config.vrfName = config.vrfNameDefault

      if isValidV6PeerAddr( address ) or isValidPeerAddr( address ):
         ipGenAddr = Arnet.IpGenAddr( str( address ) )
         config.remoteHost = IpAddrOrHostname.fromIp( ipGenAddr )
      else:
         config.remoteHost = IpAddrOrHostname.fromHostname( address )

   def noRemoteAddr( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if config:
         config.remoteHost = config.remoteHostDefault
         config.vrfName = config.vrfNameDefault

   def setActiveMode( self, port, reconnectTime ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.connectionMode = ConnectionMode.active
      config.remotePort = port
      if reconnectTime:
         config.reconnectTime = reconnectTime
      else:
         config.reconnectTime = config.reconnectTimeDefault

   def setPassiveMode( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.connectionMode = ConnectionMode.passive
      config.remotePort = config.remotePortDefault
      config.reconnectTime = config.reconnectTimeDefault

   def noMode( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      if config:
         config.connectionMode = ConnectionMode.unknown
         config.remotePort = config.remotePortDefault
         config.reconnectTime = config.reconnectTimeDefault

   def setTcpKeepalive( self, idleTime, probeInterval, probeCount ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )

      if config:
         keepAlive = Arnet.TcpUtils.TcpKeepaliveOptions(
                        idleTime, probeInterval, probeCount )
         config.keepaliveOption = keepAlive

   def noTcpKeepalive( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )

      if config:
         config.keepaliveOption = config.keepaliveOptionDefault

   def setBmpReceivedRoutes( self, exportPolicies ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )

      prePolicy = 'pre-policy' in exportPolicies
      postPolicy = 'post-policy' in exportPolicies

      config.bmpExportPolicyStationConfig = \
         BmpUtils.BmpExportPolicyConfig( prePolicy, postPolicy )

   def defaultBmpReceivedRoutes( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.bmpExportPolicyStationConfig = \
         BmpUtils.BmpExportPolicyConfigUnset()

   def noBmpReceivedRoutes( self ):
      config = bmpConfig.bmpStation.get( self.bmpStationName )
      config.bmpExportPolicyStationConfig = \
         BmpUtils.BmpExportPolicyConfig( False, False )

RouterBmpStationMode.addModelet( RouterBmpStationModelet )

# BUG65247 should figure out how to get this from Ira, rather than
# duplicating here
def getVrfNames( mode ):
   return sorted( allVrfConfig.vrf.members() )

def gotoBmpStationMode( mode, bmpStationName ):
   bmpStationConfig = bmpConfig.bmpStation.get( bmpStationName )
   if not bmpStationConfig:
      bmpConfig.bmpStation.newMember( bmpStationName )
   childMode = mode.childMode( RouterBmpStationMode,
                               bmpStationName=bmpStationName )
   mode.session_.gotoChildMode( childMode )

def deleteBmpStationMode( mode, bmpStationName ):
   if not bmpStationName in bmpConfig.bmpStation:
      return
   del bmpConfig.bmpStation[ bmpStationName ]
   t0( 'Delete Bgp monitoring station %s' % bmpStationName )

#-------------------------------------------------------------------------------
# "bgp monitoring [ station <s1> <s2>+ ]" command in router bgp config mode
#-------------------------------------------------------------------------------
class RouterBgpMonitoring( CliCommand.CliCommandClass ):
   syntax = 'bgp monitoring'
   noOrDefaultSyntax = 'bgp monitoring'
   data = {
      'bgp': bgpMatcherForConfig,
      'monitoring': 'Enable Bgp monitoring for all/specified stations',
   }

   @staticmethod
   def handler( mode, args ):
      bmpConfig.bmpActivate = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bmpConfig.bmpActivate = bmpConfig.bmpActivateDefault

RouterBgpSharedModelet.addCommandClass( RouterBgpMonitoring )

#-------------------------------------------------------------------------------
# "[no|default] neighbor PEER monitoring"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class RouterBgpNeighborBmp( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER monitoring'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'monitoring': 'Enable BGP Monitoring Protocol for this peer',
   }

   @staticmethod
   def handler( mode, args ):
      peer = args[ 'PEER' ]
      validatePeer( mode, peer )
      config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
      config.bmpActivatePresent = True
      config.bmpActivate = True
      delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def noHandler( mode, args ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.bmpActivatePresent = True
         config.bmpActivate = not config.bmpActivatePresent
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

   @staticmethod
   def defaultHandler( mode, args ):
      peer = args[ 'PEER' ]
      config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
      if config:
         config.bmpActivatePresent = False
         config.bmpActivate = not config.bmpActivatePresent
         delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

RouterBgpSharedModelet.addCommandClass( RouterBgpNeighborBmp )

def Plugin( entityManager ):
   global allVrfConfig
   global bmpConfig

   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
   bmpConfig = ConfigMount.mount( entityManager, 'routing/bmp/config',
                                  'Routing::Bmp::BmpConfig', 'w' )
   deleteRouterBgpVrfHook.addExtension( cleanupBmpConfig )
