# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=inconsistent-return-statements

import ast
import StringIO

import AgentCommandRequest
import AgentDirectory
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.BmcModel
import CliToken.Hardware
import CliToken.System
import ShowCommand

#------------------------------------------------------------------------------
# common tokens and CLI guard
#------------------------------------------------------------------------------

sysname = None

def bmcAgentIsRunning():
   return AgentDirectory.agentIsRunning( sysname, 'Bmc' )

def bmcCmdGuard( mode, token ):
   if bmcAgentIsRunning():
      return None
   return CliParser.guardNotThisPlatform

#------------------------------------------------------------------------------
# "show system bmc status"
#------------------------------------------------------------------------------

systemBmcMatcherForShow = CliMatcher.KeywordMatcher( 'bmc',
   helpdesc='Show information about the BMC' )

class ShowSystemBmcStatus( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system bmc status'
   data = {
      'system': CliToken.System.systemMatcherForShow,
      'bmc': CliCommand.Node( matcher=systemBmcMatcherForShow,
                              guard=bmcCmdGuard ),
      'status': 'Show status information'
   }
   cliModel = CliPlugin.BmcModel.BmcStatusModel

   @staticmethod
   def handler( mode, args ):
      buff = StringIO.StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager,
         'Bmc', 'bmc', 'ShowSystemBmcStatus', asyncCommand=True,
         timeout=10, stringBuff=buff )
      output = buff.getvalue()
      model = CliPlugin.BmcModel.BmcStatusModel()
      try:
         modelInputs = ast.literal_eval( output )
         busError = modelInputs.get( 'busError' )
         if busError is True:
            mode.addError( 'Smbus Error' )
            return
         elif busError != None:
            mode.addError(
               'Unexpected item found in result: ( busError, {} )'
               .format( busError ) )
            return
         model.setAttrsFromDict( ast.literal_eval( output ) )
      except SyntaxError:
         mode.addError(
            'Unable to parse the following output from BmcAgent: %s' % output )
         return
      return model

BasicCli.addShowCommandClass( ShowSystemBmcStatus )

#------------------------------------------------------------------------------
# "hardware boot ( cpu | bmc )" in exec mode
#------------------------------------------------------------------------------

hardwareBootMatcher = CliMatcher.KeywordMatcher( 'boot',
   helpdesc='Set next boot personality' )

class HardwareBootCommand( CliCommand.CliCommandClass ):
   syntax = 'hardware boot ( cpu | bmc )'
   data = {
         'hardware': CliToken.Hardware.hardwareForExecMatcher,
         'boot': CliCommand.Node( matcher=hardwareBootMatcher,
                                  guard=bmcCmdGuard ),
         'cpu': 'Set next boot personality to CPU',
         'bmc': 'Set next boot personality to BMC'
   }

   @staticmethod
   def handler( mode, args ):
      if 'cpu' in args:
         cmdString = 'HardwareBootCpu'
      else:
         cmdString = 'HardwareBootBmc'

      buff = StringIO.StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager,
         'Bmc', 'bmc', cmdString, asyncCommand=True,
         timeout=10, stringBuff=buff )
      output = buff.getvalue()
      if output == 'failed\n':
         mode.addError( 'Smbus Error' )
      elif output != 'succeeded\n':
         mode.addError( 'Unexpected result' )

BasicCli.EnableMode.addCommandClass( HardwareBootCommand )

#------------------------------------------------------------------------------
# "hardware bmc power ( on | off )" in exec mode
#------------------------------------------------------------------------------

hardwareBmcMatcher = CliMatcher.KeywordMatcher( 'bmc',
   helpdesc='Execute commands for the BMC' )

class HardwareBmcPowerCommand( CliCommand.CliCommandClass ):
   syntax = 'hardware bmc power ( on | off )'
   data = {
         'hardware': CliToken.Hardware.hardwareForExecMatcher,
         'bmc': CliCommand.Node( matcher=hardwareBmcMatcher,
                                 guard=bmcCmdGuard ),
         'power': 'Turn the BMC on or off',
         'on': 'Turn the BMC on',
         'off': 'Turn the BMC off'
   }

   @staticmethod
   def handler( mode, args ):
      if 'on' in args:
         cmdString = 'HardwareBmcPowerOn3' # attempt maximum of 3 times
      else:
         cmdString = 'HardwareBmcPowerOff'

      AgentCommandRequest.runSocketCommand( mode.entityManager,
         'Bmc', 'bmc',
         cmdString,
         asyncCommand=True,
         timeout=500 )

BasicCli.EnableMode.addCommandClass( HardwareBmcPowerCommand )

#------------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------------

def Plugin( entityManager ):
   global sysname
   sysname = entityManager.sysname()
