# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from CliModel import Model, Dict, Bool, Int, Str
import CliParser
import ConfigMount, LazyMount
import Tac
import TableOutput
import os.path
import time

blackboxCliConfig = None
blackboxHwDir = None

def blackboxGuard( mode, token ):
   if 'config' in blackboxHwDir.entryState and blackboxHwDir[ 'config' ].supported:
      return None

   return CliParser.guardNotThisPlatform

def enableBlackbox( mode, args ):
   blackboxCliConfig.enabled = True

def disableBlackbox( mode, args ):
   blackboxCliConfig.enabled = False

#-------------------------------------------------------------------------------
# show reload console status
#-------------------------------------------------------------------------------
class ConsoleLogFile( Model ):
   idVal = Int( help="ConsoleLog file id" )
   fileName = Str( help="ConsoleLog file name" )
   dumpTime = Str( help="ConsoleLog save time" )

class BlackboxStatusDetail( Model ):
   blackboxSupported = Bool( help="Console logging supported" )
   blackboxEnabled = Bool( help="Console logging enabled" )
   consoleLogFiles = Dict( keyType=int, valueType=ConsoleLogFile,
                           help="Available ConsoleLog files" )

   def render( self ):
      print "Console logging:",
      if self.blackboxSupported:
         print "Supported,",
      else:
         print "Not supported,",
      if self.blackboxEnabled:
         print "Enabled"
      else:
         print "Not enabled"

      # Even if Blackbox console log files are present, the CLI
      # will hide them if Blackbox recording is disabled.
      if self.blackboxSupported and self.blackboxEnabled:
         self.renderConsoleLogFiles()

   def renderConsoleLogFiles( self ):
      print
      print 'Console log files:'
      print

      tableHeadings = ( "Id", "Name", "Save Time" )
      idFormat = TableOutput.Format( justify='right' )
      idFormat.padLimitIs( True )
      fnFormat = TableOutput.Format( justify='left' )
      fnFormat.padLimitIs( True )
      dtFormat = TableOutput.Format( justify='left' )
      dtFormat.padLimitIs( True )

      table = TableOutput.createTable( tableHeadings )
      table.formatColumns( idFormat, fnFormat, dtFormat )

      for idVal in sorted( self.consoleLogFiles ):
         bbFile = self.consoleLogFiles[ idVal ]
         table.newRow( idVal, bbFile.fileName, bbFile.dumpTime )
      print table.output()

def doShowReloadConsoleStatus( mode, args ):
   blackboxEnabled = False
   blackboxSupported = False
   consoleLogFiles = {}

   if blackboxHwDir and 'config' in blackboxHwDir.entryState:
      blackboxSupported = blackboxHwDir[ 'config' ].supported
      if blackboxSupported and 'status' in blackboxHwDir.entryState:
         blackboxEnabled = blackboxHwDir[ 'status' ].enabled

         if blackboxEnabled:
            for idVal, bbFile in blackboxHwDir[ 'status' ].file.iteritems():
               dumpTime = time.strftime( '%Y-%m-%d %H:%M:%S',
                  time.localtime( bbFile.dumpTime ) )
               consoleLogFile = ConsoleLogFile( idVal=idVal,
                  fileName=bbFile.fileName, dumpTime=dumpTime )
               consoleLogFiles[ idVal ] = consoleLogFile

   return BlackboxStatusDetail( blackboxSupported=blackboxSupported,
                                blackboxEnabled=blackboxEnabled,
                                consoleLogFiles=consoleLogFiles )

#-------------------------------------------------------------------------------
# show reload console logs [ FILENUM ]
#-------------------------------------------------------------------------------
class ConsoleOutput( Model ):
   consoleLog = Str( help="Console log" )

   def render( self ):
      print self.consoleLog

def doShowPersistentLogs( mode, args ):
   assert 'status' in blackboxHwDir.entryState
   status = blackboxHwDir[ 'status' ]

   fileNum = args.get( 'FILENUM' )
   if fileNum:
      if fileNum not in status.file:
         raise CliParser.InformationalShowCmdError(
            "Specified file number not available" )
   else:
      if not status.file:
         raise CliParser.InformationalShowCmdError(
            "No persistent logs are available." )
      # By default, the lowest file number is chosen
      fileNum = min( status.file )

   fileName = status.file[ fileNum ].fileName
   filePath = os.path.join( status.flashDir, fileName )

   if not os.path.isfile( filePath ):
      raise CliParser.InformationalShowCmdError(
         'Could not read persistent log file %s' % filePath )
   output = Tac.run( [ '/usr/bin/ahaCatBlackbox', filePath ], stdout=Tac.CAPTURE )

   return ConsoleOutput( consoleLog=output )

##-------------------------------------------------------------------------------
## Plugin
##-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global blackboxCliConfig, blackboxHwDir
   blackboxCliConfig = ConfigMount.mount( entityManager,
      'hardware/blackbox/cliConfig', 'Hardware::Blackbox::CliConfig', 'w' )
   blackboxHwDir = LazyMount.mount( entityManager,
      'cell/%d/hardware/blackbox' % Cell.cellId(), 'Tac::Dir', 'r' )

