#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import os
import tempfile

import BasicCliUtil
import FileReplicationCmds
import Fru
import Tac
import Url

from CliPlugin import ReloadCli, BiosCliLib

class BiosInstaller( object ):
   SUCCESS = 1
   FAIL = 2
   ABORT = 3

   # Supervisors to update. Options are mutually exclusive.
   ALL = 1
   ACTIVE = 2
   STANDBY = 3

   def __init__( self, mode, sup, reboot, now ):
      self.mode = mode
      self.sup = sup
      self.reboot = reboot
      self.now = now

      # Enable loopback for file replication in simulation mode
      self.simulation = 'SIMULATION_VMID' in os.environ

      self.entityMibRoot = BiosCliLib.getEntityMibRoot( self.mode.entityManager )
      self.redStatus = BiosCliLib.getRedundancyStatus( self.mode.entityManager )
      self.sbStatus = BiosCliLib.getSecurebootStatus( self.mode.entityManager )

      # This is accessed a few times, so only get it once
      self.isModular = BiosCliLib.isModular( self.entityMibRoot )

   def updateActiveOrFixed( self ):
      return self.sup != self.STANDBY

   def updateStandby( self ):
      return ( self.isModular and self.sup != self.ACTIVE and
               BiosCliLib.isPeerStandby( self.redStatus ) )

   def validateAuf( self, filename, supeId=None ):
      '''
      Validate the AUF. Even if there is a failure, keep going, and print all the
      errors that are applicable. It would be annoying to fix one error only to find
      another later on.
      '''
      # Lazy import AUF as this import zipfile which is a memory hog. This avoids
      # it acting as a memory hog for each CLI process.  Only the CLI using this
      # command will be affected, rather than globally.
      from Auf import Auf

      try:
         auf = Auf( filename )
      except Exception as e: # pylint: disable=broad-except
         # We don't really care why it error'd, just propogate this back to the user
         self.mode.addError( 'AUF parsing failed with error: %s' % ( format( e ) ) )
         return False

      # Check signature
      if not auf.isSigned( useDevCA=self.simulation ):
         self.mode.addError( 'Invalid AUF signature' )
         return False

      # Use sysdb in simulation to get the running version. On the product just
      # read directly from /proc/cmdline
      if self.simulation:
         runningVersion = BiosCliLib.getSysdbAbootVersion( self.entityMibRoot )
      else:
         runningVersion = BiosCliLib.getRunningAbootVersion( supeId=supeId )

      version = re.match( r'Aboot-(\w+)-(\d+).(\d+).(\d+)', runningVersion )
      line = version.group( 1 )

      # Don't try and read the flash in simulation mode. Otherwise the programmed
      # version is the superior source of truth.
      if not self.simulation:
         abootReader = BiosCliLib.getAbootReader( line )
         ( programmedVersion, _ ) = \
                        abootReader.getProgrammedAndFallbackVersions( supeId=supeId )
         if programmedVersion:
            version = re.match( r'Aboot-(\w+)-(\d+).(\d+).(\d+)',
                                programmedVersion ) or version

      # Pass version matching if no version in AUF
      isLine = int( version.group( 2 ) ) == auf.line
      isMajor = int( version.group( 3 ) ) == auf.major
      isMinor = int( version.group( 4 ) ) <= auf.minor

      # Verify version is compatible
      if ( not isLine or not isMajor or not isMinor ):
         self.mode.addError( 'Incompatible AUF. Running Aboot version is '
                             '%s.%s.%s but AUF is for version %s.%s.%s.' %
                             ( version.group( 2 ), version.group( 3 ),
                               version.group( 4 ), auf.line, auf.major,
                               auf.minor ) )
         return False

      # If exists, run compat script
      compatScript = auf.getCompatibilityScript()
      if compatScript:
         cmd = 'bash %s' % ( compatScript )
         cmd = cmd.split()
         if supeId:
            # Copy script to standby
            compatScriptDst = '/tmp/compat_check.sh'
            Tac.run( FileReplicationCmds.copyFile( supeId, compatScriptDst,
                                                   compatScript, useKey=True,
                                                   loopback=self.simulation ),
                     asRoot=True )

            # Run compat script on standby
            cmd = 'bash %s' % ( compatScriptDst )
            cmd = FileReplicationCmds.runCmd( supeId, cmd, useKey=True,
                                              loopback=self.simulation )

         try:
            Tac.run( cmd, asRoot=True )
         except Tac.SystemCommandError as e:
            self.mode.addError( 'AUF compatibility script failed with error code: %d'
                                % ( e.error ) )
            return False

      return True

   def moveAuf( self, src, dest ):
      # Put auf in correct location
      if self.updateActiveOrFixed():
         self.mode.addMessage( 'Moving AUF to the update directory...' )
         Tac.run( [ 'cp', src.localFilename(), dest.localFilename() ], asRoot=True )

      # Send to standby if one exists
      if self.updateStandby():
         self.mode.addMessage( 'Transferring AUF to the standby supervisor...' )

         standbyDir = os.path.dirname( dest.localFilename() )
         standbyFilename = os.path.join( standbyDir, dest.basename() )

         # Create aboot update directory
         cmd = FileReplicationCmds.createDir( Fru.slotId(), standbyDir,
                                              useKey=True,
                                              loopback=self.simulation )
         Tac.run( cmd, asRoot=True )

         # Send AUF
         cmd = FileReplicationCmds.copyFile( Fru.slotId(), standbyFilename,
                                             src.localFilename(), useKey=True,
                                             loopback=self.simulation )
         Tac.run( cmd, asRoot=True )

         # Sync file
         cmd = FileReplicationCmds.syncFile( Fru.slotId(), standbyFilename,
                                             useKey=True, loopback=self.simulation )
         Tac.run( cmd, asRoot=True )

      self.mode.addMessage( 'Done.' )

   def promptUser( self, question ):
      if self.now:
         return True

      return BasicCliUtil.confirm( self.mode, question )

   def run( self, src, destDir ):
      # Make sure we're running on the right sup
      if self.isModular and not BiosCliLib.isActive( self.redStatus ):
         self.mode.addError( 'This command is only available on the active '
                             'supervisor' )
         return self.FAIL

      # Check if src is a valid filetype
      filename = src.basename()
      if not filename.endswith( '.auf' ):
         self.mode.addError( 'File is not a \'.auf\'.' )
         return self.ABORT

      # Create our destination in /mnt/flash/aboot/update/
      dest = Url.parseUrl( os.path.join( destDir, filename ),
                           Url.Context( *Url.urlArgsFromMode( self.mode ) ) )

      # Create aboot update dir if it doesn't exist
      Tac.run( [ 'mkdir', '-p', os.path.dirname( dest.localFilename() ) ] )

      # Download if file isn't on system
      if src.fs.fsType == 'network':
         question = 'Download AUF? [confirm]'
         if not self.promptUser( question ):
            return self.ABORT

         downloadUrl = src

         # Create temporary file for validation
         tmp = tempfile.NamedTemporaryFile( suffix='.auf' )
         src = Url.parseUrl( 'file:' + tmp.name,
                             Url.Context( *Url.urlArgsFromMode( self.mode ) ) )

         self.mode.addMessage( 'Downloading AUF...' )
         try:
            downloadUrl.writeLocalFile( downloadUrl, src.localFilename() )
         except EnvironmentError as ( _, error ):
            self.mode.addError( error )
            return self.FAIL
         self.mode.addMessage( 'Done.' )

      # Validate the AUF on a fixed system or the active sup
      if self.updateActiveOrFixed():
         self.mode.addMessage( 'Validating the AUF...' )
         if not self.validateAuf( src.localFilename() ):
            return self.FAIL

      # Validate the AUF on the standby sup
      if self.updateStandby():
         self.mode.addMessage( 'Validating the AUF on the standby supervisor...' )

         # In simulation mode don't use FileReplicationCmds
         slotId = None if self.simulation else Fru.slotId()
         if not self.validateAuf( src.localFilename(), slotId ):
            return self.FAIL

      if BiosCliLib.isSecurebootSupported( self.sbStatus ):
         self.moveAuf( src, dest )
      else:
         # TODO BUG475153: If non-secureboot, flash the BIOS
         pass

      # Reload both supes
      if self.reboot:
         if self.updateStandby():
            self.mode.addMessage( 'Reloading standby supervisor...' )
            ReloadCli.reloadPeerSupervisor( quiet=True )
            # This is set so that we don't warn the user that the
            # peer supervisor is disabled, since we just reloaded it
            setattr( self.mode, 'disableElectionCheckBeforeReload', True )

         if self.updateActiveOrFixed():
            self.mode.addMessage( 'Reloading%s...' %
                                  ( ' this supervisor' if self.isModular else '' ) )
            ReloadCli.doReload( self.mode, power=True, now=self.now )

         setattr( self.mode, 'disableElectionCheckBeforeReload', False )

      # TODO: right now unconditionally return success here, but we should make sure
      # this actually reflects if the upgrade was a success of failure (in the case
      # of legacy systems, and secureboot + reboot).
      return self.SUCCESS
