#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

from CliModel import Model, Str, Submodel
from CliPlugin import BiosCliLib
from TableOutput import createTable, Format

flashReadErr = "Failed to read version from flash"

def getRunningVersionAndSysName( mode ):
   running = None
   sysName = None
   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )
   if entityMibRoot is None:
      pass
   elif BiosCliLib.isModular( entityMibRoot ):
      card = BiosCliLib.getActiveSupervisorCard( entityMibRoot )
      if card:
         running = BiosVersion().toModel( card.firmwareRev )
         sysName = "%s%d" % ( card.tag, int( card.label ) )
   else:
      running = BiosVersion().toModel( entityMibRoot.firmwareRev )
      sysName = "FixedSystem"
   return ( running, sysName )

class BiosVersion( Model ):
   line = Str( help="Aboot line (eg. norcal9)", optional=True )
   version = Str( help="Version string (eg. 7.2.1)", optional=True )
   pcieSpeed = Str( help="PCIe speed string (if present)", optional=True )
   core = Str( help="Core string (if present)", optional=True )
   changeNumber = Str( help="Change number", optional=True )
   error = Str( help="Reason for error if we failed to read the BIOS",
                optional=True )

   def toModel( self, inputString, referenceVersion=None ):
      if not inputString:
         self.error = flashReadErr
         return self

      m = re.search( BiosCliLib.abootRe, inputString )
      if not m:
         self.error = flashReadErr
         return self
      versionString = m.group( 1 )
      self.changeNumber = m.group( 2 )
      m = re.search( r'Aboot-(\w+)-', versionString )
      if m:
         self.line = m.group( 1 )
      m = re.search( r'-(\d+\.\d+\.\d+)', versionString )
      if m:
         self.version = m.group( 1 )
      m = re.search( r'-(pcie)?(\d+x\d+)-', versionString )
      if m:
         self.pcieSpeed = 'pcie%s' % m.group( 2 )
      m = re.search( r'-(\d+core)-', versionString )
      if m:
         self.core = m.group( 1 )
      if referenceVersion:
         self.reconcileMissingFields( referenceVersion )
      return self

   def reconcileMissingFields( self, referenceVersion ):
      # Sometimes there are discrepancies between the running and programmed
      # aboot version formats, but if they have the same change # it's pretty
      # safe to assume that they should be the same
      if ( self.changeNumber == referenceVersion.changeNumber and
           self.line == referenceVersion.line ):
         if not self.version:
            self.version = referenceVersion.version
         if not self.pcieSpeed:
            self.pcieSpeed = referenceVersion.pcieSpeed
         if not self.core:
            self.core = referenceVersion.core

   def toString( self ):
      if self.error:
         return self.error
      version = self.version if self.version else "?.?.?"
      return "-".join( filter( None, [ 'Aboot', self.line, version,
                     self.pcieSpeed, self.core, self.changeNumber ] ) )

class BiosVersions( Model ):
   sysName = Str( help="Name of the system" )
   runningVersion = Submodel( valueType=BiosVersion,
                              help="BIOS version currently running on the switch" )
   programmedVersion = Submodel( valueType=BiosVersion,
                                 help="BIOS version programmed on the switch" )
   fallbackVersion = Submodel( valueType=BiosVersion,
                               help="BIOS fallback version programmed on the switch",
                               optional=True )

   def render( self ):
      if not self.sysName:
         print( "System not yet initialized" )
         return
      if self.runningVersion.error:
         print( "System not supported" )
         return
      print( "%s BIOS versions" % self.sysName )
      t = createTable( ( "Location", "Version" ), indent=2 )
      f = Format( justify="left" )
      f.noPadLeftIs( True )
      f.padLimitIs( True )
      t.formatColumns( f, f )
      t.newRow( "Running", self.runningVersion.toString() )
      t.newRow( "Programmed", self.programmedVersion.toString() )
      if self.fallbackVersion:
         t.newRow( "Fallback", self.fallbackVersion.toString() )
      print( t.output() )
