#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

import Cell
import FileReplicationCmds
import FirmwareRev
import LazyMount
import Tac

#--------------------------------------------------------------------------------
# Sysdb helpers
#--------------------------------------------------------------------------------

# Entity MIB
def getEntityMibRoot( entityManager ):
   return LazyMount.mount( entityManager,
                           'hardware/entmib',
                           'EntityMib::Status', 'r' ).root

def isModular( entityMibRoot ):
   return "Chassis" in entityMibRoot.tag

def getActiveSupervisorCard( entityMibRoot ):
   assert isModular( entityMibRoot )

   for cs in entityMibRoot.cardSlot:
      card = entityMibRoot.cardSlot[ cs ].card
      if card and card.firmwareRev:
         return card

   return None

def getSysdbAbootVersion( entityMibRoot ):
   if isModular( entityMibRoot ):
      card = getActiveSupervisorCard( entityMibRoot )
      return card.firmwareRev
   return entityMibRoot.firmwareRev

# Redundancy status
def getRedundancyStatus( entityManager ):
   return LazyMount.mount( entityManager,
                           Cell.path( 'redundancy/status' ),
                           'Redundancy::RedundancyStatus',
                           'r' )

def isActive( redundancyStatus ):
   return redundancyStatus.mode == 'active'

def isPeerStandby( redundancyStatus ):
   return redundancyStatus.peerMode == 'standby'

# Secureboot
def getSecurebootStatus( entityManager ):
   return LazyMount.mount( entityManager,
                           Cell.path( "aboot/sb/status" ),
                           "Aboot::Secureboot::Status", "r" )

def isSecurebootSupported( sbStatus ):
   return sbStatus.supported

def isSecurebootEnabled( sbStatus ):
   return not sbStatus.securebootDisabled

#--------------------------------------------------------------------------------
# Aboot helpers
#--------------------------------------------------------------------------------

# Aboot firmware readers
abootRe = r'(Aboot-[0-9a-z\-\.]*-)(eng\d*|\d+)'

class AbootReaderBase( object ):
   imageSection = None
   fallbackSection = None

   def getProgrammedAndFallbackVersions( self, supeId=None ):
      m = re.search( abootRe, self.readFlash( section=self.imageSection,
                                              supeId=supeId ) )
      abootString = m.group( 0 ) if m else ''
      programmed = abootString
      if self.fallbackSection:
         m = re.search( abootRe, self.readFlash( section=self.fallbackSection ) )
         abootString = m.group( 0 ) if m else ''
         fallback = abootString
      else:
         fallback = None
      return ( programmed, fallback )

   def readFlash( self, section=None, supeId=None ):
      if section:
         cmd = "flashUtil -r %s -" % section
      else:
         # Read the entire flash
         cmd = "flashrom -r -"

      if supeId:
         cmd = FileReplicationCmds.runCmd( supeId, cmd, useKey=True )
      else:
         cmd = cmd.split()

      return Tac.run( cmd, asRoot=True,
                      stdout=Tac.CAPTURE, stderr=Tac.DISCARD )

class AbootUnknownReader( AbootReaderBase ):
   def getProgrammedAndFallbackVersions( self, supeId=None ):
      output = self.readFlash( supeId=supeId )
      aboots = re.findall( abootRe, output )
      # assume fallback section comes before programmed section
      if not aboots:
         return ( None, None )
      elif len( aboots ) == 1:
         return ( aboots[ 0 ].group( 0 ), None )
      else:
         return ( aboots[ -1 ].group( 0 ), aboots[ 0 ].group( 0 ) )

class Aboot2Reader( AbootReaderBase ):
   imageSection = 'image'
   fallbackSection = None

class Aboot6Reader( AbootReaderBase ):
   imageSection = 'image'
   fallbackSection = 'fallback'

class Aboot7Reader( AbootReaderBase ):
   imageSection = 'normal'
   fallbackSection = 'fallback'

class Aboot9Reader( AbootReaderBase ):
   imageSection = 'coreboot'
   fallbackSection = 'bootblock'

def getAbootReader( abootLine ):
   if abootLine == 'norcal2':
      abootReader = Aboot2Reader()
   elif abootLine in [ 'norcal4', 'norcal6' ]:
      abootReader = Aboot6Reader()
   elif abootLine == 'norcal7':
      abootReader = Aboot7Reader()
   elif abootLine == 'norcal9':
      abootReader = Aboot9Reader()
   else:
      abootReader = AbootUnknownReader()
   return abootReader

#--------------------------------------------------------------------------------
# Misc
#--------------------------------------------------------------------------------

def getRunningAbootVersion( supeId=None ):
   if supeId:
      cmd = 'python -c "import FirmwareRev; print( ' \
            'FirmwareRev.abootFirmwareRev() )"'
      cmd = FileReplicationCmds.runCmd( supeId, cmd, useKey=True )
      return Tac.run( cmd, asRoot=True,
                      stdout=Tac.CAPTURE, stderr=Tac.DISCARD )

   return FirmwareRev.abootFirmwareRev()
